{

							  Q U I N E                                 
																				
			   A Scratchpad for Truth-functional Logic                  
																				
							Version  2.0

			 Copyright (c) 1997   Gianfranco Boggio-Togna                              *

							 C.P. 14021                                   
					   I-20140 Milano (Italy)                             

					  e-mail: gbt@computer.org                                                     *




	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

}



{$R+}	 
{$S+}

PROGRAM  Quine ;

USES Crt, Dos, KB_Unit, MS_Unit, WND_Unit, MN_Unit, FS_Unit, HLP_Unit ;                                                    

{$I tfeval.pas}			{ the evaluation engine }                      

CONST

		{ symbols used for logical operators on screen }

		ext_not_symbol   = 173 ;        { '-' + 128 } 
		ext_and_symbol   = 174 ;        { '.' + 128 }
		ext_or_symbol    = 246 ;        { 'v' + 128 }
		ext_impl_symbol  = 190 ;        { '>' + 128 }
		ext_equiv_symbol = 189 ;        { '=' + 128 }


		{ symbols used within files only }

		{ the escape sequence '\_' represents the "overline" 'not' symbol }

		over_not         =  95 ;        { '_' }
		end_of_record    =  13 ;        { CR }
		line_feed        =  10 ;        { LF }


		{ screen dimensions }

		first_line   = 2 ;
		max_lines    = 25 ; 
		max_cols     = 80 ;
				

		{ screen control parameters	}

		x_eval    = 1 ;
		x_name    = 3 ;
		x_select  = 4 ;

		min_x_index     = 6 ;
		max_x_index     = 80 ;

		min_y_index     = 3 ;
		max_y_index     = max_lines  ; 

		screenful       = max_y_index - min_y_index + 1 ;

		line_data_length   = 75 ;
		tablet_depth	   = 6  ;

TYPE    
		Direction = (Up, Down, Left, Right) ;

		Line_ptr = ^Line ;
		Line =  RECORD
				  line_next:    Line_ptr ;
				  line_prev:    Line_ptr ;
				  line_parent:  Line_ptr ; { for results of expansion }
				  line_end:     Byte ;     { 255 for list head }
				  line_screen:  Byte ;     { the screen y coordinate }
				  line_eval:    Word ;
				  line_space1:  Word ;
				  line_name:    Word ;
				  line_select:  Word ;
				  line_mark:    Word ;
				  line_data:    ARRAY [1..line_data_length] OF Word ;
				END ;

VAR
		{ screen handling }

		video_initialized : Boolean ;
		video_buffer:   	Word ; 
		screen_lines:   	ARRAY [1..max_lines] OF Line_ptr ;
		buffer_offset:  	ARRAY [1..max_lines] OF Word ;
		empty_line:     	Line ;
		blank:          	Word ;
		end_marker:			Word ;
		end_marker_line:	Integer ;

		{ the scratchpad is implemented as a doubly-linked list }
		{ the list head has line_end = 255						}

		scratchpad_list_head:   Line ;
		paste_list_head:        Line ;

		{ limits of marked block }

		mark_first,
		mark_last:  Line_ptr ;

		{ user input }

		key:        Word ;
		action:     Word ;
		escape:		Boolean ;		{ last character typed was backslash }

		{ current position on screen }

		x_index,
		y_index : Integer ;

		{ pop-up windows }

		tablet_save,
		about_save,
		aboutc_save:     Integer ;

		{ menus }

		scratchpad_menu,
		edit_menu,
		formula_menu,
		options_menu,
		color_set_menu,
		help_menu:    Integer ;

		menu_bar,            
		about_bar,           
		scroll_up_bar,       
		scroll_down_bar,     
		scroll_up_page_bar,  
		scroll_down_page_bar: Integer ;

		{ miscellaneous }

		show_evaluation_pos:  Integer ;
		formula:			  Formula_buffer  ;	 { type defined in tfeval.pas}
		current_file_name:	  String ;

  CONST

		{ significant characters }

		escape_char             = '\' ;
		comment_char            = '-' ;


		{ palette number assignment }

		background_color                =  0 ;
		char_normal_overline_color      =  1 ;
		char_normal_color               =  2 ;
		button_special_fg_color         =  3 ;
		button_emphasis_fg_color        =  4 ;
		button_bg_color                 =  5 ;
		emphasis_bg_color               =  6 ;
		white_color                     =  7 ;

		black_color                     =  8 ;
		char_greek_overline_color       =  9 ;
		char_greek_color                = 10 ;
		char_operator_color             = 11 ;
        unused_color         			= 12 ;
		emphasis_color                  = 13 ;
		dark_gray_color	                = 14 ;
		intense_white_color             = 15 ;
		overscan_color                  = 16 ;

		char_eval_color                 = emphasis_color ;


TYPE
		Font_16 = ARRAY [0..4095] OF Byte ;

		Char_type = ( char_normal, char_operator, char_greek, char_eval ) ;

		Palette_registers = PACKED ARRAY [0..16] OF Byte ;

		File_name = String [64] ;

		Proc_type = Procedure (lp: Line_Ptr) ;


VAR
		{ video handling }

		scan_lines :            Integer ;
		regs :                  Registers ;

		old_palette :           Palette_registers ;
		old_underline :         Byte ;
		new_palette :           Palette_registers ;

		font_buffer :           Font_16 ;
	    color_set:				Integer ;

		{ miscellaneous }

		file_var:	            Text ;
		scratchpad_modified:    Boolean ;
		warning_on_exit:		Boolean ;
		warning_on_clear:		Boolean ;


CONST

		{ standard EGA/VGA palette }

		standard_palette : Palette_registers =
						   (0,1,2,3,4,5,20,7,56,57,58,59,60,61,62,63,0) ;


		{ standard files names }

		font_file_name :        File_name   = 'QNEFONT.F' ;
		help_file_name :        File_name   = 'QNEHELP.HLP' ;






{ ************************************************************************
  *                                                                      *
  *			   				   Fatal error 							     *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  restore_video (restore_palette: Boolean) ; FORWARD ;

PROCEDURE  fatal_error (message: String) ;
BEGIN
		IF  video_initialized  THEN
			restore_video (True) ;
		writeln ('QUINE  Fatal error - ', message) ;
		Halt ;
END ;


{ ************************************************************************
  *                                                                      *
  *			        Read the special font from a file  				     *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  display_message (title, message: String; lp: Line_ptr) ; FORWARD ;

PROCEDURE  read_font ;
TYPE
		Font_14 = ARRAY [0..3583] OF Byte ;
VAR
		font_fname:  File_name ;
		suffix:      String [8] ;                                       
		file_var_14: FILE OF Font_14 ;
		file_var_16: FILE OF Font_16 ;
		buffer_14:   Font_14 ;
		i:           Integer ;
BEGIN
		WITH  regs  DO
		  BEGIN
			ax := $1A00 ;
			Intr ($10, regs) ;  { identify the video system }

			IF  (Lo(bx) = 4)  OR  (Lo(bx) = 5)  THEN
			  scan_lines := 14  { EGA }
			ELSE IF  (Lo(bx) = 7)  OR  (Lo(bx) = 8)  THEN
			  scan_lines := 16  { VGA }
			ELSE
			  BEGIN
				bx := $10 ;
				ax := $1200 ;
				Intr ($10,regs) ;
				IF  Lo(bx) <>  $10  THEN
				  scan_lines := 14
				ELSE
				  fatal_error ('The program requires an EGA or VGA video sistem') ;
			  END ;
		  END ;

		str (scan_lines, suffix) ;                                 
		font_fname := concat (font_file_name, suffix) ;            

		IF  scan_lines = 14  THEN
		  BEGIN
			Assign (file_var_14, font_fname) ;
{$i-}       Reset (file_var_14) ;                                       {$i+}
		  END
		ELSE
		  BEGIN
			Assign (file_var_16, font_fname) ;                     
{$i-}       Reset (file_var_16) ;                                       {$i+}
		  END ;

		IF  Ioresult <> 0  THEN                                    
		   BEGIN
			 fatal_error ('Font file ' + font_fname + ' not found' ) ;
		   END ;

		IF  scan_lines = 14  THEN
		  BEGIN
			Read (file_var_14, buffer_14) ;
			FOR  i := 0  to 3583  DO
			  font_buffer [i] := buffer_14 [i] ;
			  Close (file_var_14) ;
		  END
		ELSE
		  BEGIN
			Read (file_var_16, font_buffer) ;
			Close (file_var_16) ;
		  END ;         

END ;


{ ************************************************************************
  *                                                                      *
  *					      Set the screen colors						     *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  set_colors (color_set: Integer) ;
BEGIN
		IF  (color_set < 1)  OR	 (color_set > 5)  THEN
			color_set := 1 ;

		CASE  color_set  OF

		1:	BEGIN
				new_palette [black_color]                     :=  0 ;
				new_palette [dark_gray_color]                 := 56 ;
				new_palette [char_normal_overline_color]      :=  2 ;
				new_palette [char_normal_color]               :=  2 ;
				new_palette [background_color]                :=  0 ;
				new_palette [emphasis_bg_color]               :=  4 ;
				new_palette [emphasis_color]                  := 44 ;
				new_palette [button_bg_color]                 := 16 ;
				new_palette [button_emphasis_fg_color]        := 46 ;
				new_palette [char_greek_overline_color]       :=  7 ;
				new_palette [char_greek_color]                :=  7 ;
				new_palette [char_operator_color]             := 18 ;
				new_palette [button_special_fg_color]         := 20 ;
				new_palette [unused_color]   			      := 60 ;
				new_palette [overscan_color]                  :=  0 ;
			END ;

	    2: 	BEGIN
				new_palette [black_color]                     :=  0 ;
				new_palette [dark_gray_color]                 := 56 ;
				new_palette [char_normal_overline_color]      :=  7 ;
				new_palette [char_normal_color]               :=  7 ;
				new_palette [background_color]                :=  1 ;
				new_palette [emphasis_bg_color]               :=  4 ;
				new_palette [emphasis_color]                  := 44 ;
				new_palette [button_bg_color]                 :=  8 ;
				new_palette [button_emphasis_fg_color]        := 46 ;
				new_palette [char_greek_overline_color]       := 27 ;
				new_palette [char_greek_color]                := 27 ;
				new_palette [char_operator_color]             := 54 ;
				new_palette [button_special_fg_color]         := 11 ;
				new_palette [unused_color]         			  := 39 ;
				new_palette [overscan_color]                  :=  1 ;
			END ;

	    3:	BEGIN
				new_palette [black_color]                     :=  0 ;
				new_palette [dark_gray_color]                 := 56 ;
				new_palette [char_normal_overline_color]      :=  7 ;
				new_palette [char_normal_color]               :=  7 ;
				new_palette [background_color]                := 16 ;
				new_palette [emphasis_bg_color]               :=  4 ;
				new_palette [emphasis_color]                  := 44 ;
				new_palette [button_bg_color]                 := 32;
				new_palette [button_emphasis_fg_color]        := 46 ;
				new_palette [char_greek_overline_color]       := 14 ;
				new_palette [char_greek_color]                := 14 ;
				new_palette [char_operator_color]             := 54 ;
				new_palette [button_special_fg_color]         :=  2 ;
				new_palette [unused_color]         			  := 39 ;
				new_palette [overscan_color]                  := 16 ;
			END ;

		4:	BEGIN
				new_palette [black_color]                     :=  0 ;
				new_palette [dark_gray_color]                 := 56 ;
				new_palette [char_normal_overline_color]      :=  7 ;
				new_palette [char_normal_color]               :=  7 ;
				new_palette [background_color]                := 24 ;
				new_palette [emphasis_bg_color]               :=  4 ;
				new_palette [emphasis_color]                  := 44 ;
				new_palette [button_bg_color]                 :=  0 ;
				new_palette [button_emphasis_fg_color]        := 46 ;
				new_palette [char_greek_overline_color]       := 39 ;
				new_palette [char_greek_color]                := 39 ;
				new_palette [char_operator_color]             := 54 ;
				new_palette [button_special_fg_color]         := 35 ;
				new_palette [unused_color]         			  := 39 ;
				new_palette [overscan_color]                  := 24 ;
			END ;

		5:	BEGIN
				new_palette [black_color]                     :=  0 ;
				new_palette [dark_gray_color]                 := 56 ;
				new_palette [char_normal_overline_color]      :=  7 ;
				new_palette [char_normal_color]               :=  7 ;
				new_palette [background_color]                :=  0 ;
				new_palette [emphasis_bg_color]               :=  4 ;
				new_palette [emphasis_color]                  := 59  ;
				new_palette [button_bg_color]                 := 49 ;
				new_palette [button_emphasis_fg_color]        := 39 ;
				new_palette [char_greek_overline_color]       := 23 ;
				new_palette [char_greek_color]                := 23 ;
				new_palette [char_operator_color]             := 63 ;
				new_palette [button_special_fg_color]         := 21 ;
				new_palette [unused_color]         			  := 39 ;
				new_palette [overscan_color]                  :=  0 ;
			END ;
		END ;
		
		regs.ax := $1002 ;      
		regs.es := Seg(new_palette) ;
		regs.dx := Ofs(new_palette) ;
		Intr ($10, regs) ;           { set all palette registers }

END ;


{ ************************************************************************
  *                                                                      *
  *			  		Load font and set underline location 			     *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  set_font_and_underline ;
BEGIN
		WITH  regs  DO
		  BEGIN
			ax := $1110 ;
			bx := (scan_lines * 256) + 0 ;
			cx := 256 ;
			dx := 0 ;
			es := Seg (font_buffer) ;
			bp := Ofs (font_buffer) ;
			Intr ($10, regs) ;          { load  font }

			cx := $707 ;
			ax := $100 ;
			Intr ($10,regs) ;           {set alphanumeric cursor size }
				
			bl := 0 ;
			ax := $1003 ;
			Intr ($10,regs) ;           {disable blinking}
		  END ;


		port [$3D4] := $14 ;            { Underline location }
		old_underline := port [$3D5] ;

		port [$3D4] := $14 ;            { Underline location }
		port [$3D5] := 0 ;

END ;


{ ************************************************************************
  *                                                                      *
  *				  		Initialize the video system	 				     *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  initialize_video ;
VAR
		i: Integer ;
BEGIN
		regs.ax := 3 ;          { select 80 by 25  16-color alphanumeric }
		Intr ($10, regs) ;
		video_buffer := $b800  ;

		old_palette := standard_palette ;

		regs.ax := $1009 ;      { read all palette registers (VGA only)  }
		regs.es := Seg(old_palette) ;
		regs.dx := Ofs(old_palette) ;
		Intr ($10, regs) ;

		new_palette := standard_palette ;

		read_font ;
		set_colors (color_set) ;
		set_font_and_underline ;
		video_initialized := True ;
END ;


{ ************************************************************************
  *                                                                      *
  *						Restore the video system 					     *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  restore_video (restore_palette: Boolean) ;
BEGIN
		IF  restore_palette  THEN
		  BEGIN
			TextBackground (black_color) ;
			ClrScr ;
		  END ;

		IF restore_palette  THEN
		  BEGIN
			regs.ax := 3 ;              { select 80 by 25  16-color alphanumeric }
			Intr ($10, regs) ;
			regs.ax := $1002 ;          { set all palette registers }
			regs.es := Seg(old_palette) ;
			regs.dx := Ofs(old_palette) ;
			Intr ($10, regs) ;
		  END ;

		port [$3D4] := $14 ;   			{ Underline location }
		port [$3D5] := old_underline ;

		IF  scan_lines = 14  THEN
				regs.ax := $1101        { load 8x14 font }
		ELSE
				regs.ax := $1104 ;      { load 8x16 font }

		regs.bx := 0 ;
		Intr ($10, regs) ;

		regs.bl := 0 ;
		regs.ax := $1003 ;
		Intr ($10,regs) ;          		{disable blinking}

END ;


{ ************************************************************************
  *                                                                      *
  *			   		Get a character from the keyboard				     *
  *                                                                      *
  ************************************************************************ }

FUNCTION  special_key : Boolean ;
BEGIN
		special_key := (Hi(key) = 0)        OR
					   (Lo(Key) < Ord(' ')) OR
					   (key = KB_GrayIns)   OR
					   (key = KB_GrayMinus) OR
					   (key = KB_GrayPlus)  OR
					   (key = KB_GrayPeriod) ;
END ;

PROCEDURE  get_key ;
BEGIN
		key := KB_read ;
END ;


{ ************************************************************************
  *                                                                      *
  *						Set fields in video buffer word				     *
  *                                                                      *
  ************************************************************************ }

FUNCTION  video_char_attr (c, fg, bg: Byte) : Word ;
BEGIN
		Video_char_attr := c + ((bg shl 4) + fg) shl 8 ; 
END ;

PROCEDURE  video_char (c: Byte; VAR w: Word) ;
BEGIN
		w := (w AND $ff00) OR c ;
END ;

PROCEDURE  video_fg (fg: Byte; VAR w: Word) ;
BEGIN
		w := (w AND $f0ff) OR (fg shl 8) ;
END ;

PROCEDURE  video_bg (bg: Byte; VAR w: Word) ;
BEGIN
		w := (w AND $0fff) OR (bg shl 12) ;
END ;

FUNCTION  overline (w: Word) : Boolean ;
VAR
		fg:     Byte ;
BEGIN
		fg := Hi(w) AND $0f ;
		Overline := (fg = char_normal_overline_color) OR
					(fg = char_greek_overline_color) ;
END ;


{ ************************************************************************
  *                                                                      *
  *						 	Write heading  							     *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  write_heading ;
BEGIN
		MN_show_menu_bar (about_bar, 1, 1, 9) ;
		MN_show_menu_bar (menu_bar, 1, 12, 63) ;

		MN_show_menu_bar (scroll_up_page_bar, 1, 66, 68) ;
		MN_show_menu_bar (scroll_down_page_bar, 1, 70, 72) ;
		MN_show_menu_bar (scroll_up_bar, 1, 74, 76) ;
		MN_show_menu_bar (scroll_down_bar, 1, 78, 80) ;
END ;


{ ************************************************************************
  *                                                                      *
  *					   Show evaluation in progress					     *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  show_evaluation (on: Boolean) ;
VAR
		c:	Char ; 
		p:  Integer ;
CONST
		d: ARRAY [0..1] OF Integer = (18, 20) ;
BEGIN
		p := show_evaluation_pos ;
		IF  on  THEN
		  BEGIN
			c := Chr(7) ;
			Move (c, Mem [Video_buffer:buffer_offset[1]+d[p]], 1) ;
			c := ' ' ;
			p := 1 - p ;
			Move (c, Mem [Video_buffer:buffer_offset[1]+d[p]], 1) ;
			show_evaluation_pos := p ;
		  END
		ELSE
		  BEGIN
			c := ' ' ;
			Move (c, Mem [Video_buffer:buffer_offset[1]+d[0]], 1) ;
			Move (c, Mem [Video_buffer:buffer_offset[1]+d[1]], 1) ;
		  END ;
END ;


{ ************************************************************************
  *                                                                      *
  *					  Check if line is under tablet					     *
  *                                                                      *
  ************************************************************************ }

FUNCTION  under_tablet (l: Integer) : Boolean ;
BEGIN
		under_tablet :=	 (tablet_save <> 0)  AND  
						 (l >= y_index + 1)  AND
						 (l <= y_index + 1 + tablet_depth) ;	

END ;


{ ************************************************************************
  *                                                                      *
  *					  		Check for blank line  					     *
  *                                                                      *
  ************************************************************************ }

FUNCTION  line_is_blank (lp: Line_ptr) : Boolean ;
VAR
		i: 	Integer ;
		ib:	Boolean	;
BEGIN
		ib := True ;
		IF  (lp^.line_end = 255)  THEN
		  	ib := False 
		ELSE
		  BEGIN
			i := 1 ;
			WHILE   (i <= line_data_length)  AND  ib  DO
		  		IF  lp^.line_data [i] <> blank  THEN
		  			ib := False 
		  		ELSE
		  			Inc (i) ;
		  END ;
		line_is_blank := ib ;
END	;


{ ************************************************************************
  *                                                                      *
  *					  Display end of scratchpad marker				     *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  show_end_marker  ;
VAR
		lp:		Line_ptr ;
		ln:		Integer ;

BEGIN
		IF  end_marker_line <> 0  THEN
			IF  NOT  under_tablet (end_marker_line)  THEN
				Move (blank,
					  Mem [Video_buffer:buffer_offset[end_marker_line]], 2);
		end_marker_line := 0 ;		  

		lp := scratchpad_list_head.line_prev ;
		WHILE  line_is_blank (lp)  DO
			lp := lp^.line_prev ;
		ln := lp^.line_screen ;
		IF  (ln <> 0)  AND (ln < max_lines)  THEN
		  BEGIN
			IF  NOT  under_tablet (ln+1)  THEN
				Move (end_marker, Mem [Video_buffer:buffer_offset[ln+1]], 2);
			end_marker_line := ln + 1 ;
		  END ;
END ;


{ ************************************************************************
  *                                                                      *
  *					  		Move lines to screen 					     *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  to_screen (lp: Line_Ptr; first, max: Integer) ;
VAR
		l:      Line_Ptr ;
BEGIN
		IF  lp = Nil  THEN
		  BEGIN
			show_end_marker ;
		  	Exit ;
		  END ;

		MS_Hide ;
		IF lp^.line_end = 255  THEN
		  IF  first = min_y_index  THEN
			BEGIN
			  l := lp^.line_prev ;
			  WHILE  l^.line_end <> 255  DO
				BEGIN
				  l^.line_screen := 0 ;
				  l := l^.line_prev ;
				END ;
			END ;
		WHILE  (lp^.line_end <> 255)  AND  (first <= max)  DO
		  BEGIN
			IF  NOT  under_tablet (first)  THEN
			  Move (lp^.line_eval, Mem[Video_buffer:buffer_offset[first]], 160);
			screen_lines [first] := lp ;
			lp^.line_screen := first ;
			lp := lp^.line_next ;
			Inc(first) ;
		  END ;

		IF  first <= max  THEN
		  WHILE  first <= max  DO
			BEGIN
			  screen_lines [first] := Nil ;
			  IF  NOT  under_tablet (first)  THEN
					Move (empty_line.line_eval,
						  Mem [Video_buffer:buffer_offset[first]], 160);
			  Inc(first) ;
			END
		ELSE
		  IF  max = max_y_index  THEN
			BEGIN
			  WHILE  lp^.line_end <> 255  DO
				BEGIN
				  lp^.line_screen := 0 ;
				  lp := lp^.line_next ;
				END ;
			END ;
		MS_Show ;
		show_end_marker ;
END ;


{ ************************************************************************
  *                                                                      *
  *					   		Scroll the screen						     *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  close_tablet ; FORWARD ;
PROCEDURE  check_mark   ; FORWARD ;

PROCEDURE  scroll_screen (arrow: Direction; n: Integer) ;
VAR
		lp:     Line_ptr ;
BEGIN
		lp := screen_lines [min_y_index] ;
		IF  lp = Nil  THEN
		  Exit ;

		IF  tablet_save <> 0  THEN
		  close_tablet ;

		WHILE  (lp^.line_end <> 255)  AND  (n > 0)  DO
		  BEGIN
			lp^.line_screen := 0 ;
			IF  arrow = Down  THEN
			  lp := lp^.line_next
			ELSE
			  lp := lp^.line_prev ;
			Dec(n) ;
		  END ;

		IF  (n = 0)  AND  (lp^.line_end <> 255)  THEN
		  to_screen (lp, min_y_index, max_y_index) 
		ELSE
		  to_screen (lp^.line_next, min_y_index, max_y_index) ; 

		IF  mark_first <> Nil  THEN
		  check_mark ;
		show_end_marker ;
END ;


{ ************************************************************************
  *                                                                      *
  *						Update a line on screen						     *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  update_line (lp: Line_Ptr) ;
BEGIN
		IF  (lp <> Nil)  AND (lp^.line_screen <> 0)  THEN
		  WITH  lp^  DO
			IF  NOT  under_tablet (lp^.line_screen)  THEN
			  BEGIN
				MS_hide ;
				Move (line_eval, Mem [Video_buffer:buffer_offset[line_screen]],
				  	  160);
				MS_show ;
				show_end_marker ;
			  END ;
END ;


{ ************************************************************************
  *                                                                      *
  *						Open, close, move tablet 					     *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  open_tablet ;
CONST
		operator_symbols: array [1..5] OF Byte  = ( ext_not_symbol,
													ext_and_symbol,
													ext_or_symbol,
													ext_impl_symbol,
													ext_equiv_symbol ) ; 

		parentheses: array [1..6] OF Char = ( '(', ')', '[', ']', '{', '}' ) ;

		cursor_movement: array [1..5] OF Byte = ( 17, 27, 29, 26, 16) ;

		actions: array [1..3, 1..2] OF String =	( ('   Na', 'me   '),
												  ('  Sel', 'ect  '),
												  (' Eval', 'uate ')  ) ;

VAR
		y_pos, x, x_cur, y_cur, i, dummy_save :  Integer ;
BEGIN
		IF  tablet_save <> 0  THEN
			Exit ;

		IF  y_index > (max_y_index - 1 - tablet_depth)  THEN
		  BEGIN
			scroll_screen (Down, y_index - max_y_index + 1 + tablet_depth) ;
			y_index := y_index - (y_index - max_y_index + 1 + tablet_depth) ;
		  END ;
		y_pos := y_index + 1 ;

		WND_Save_Cursor (False) ;

		tablet_save := 1 ;
		dummy_save  := 0 ;

		WND_Open_Window (y_pos, 1, y_pos+6, 80,
						 button_bg_color, white_color,
						 WND_no_shadow, WND_no_frame,
						 '',
						 tablet_save) ;

		IF  tablet_save = 0 THEN
			fatal_error ('Internal error 1') ;

		WND_Open_Window (y_pos, 2, y_pos+6, 79,
						 button_bg_color, white_color,
						 WND_no_shadow, WND_single_frame,
						 '',
						 dummy_save) ;

		x := 50 ;
		FOR  i := 1  TO  5  DO
		  BEGIN
			gotoXY (x, y_pos+3) ;
			TextColor(background_color) ;
			TextBackground      (background_color) ;
			write ( Chr(operator_symbols[i]) ) ;
			TextColor(char_operator_color) ;
			write ( Chr(operator_symbols[i]) ) ;
			TextColor(background_color) ;
			write ( Chr(operator_symbols[i]) ) ;
			Inc (x,6) ;
		  END ;

		x := 19 ;
		FOR  i := 1  TO  6  DO
		  BEGIN
			gotoXY (x, y_pos+5) ;
			TextColor(background_color) ;
			TextBackground      (background_color) ;
			write ( parentheses[i] ) ;
			TextColor(char_normal_color) ;
			write ( parentheses[i] ) ;
			TextColor(background_color) ;
			write ( parentheses[i] ) ;
			Inc (x,5) ;
		  END ;

		gotoXY (19, y_pos+1) ;
		TextColor(char_normal_color) ;
		write ( '<abcdefghijklmnopqrstuvwxyz>' ) ;

		gotoXY (19, y_pos+3) ;
		TextColor(char_normal_color) ;
		write ( '<ABCDEFGHIJKLMNOPQRSTUVWXYZ>' ) ;

		gotoXY (50, y_pos+1) ;
		TextColor(char_greek_color) ;
		TextBackground(background_color) ;
		write ('<') ;
		FOR  i:= Ord('a')  TO  Ord('z')  DO
		  IF  i <> Ord('v')  THEN
			write (Chr(i+128)) ;
		write ('>') ;

		x := 50 ;
		FOR  i := 1  TO  5  DO
		  BEGIN
			gotoXY (x, y_pos+5) ;
			IF  (i = 1)  OR  (i = 5)  THEN
			  BEGIN
				TextColor (emphasis_bg_color) ;
				TextBackground (emphasis_bg_color) ;
			  END
			ELSE
			  BEGIN
				TextColor (button_bg_color) ;
				TextBackground (button_bg_color) ;
			  END ;
			write ( Chr(cursor_movement[i]) ) ;

			IF  i <> 3  THEN
			  TextColor(white_color) ;
			write ( Chr(cursor_movement[i]) ) ;

			IF  (i = 1)  OR  (i = 5)  THEN
			  BEGIN
				TextColor (emphasis_bg_color) ;
				TextBackground (emphasis_bg_color) ;
			  END
			ELSE
			  BEGIN
				TextColor (button_bg_color) ;
				TextBackground (button_bg_color) ;
			  END ;
			write ( Chr(cursor_movement[i]) ) ;
			Inc (x,6) ;
		  END ;

		x := 5 ;
		FOR  i := 1  TO  3  DO
		  BEGIN
			TextColor (white_color) ;
			TextBackground (button_bg_color) ;
			gotoXY (x, y_pos+1+(i-1)*2) ;     
			write (actions [i,1]) ;
			TextBackground (emphasis_bg_color) ;
			write (actions [i,2]) ;
		  END ;

		WND_Restore_Cursor ;
END ;


PROCEDURE  close_tablet ;
BEGIN
		IF  tablet_save <> 0  THEN
		  BEGIN
			WND_Close_Window (tablet_save) ;
			tablet_save := 0 ;
			to_screen (screen_lines [y_index+1],
					  y_index + 1, y_index + 1 + tablet_depth) ;
		  END ;
END ;


PROCEDURE  move_tablet ;
BEGIN
		IF  tablet_save = 0  THEN
		  Exit ;

		IF  y_index > (max_y_index - 1 - tablet_depth)  THEN
		  BEGIN
			close_tablet ;
			scroll_screen (Down, y_index - max_y_index + 1 + tablet_depth) ;
			y_index := y_index - (y_index - max_y_index + 1 + tablet_depth) ;
			open_tablet ;
		  END 
		ELSE
		  BEGIN
			MS_Hide ;
			WND_Move_Window (y_index+1, 1, tablet_save) ;
			MS_Show ;
		  END ;
END ;


{ ************************************************************************
  *                                                                      *
  *				   Move a character to evaluation field				     *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  set_eval (c: Char; lp: Line_ptr) ;
VAR
		p:      Line_ptr ;
BEGIN
		p := lp ;
		IF  p = Nil  THEN
			p := screen_lines [y_index] ;
		Video_char (Ord(c), p^.line_eval) ;
END ;


{ ************************************************************************
  *                                                                      *
  *						 	Get a new line 							     *
  *                                                                      *
  ************************************************************************ }

FUNCTION  get_line : Line_ptr ;
VAR
		p: Line_ptr ;
BEGIN
		IF  MemAvail < 4096  THEN
		  BEGIN
			fatal_error ('Memory exhausted') ;
			p := Nil ;
		  END
		ELSE
		  BEGIN
			new (p) ;
			move (empty_line.line_next, p^.line_next, SizeOf(empty_line)) 
		  END ;
		get_line := p ;
END ;


{ ************************************************************************
  *                                                                      *
  *                       	List handling                                *
  *                                                                      *
  ************************************************************************ }


FUNCTION  precedes (l1, l2: Line_ptr) : Boolean ;
BEGIN
		IF  l1 = l2  THEN
		  precedes := True 
		ELSE
		  BEGIN
			REPEAT
			  l1 := l1^.line_next ;
			UNTIL (l1^.line_end = 255)  OR  (l1 = l2) ;
			precedes :=  (l1^.line_end <> 255) ;
		  END ;
END ;


FUNCTION  next_line (lp: Line_ptr) : Line_ptr ;
BEGIN
		IF  lp = Nil  THEN
		  next_line := Nil 
		ELSE
		  BEGIN
			lp := lp^.line_next ;
			IF  lp^.line_end <> 255  THEN
			  next_line := lp 
			ELSE
			  next_line := Nil ;
		  END ;
END ;

FUNCTION  previous_line (lp: Line_ptr) : Line_ptr ;
BEGIN
		IF  lp = Nil  THEN
		  previous_line := Nil 
		ELSE
		  BEGIN
			lp := lp^.line_prev ;
			IF  lp^.line_end <> 255  THEN
			  previous_line := lp 
			ELSE
			  previous_line := Nil ;
		   END ;
END ;


{ Insert a new node AFTER an existing one }

PROCEDURE  link_after (new, old: Line_ptr) ;
VAR
		f:      Line_ptr ;
BEGIN
		f := old^.line_next ;  
		old^.line_next := new ;
		new^.line_prev := old ; 
		new^.line_next := f ;
		f^.line_prev   := new ;
END ;



{ Insert a new node BEFORE an existing one }

PROCEDURE  link_before (new, old: Line_ptr) ;
VAR
		f:      Line_ptr ;
BEGIN
		f := old^.line_prev ;  
		old^.line_prev := new ;
		new^.line_prev := f ; 
		new^.line_next := old ;
		f^.line_next   := new ;
END ;


{ Remove a line from the list }

PROCEDURE  unlink_line (l: Line_ptr) ;
VAR
		p, n:   Line_ptr ;
BEGIN
		p := l^.line_prev ;
		n := l^.line_next ;
		n^.Line_prev := p;
		p^.line_next := n ;
END ;


{ ************************************************************************
  *                                                                      *
  *                       Check new x index                              *
  *                                                                      *
  ************************************************************************ }

FUNCTION  new_x_index (index: Integer) : Integer ;
BEGIN
		IF  index < min_x_index THEN
		  new_x_index := min_x_index 
		ELSE
		  IF  index > max_x_index THEN
			new_x_index := max_x_index 
		  ELSE
			new_x_index := index ;
END ;


{ ************************************************************************
  *                                                                      *
  *                       Insert blank lines                             *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  insert_lines ;
VAR
		i:      Integer ;
		lp:     Line_ptr ;
BEGIN
		IF  screen_lines [y_index]  = Nil  THEN
		  FOR  i := min_y_index  TO  y_index  DO
			IF  screen_lines [i]  = Nil  THEN
			  BEGIN
				lp := get_line ;
				IF  lp <> Nil  THEN
				  BEGIN
					screen_lines [i] := lp ;
					lp^.line_screen := i ;
					link_before (lp, @ scratchpad_list_head) ;
					update_line (lp) ;
				  END ;
			  END ;     
END ;


{ ************************************************************************
  *                                                                      *
  *                          Move cursor                                 *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  move_cursor (arrow: Direction ; times: Integer) ;
VAR 
		i:      Integer ;
		lp:     Line_ptr ;
		lp1:    Line_ptr ;
		tablet: Boolean ;
BEGIN
		FOR  i := 1 TO times  DO
		  CASE arrow OF

			 Up:        BEGIN
						  tablet := tablet_save <> 0 ;
						  Dec(y_index) ;
						  IF  y_index < min_y_index  THEN
							BEGIN
							  lp :=  previous_line (screen_lines [min_y_index]) ;
							  IF  lp <> Nil  THEN
								BEGIN
								  To_screen (lp, min_y_index, max_y_index) ;
								  close_tablet ;
								END ;
							  y_index := min_y_index ;
							END ;
						  IF  mark_first <> Nil  THEN
							check_mark ;
						  IF  tablet  AND  (i = times)  THEN
							IF  tablet_save <> 0  THEN
							  move_tablet 
							ELSE
							  open_tablet ;
						END ;

			 Down:      BEGIN
						  tablet := tablet_save <> 0 ;
						  Inc(y_index) ;
						  IF  y_index > max_y_index  THEN
							BEGIN
							  lp :=  next_line (screen_lines [min_y_index]) ;
							  IF  lp <> Nil  THEN
								BEGIN
								  lp1 := next_line (screen_lines [max_y_index]) ;
								  IF  lp1 = Nil  THEN
									BEGIN
									  lp1 := get_line ;
									  IF  lp1 <> Nil THEN
										link_before (lp1, @ scratchpad_list_head) ;
									END ;
							  	  To_screen (lp, min_y_index, max_y_index) ;
								  close_tablet ;
								END ;
							  y_index := max_y_index ;
							END ;
						  IF  mark_first <> Nil  THEN
							check_mark ;
						  IF  tablet  AND  (i = times)  THEN
							IF  tablet_save <> 0  THEN
							  move_tablet 
							ELSE
							  open_tablet ;
						END ;

			 Left:      x_index := new_x_index (x_index - 1);
						
			 Right:     x_index := new_x_index (x_index + 1);
						
		  END ;
				
		gotoXY (x_index, y_index) ;   

END ;


{ ************************************************************************
  *                                                                      *
  *                       Move to top of screen                          *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  origin ;
BEGIN
		gotoXY (min_x_index, min_y_index) ;                      
		x_index := min_x_index ;
		y_index := min_y_index ;
END ;


{ ************************************************************************
  *                                                                      *
  *                  Move to start of current line                       *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  to_start_of_line ;
BEGIN
		IF  x_index >= min_x_index  THEN
		  move_cursor (Left, x_index - min_x_index) ;
END ;


{ ************************************************************************
  *                                                                      *
  *                  Move to end of current line                         *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  to_end_of_line ;
BEGIN
		IF  screen_lines[y_index] <> Nil  THEN
		  WITH  screen_lines[y_index]^  DO
			IF  x_index - min_x_index + 1 <= line_end THEN
			  move_cursor (Right, line_end - (x_index - min_x_index) ) ;
END ;


{ ************************************************************************
  *                                                                      *
  *               	Move to beginning of following line                  *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  down_a_line ;
BEGIN
		move_cursor (Down, 1) ;
		move_cursor (Left, 100) ;
END ;


{ ************************************************************************
  *                                                                      *
  *                       Clear current line                             *               
  *                                                                      *
  ************************************************************************ }

PROCEDURE  clear_line ;
VAR
		i:      Integer ;
BEGIN
		IF  screen_lines[y_index] <> Nil  THEN
		  BEGIN
			FOR  i := 1 TO line_data_length  DO
			  screen_lines[y_index]^.line_data[i]  :=  blank ;
			move_cursor (Left, 100) ;
			update_line (screen_lines [y_index]) ;
			scratchpad_modified := True ;
		  END ;
END ;


{ ************************************************************************
  *                                                                      *
  *                  	Delete to end of line 		                     *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  delete_to_end_of_line ;
VAR
		i:      Integer ;
		x_data: Integer ;
BEGIN
		IF  screen_lines [y_index] <> Nil  THEN
		  BEGIN
			x_data := x_index - min_x_index + 1 ;
			WITH  screen_lines [y_index]^  DO
			  BEGIN
				IF line_end >= x_data  THEN
				  BEGIN
					FOR  i := x_data  TO  line_end  DO
					  line_data [i] := blank ;
					IF  x_data  > 1  THEN
						line_end := x_data - 1 
					ELSE
						line_end := x_data ; 
					update_line (screen_lines [y_index]) ;
				  END ;
			  END ;
			scratchpad_modified := True ;
		  END ;
END ;


{ ************************************************************************
  *                                                                      *
  *                  Delete from start of line to cursor                 *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  delete_from_start_of_line ;
VAR
		i:      Integer ;
		x_from:	Integer ;
		x_to:	Integer ;
BEGIN
		IF  screen_lines [y_index] <> Nil  THEN
		  BEGIN
			x_from := x_index - min_x_index + 1 ;
			x_to := 1 ;
			WITH  screen_lines [y_index]^  DO
			  BEGIN
				WHILE  x_from <= line_end DO
				  BEGIN
					line_data [x_to] := line_data [x_from] ;
					Inc (x_from) ;
					Inc (x_to) ;
				  END ;
				WHILE  x_to <= line_end  DO
				  BEGIN
					line_data [x_to] := blank ;
					Inc (x_to) ;
				  END ;
				WHILE  (line_end > 0)  AND  (line_data[line_end] = blank)  DO
					Dec (line_end) ;
				move_cursor (Left, 100) ;
				update_line (screen_lines [y_index]) ;
			  END ;
			scratchpad_modified := True ;
		  END ;
END ;


{ ************************************************************************
  *                                                                      *
  *                  	Process a text character                         *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  process_text_character (character: Word) ;
VAR
		c, ch:    Char ;
		lp:       Line_ptr ;
		x_data:   Integer ;             
		fg, bg:   Byte ;
		advance:  Boolean ;

  PROCEDURE  insert_char (character: Char) ;
  VAR
		i:       Integer ;
  BEGIN
	WITH  lp^  DO
	  BEGIN
		IF  x_data <= line_end THEN
		  BEGIN
			IF  line_end >= line_data_length  THEN
			  Exit ;
			Move ( line_data [x_data],
				   line_data [x_data+1],
				   (line_end - x_data + 1) Shl 1) ;
			Inc(line_end) ;
		  END
		ELSE
		  BEGIN
			FOR  i := line_end + 1  TO  x_data - 1  DO
			  line_data [i] := blank ;
			line_end := x_data ;
		  END ;

		line_data[x_data] := Video_char_attr ( Ord(character), fg, bg) ;

	  END ;
  END ;

BEGIN
		insert_lines ;
		x_data := x_index - min_x_index + 1 ;
		advance := True ;
		c := Chr(Lo(character)) ;
		ch := c ;
		lp := screen_lines [y_index] ;

		fg := char_normal_color ;
		bg := background_color ;

		IF  escape  AND  (character <> KB_BackSlash)  THEN
		  BEGIN
			c := Chr(Ord(c) + 128) ;
			IF  Ord(c)  IN [ ext_not_symbol,
							 ext_and_symbol,
							 ext_or_symbol, 
							 ext_impl_symbol,
							 ext_equiv_symbol ]  THEN
					fg := char_operator_color 
			ELSE
					fg := char_greek_color ;

		  END ;

		escape := False ;

		IF  x_index > min_x_index  THEN
		  BEGIN
			WITH  lp^ DO
			  BEGIN
				IF  Lo(line_data[x_data-1]) = ext_not_symbol  THEN
				  BEGIN
					IF  (  ((ch >= 'A') AND (ch <= 'Z')) OR
						   ((ch >= 'a') AND (ch <= 'z'))
						)  AND (Ord(c) <> ext_or_symbol)   THEN
						{)  AND NOT (escape AND (ch = 'v'))  THEN}
					  BEGIN
						line_data[x_data-1] :=
						  Video_char_attr ( Ord(c),
											char_normal_overline_color,
											background_color) ;
						IF  Ord(c) >= 128  THEN
						  Video_fg (char_greek_overline_color,
									line_data[x_data-1] ) ;
						advance := False ;
					  END
					ELSE
					  insert_char (c) ;
				  END
				ELSE
				  insert_char (c) ;
			  END ;
		  END
		ELSE
		  insert_char (c) ;

		scratchpad_modified :=  True ;
		IF advance THEN
				move_cursor (right,1) ;
		update_line (lp) ;
END ;


{ ************************************************************************
  *                                                                      *
  *        	   Transcribe formula to format accepted by tfeval           *
  *                                                                      *
  ************************************************************************ }

FUNCTION  build_formula (lp: Line_ptr) : Boolean ;
LABEL
		1 ;
VAR
		i, j : Integer ;
BEGIN
		WITH  lp^  Do
		  BEGIN
			i := 1 ;
			FOR  j := 1  TO  line_end  DO       

			  IF  Lo(line_data[j]) <> Ord(' ')  THEN
				BEGIN 
				  IF  j < max_x_index  THEN
					IF  (Lo(line_data[j])  =  Ord(comment_char)) AND
						(Lo(line_data[j+1])  = Ord(comment_char)) THEN
								GOTO 1 ;

				  IF  Overline(line_data[j])  THEN
					BEGIN               
					  formula[i] := Not_symbol ;
					  i := i + 1 ;
					END ;

				  CASE  Lo(line_data[j])  OF

						Ext_not_symbol   :  formula[i] := Not_symbol ;
						Ext_and_symbol   :  formula[i] := And_symbol ;
						Ext_or_symbol    :  formula[i] := Or_symbol ;
						Ext_impl_symbol  :  formula[i] := Impl_symbol ;
						Ext_equiv_symbol :  formula[i] := Equiv_symbol ;

				  ELSE
						formula[i] := Chr(Lo(line_data[j])) ;
				  END ;

				  i := i + 1 ;  
										
				END ;
1:
				IF  i > 1  THEN
				  BEGIN 
					FOR  i := i  TO  Formula_High_Bound  DO 
						formula [i] := ' ' ;    
					build_formula := True ;
				  END
				ELSE
				  build_formula := False ;
		  END ;
END ;


{ ************************************************************************
  *                                                                      *
  *		    Apply editing function to marked area or all lines		     *	
  *                                                                      *
  ************************************************************************ }

PROCEDURE   apply_to_marked_or_all (proc: Proc_type) ;
VAR
		lp,
		lp_start,
		lp_end,
		last:	 Line_ptr ;

		first_on_screen: Integer ;

		x, y:  Integer ;
BEGIN
		x := x_index ;  
		y := y_index ;  
		WND_Save_Cursor (False) ;

		first_on_screen := 0 ;
		IF  mark_first <> Nil  THEN
		  BEGIN
			lp_start := mark_first ;
			lp_end := mark_last ;
			IF  precedes(mark_last, mark_first)  THEN
			  BEGIN
				lp_start := mark_last ;
				lp_end := mark_first ;
			  END ;
			lp := lp_start ;
			REPEAT
			  IF (first_on_screen = 0)  AND  (lp^.line_screen <> 0)  THEN
				first_on_screen := lp^.line_screen ;
			  last := lp ;
			  proc (lp) ;
			  lp := lp^.line_next ;
			UNTIL  last = lp_end ;
		  END
		ELSE
		  BEGIN
			first_on_screen := min_y_index ;
			lp := scratchpad_list_head.line_next ;
			WHILE  lp^.line_end <> 255  DO
			  BEGIN
				proc (lp) ;
				lp := lp^.line_next ;
			  END ;
		  END ;
		To_Screen (screen_lines[first_on_screen], first_on_screen, max_y_index) ;
		x_index := x ;
		y_index := y ;
		WND_Restore_Cursor ;
END ;


{ ************************************************************************
  *                                                                      *
  *           Remove current formula from list within tfeval             *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  clear_eval_all ;	FORWARD ;

PROCEDURE  remove_formula (lp: Line_ptr) ; FAR ;
VAR
		f:      Char ;
		p:      Line_ptr ;
BEGIN
		p := lp ;
		IF  lp = Nil  THEN
				p := screen_lines [y_index] ;

		IF  p <> Nil  THEN
		  WITH  p^  DO
			BEGIN       
			  IF  (Lo(line_select) <> Ord(' '))  AND
				  (Lo(line_name)   <> Ord(' '))  THEN 
				BEGIN
				  f := Chr(line_name) ;       
				  IF  remove (f)  THEN        
					BEGIN
					  Video_char (Ord(' ') ,line_select) ;
					  clear_eval_all ;
					END
				  ELSE
					Video_char (Ord('?'), line_select) ;
				END ;
			END ;

		IF  (lp = Nil)  AND  (p <> Nil)  THEN
				update_line (p) ; 
END ;


{ ************************************************************************
  *                                                                      *
  *           Remove all formulas from list within tfeval                *
  *                                                                      *
  ************************************************************************ }

PROCEDURE remove_all ;
BEGIN
		apply_to_marked_or_all (remove_formula) ;
END ;


{ ************************************************************************
  *                                                                      *
  *                       	Marking a block                              *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  set_mark (on: Boolean) ;
VAR
		lp,
		lp_start,
		lp_end,
		last:      Line_ptr ;
BEGIN
		lp_start := mark_first ;
		lp_end := mark_last ;
		IF  precedes(mark_last, mark_first)  THEN
		  BEGIN
				lp_start := mark_last ;
				lp_end := mark_first ;
		  END ;
				
		lp := lp_start ;
		REPEAT
			last := lp ;
			IF  on  THEN
			  Video_Char (Ord(''), lp^.line_mark) 
			ELSE                    
			  Video_Char (Ord(' '), lp^.line_mark) ;
			IF  lp^.line_screen <> 0  THEN
			  update_line (lp) ;
			lp := lp^.line_next ;
		UNTIL   last = lp_end ;
END ;


PROCEDURE  mark_lines ;
VAR
		lp:     Line_ptr ;
BEGIN
		lp := screen_lines [y_index] ;
		IF  lp = Nil  THEN
		  BEGIN
			insert_lines ;
			lp := screen_lines [y_index] ;
		  END ;

		IF  mark_first = Nil  THEN
		  BEGIN
			mark_first := lp ;
			mark_last := lp ;
			set_mark (True) ;
		  END
		ELSE
		  BEGIN
			set_mark (False) ;
			mark_first := Nil ;
			mark_last := Nil ;
		  END ;

		To_Screen (screen_lines [min_y_index], min_y_index, max_y_index) ; 
END ;


PROCEDURE  check_mark ;
VAR
		lp:     Line_ptr ;
BEGIN
		lp := screen_lines [y_index] ;
		IF  lp = Nil  THEN
		  BEGIN
			Exit ;
			insert_lines ;
			lp := screen_lines [y_index] ;
		  END ;
		set_mark (False) ;
		mark_last := lp ;
		set_mark (True) ;

		To_Screen (screen_lines [min_y_index], min_y_index, max_y_index) ; 
END ;


{ ************************************************************************
  *                                                                      *
  *           				Delete a line  				                 *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  delete_line (lp: Line_ptr) ;
VAR
		this, next:     Line_Ptr ;      
BEGIN
		next := lp^.line_next ; 
		unlink_line (lp) ;
		dispose (lp) ;

END ;


{ ************************************************************************
  *                                                                      *
  *           			 Delete the paste buffer		                 *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  delete_paste ;
VAR
		lp, next:       Line_ptr ;
BEGIN                                           
		lp := paste_list_head.line_next ;
		WHILE  lp^.line_end <> 255  DO
		  BEGIN
			next := lp^.line_next ;
			delete_line (lp) ;
			lp := next ;
		  END ;
END ;


{ ************************************************************************
  *                                                                      *
  *           			Process a block of lines 					     *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  process_block (cut, save: Boolean) ;
VAR
		lp,
		lp_start,
		lp_end,
		this,
		next,
		last,
		new:   Line_Ptr ;     

		first_on_screen:   Integer ;
BEGIN
		IF  NOT cut  THEN
		  save := True ;
		IF  save  THEN
		  delete_paste ;  

		IF  mark_first = Nil  THEN
		  BEGIN
			this := screen_lines[y_index] ;     
			IF  this <> Nil  THEN
			  BEGIN
				IF  cut  THEN
				  BEGIN
					next := this^.line_next ;
					remove_formula (this) ;
					unlink_line (this) ;
				  END
				ELSE
				  BEGIN
					new := get_line ;
					IF  new <> Nil THEN
					   Move (this^, new^, SizeOf(Line)) ;
					this := new ;
				  END ;
			    IF  save  AND (this <> Nil) THEN
				    link_after (this, @ paste_list_head) ;
			    IF  cut THEN
				  BEGIN
				    To_screen (next, y_index, max_y_index) ;
				    to_start_of_line ;
				  END ;
			  END ;
		  END 
		ELSE
		  BEGIN
			set_mark (False) ;
			first_on_screen := 0 ;
			lp_start := mark_first ;
			lp_end := mark_last ;
			IF  precedes(mark_last, mark_first)  THEN
			  BEGIN
				lp_start := mark_last ;
				lp_end := mark_first ;
			  END ;
				
			this := lp_start ;
			REPEAT
			  last := this ;
			  next := this^.line_next ;
			  IF  cut  THEN
				BEGIN
				  IF (first_on_screen = 0)  AND  (this^.line_screen <> 0)  THEN
					first_on_screen := this^.line_screen ;
					BEGIN
					  remove_formula (this) ;
					  unlink_line (this) ; 
					END	;
				END
			  ELSE
				BEGIN
				  new := get_line ;
				  IF  new <> Nil  THEN
					Move (this^, new^, SizeOf(Line)) ;
				  this := new ;
			  END ;
			  IF  save  AND (this <> Nil)  THEN
				  link_before (this, @ paste_list_head) ;	  
			  this := next ;
			UNTIL last = lp_end ;

			mark_first := Nil ;
			mark_last := Nil ;

			IF  cut  THEN
			  BEGIN
				To_Screen (next, first_on_screen, max_y_index) ;
				IF  y_index > first_on_screen  THEN
				  move_cursor (Up, y_index - first_on_screen) ;
			  END ;                   
		  END ;
		IF  cut  THEN
		  scratchpad_modified := True ;
END ;


{ ************************************************************************
  *                                                                      *
  *           			 Insert the paste buffer		                 *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  paste_lines ;
VAR
		lp,
		this,
		new,
		first,
		last:  Line_Ptr ;       
BEGIN
		insert_lines ;
		this := screen_lines[y_index] ; 
		lp :=  paste_list_head.line_next ;
		first := Nil ;
		WHILE  lp^.line_end <> 255  DO
		  BEGIN
			new := get_line ;
			IF  new <> Nil  THEN
			  BEGIN
				IF  first = Nil  THEN
				  first := new ;
				Move (lp^, new^, Sizeof(Line)) ;
				Video_Char (Ord(' '), new^.line_select) ;
				new^.line_parent := Nil ;
				link_before (new, this) ;
			  END ;
			lp := lp^.line_next ;
		  END ;
	   IF  first <> Nil  THEN
		  To_screen (first, y_index, max_y_index) ;
END ;


{ ************************************************************************
  *                                                                      *
  *               	Delete a character or a marked block                 *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  delete_char_or_block ;
VAR
		i: Integer ;
BEGIN
		scratchpad_modified :=  True ;
		IF  mark_first = Nil  THEN
		  BEGIN
			IF  screen_lines [y_index]  <>  Nil  THEN
			  BEGIN
				i := x_index - min_x_index + 1 ;
				WITH  screen_lines [y_index]^  DO
				  BEGIN
					IF  i > line_end  THEN
					  Exit ;
					Move ( line_data[i+1], line_data[i],
						   (line_end - i) Shl 1) ;
					line_data [line_end] := blank ;
					Dec (line_end) ;
				  END ;
				update_line (screen_lines [y_index]) ;
			  END ;
		  END
		ELSE
		  process_block (True, False) ;
END ;


{ ************************************************************************
  *                                                                      *
  *                       Insert a blank line                            *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  insert_blank_line ;
VAR
		new:    Line_ptr ;
BEGIN
		IF  screen_lines[y_index] <> Nil  THEN
		  BEGIN
			new := get_line ;
			IF  new <> Nil  THEN
			  BEGIN
				link_before (new, screen_lines[y_index]) ;
				end_marker_line	:= 0 ;
				To_screen (new, y_index, max_y_index) ;
				To_start_of_line ;
			  END ;
		  END ;         
END ;


{ ************************************************************************
  *                                                                      *
  *                   		Expanding a formula                          *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  expand_formula (lp: Line_ptr) ;
LABEL
		1, 2 ;
CONST
		max_sets = 3 ;
VAR
		model_line:     ARRAY [1..line_data_length] OF Word ;
		sets:           ARRAY [1..max_sets, 1..16] OF Word ;
		set_count:      ARRAY [1..max_sets] OF Integer ; 
		position:       ARRAY [1..max_sets] OF Integer ;
		n_sets:         Integer ;
		p, nl, ol:      Line_ptr ;
		i, j, k, l:     Integer ;
		length:         Integer ;
		nesting:        Integer ;
BEGIN
		p := lp ;
		IF  lp = Nil  THEN
		  p := screen_lines [y_index] ;
		IF  p = Nil  THEN
		  Exit ;

		FOR  i := 1  TO  max_sets  DO
		  BEGIN
			set_count [i] := 1 ;
			position [i] := 0 ;
		  END ;
		n_sets := 0 ;
		nesting := 0 ;

		WITH  p^ DO
		  BEGIN
			length := 0 ;
			FOR  i := 1  TO  line_end  DO
			  IF  Chr(Lo(Line_data[i])) = '<' THEN
				BEGIN
				  IF nesting > 0 THEN
					BEGIN
					  display_message ('ERROR','Nested sets', p) ;
					  Exit ;
					END ;
				  Inc (nesting) ;
				  Inc (n_sets) ;
				  IF  n_sets > max_sets THEN
					BEGIN
					  display_message ('ERROR','Too many sets', p) ;
					  Exit ;
					END ;
				  Inc (length) ;
				  position [n_sets] := length ;
				  set_count [n_sets] := 0 ;
				END
			  ELSE
				IF  Chr(Lo(line_data[i])) = '>' THEN
				  BEGIN
					IF  nesting = 0  THEN
					  BEGIN
						display_message ('ERROR','Missing <', p) ;
						Exit ;
					  END ;
					IF  set_count [n_sets] = 0  THEN
					  BEGIN
						display_message ('ERROR','Empty set', p) ;
						Exit ;
					  END ;
					Dec (nesting) ;
				  END
				ELSE
				  BEGIN
					IF  (Chr(Lo(line_data[i])) = '-') AND
					    (Chr(Lo(line_data[i+1])) = '-')  THEN
						GOTO 1 ;
				  	IF  nesting <> 0 THEN
					  BEGIN
					  	Inc (set_count [n_sets]) ;
					  	IF  set_count [n_sets] > 16  THEN
						  BEGIN
						  	display_message ('ERROR','Too many symbols in set', p) ;
						  	Exit ;
						  END ;
					  	sets [n_sets, set_count [n_sets]] := line_data [i] ;
					  END
				  	ELSE
					  BEGIN
					  	Inc (length) ;
					  	model_line [length] := line_data [i] ;
					  END ;
				  END ;
1:
			p^.line_parent := @empty_line ;
		  END ;

		IF nesting > 0 THEN
		  BEGIN
			display_message ('ERROR','Missing >', p) ;
			Exit ;
		  END ;

		ol := p ;

		FOR  i := 1  TO  set_count [1]  DO
		  BEGIN
			IF  position [1] <> 0  THEN
				model_line [position[1]] := sets [1, i] ;
			FOR  j := 1  TO  set_count [2]  DO
			  BEGIN
				IF  position [2] <> 0  THEN
				  model_line [position[2]] := sets [2, j] ;
				FOR  k := 1  TO  set_count [3]  DO
				  BEGIN
					IF  position [3] <> 0  THEN
					  model_line [position[3]] := sets [3, k] ;
					nl := get_line ;
					IF  nl = Nil  THEN
					  GOTO 2 
					ELSE
					  BEGIN
						FOR  l := 1  TO length  DO
						  nl^.line_data[l] := model_line [l] ;
						nl^.line_parent := p ;
						nl^.line_end := length ;
						link_after (nl, ol) ;
						ol := nl ;
					  END ;
				  END ;
			  END ;
		  END ;
2:
		To_Screen (screen_lines [min_y_index], min_y_index, max_y_index) ; 
END ;


PROCEDURE  remove_expansion (lp: Line_ptr) ;
VAR
		p, pp, next:    Line_ptr ;
BEGIN
		p := lp ;
		IF  lp = Nil  THEN
		  p := screen_lines [y_index] ;
		IF  p = Nil  THEN
		  Exit ;

		pp := scratchpad_list_head.line_next ;
		WHILE  pp^.line_end <> 255  DO
		  BEGIN
			next := pp^.line_next ;
			IF  pp^.line_parent = p  THEN
			  BEGIN
				unlink_line (pp) ;
				dispose (pp) ;
			  END ;
			pp := next ;
		  END ;
		To_Screen (scratchpad_list_head.line_next, min_y_index, max_y_index) ;
END ;


{ ************************************************************************
  *                                                                      *
  *               Add current formula to list within tfeval              *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  re_evaluate_all ; FORWARD ;

PROCEDURE  select_formula (lp: Line_ptr) ; FAR ;
VAR
		s:       Integer ;
		f:       Char ;
		p,q:     Line_ptr ;
		changed: Boolean ;
BEGIN
		p := lp ;
		IF  p = Nil  THEN
		  	p := screen_lines [y_index] ;

		IF  p <> Nil THEN
		  BEGIN
			WITH  p^  DO
			  BEGIN
				IF  line_parent = @empty_line  THEN
				  Exit ;
				IF  Lo(line_name) <> Ord(' ')  THEN 
				  BEGIN
					IF  build_formula(p)  THEN
					  BEGIN
						f := Chr(Lo(line_name)) ;
						s :=  select (f, formula, changed) ;
						IF  s = 0  THEN
						  BEGIN
							q := scratchpad_list_head.line_next ;
							WHILE  q^.line_end <> 255  DO
						  	  BEGIN
								IF  (p <> q)  AND  
									(Chr(Lo(q^.line_name)) = f)  THEN
							  	  BEGIN
									Video_Char (Ord(' '), q^.line_select) ;
		   							update_line (q) ; 
								  END ;
							  	q := q^.line_next ;
						  	  END ;
						  	Video_Char (Ord(':'), line_select) ;
							IF  changed  THEN
								re_evaluate_all ;
						  END
						ELSE
						  display_message ('ERROR','Syntax error in formula', p) ;
					  END 
					ELSE
					  display_message ('ERROR','Error in formula', p) ;
				   END 
				 ELSE
				   display_message ('ERROR','No name for formula', p) ;    
			  END ;
		  END ;
		IF  (lp = Nil)  AND  (p <> Nil)  THEN
		  update_line (p) ; 
END ;


{ ************************************************************************
  *                                                                      *
  *               		Select all named formulas                        *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  select_all ;
BEGIN
		apply_to_marked_or_all (select_formula) ;
END ;


{ ************************************************************************
  *                                                                      *
  *               	Clear all 'evaluate' fields                          *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  clear_eval (Lp: Line_Ptr) ; FAR ;
VAR
		p:      Line_ptr ;
BEGIN
		p := lp ;
		IF  p = Nil  THEN
			p := screen_lines [y_index] ;
		IF  p <> Nil  THEN
		  IF  Lo(p^.line_eval) <> Ord(' ')  THEN 
			BEGIN
			  set_eval (' ', p) ;
			  update_line (p) ;
			END ;
END ;

PROCEDURE  clear_eval_all ;
BEGIN
		apply_to_marked_or_all (clear_eval) ;
END ;


{ ************************************************************************
  *                                                                      *
  *                       Evaluate current formula                       *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  evaluate_formula (lp: Line_ptr) ;  FAR ;
VAR
		p:      Line_ptr ;
		dummy:  Boolean ;
		f, g:   Char ;
		s:      Integer ;
BEGIN
		p := lp ;
		IF  p = Nil  THEN
			p := screen_lines [y_index] ;

		IF  p^.line_parent = @empty_line  THEN
			Exit ;
		
		show_evaluation (True) ;

		IF  (p <> Nil)  AND build_formula (p) THEN
		  BEGIN
			IF  Lo(p^.line_name) = Ord(' ') THEN
			  BEGIN
				f := ' ' ;
				s := select (f, formula, dummy) ;
				IF  s = 0  THEN
				  BEGIN
					g := eval (f) ;
					set_eval (g, p) ;
				  END 
				ELSE
				  BEGIN
					display_message ('ERROR','Error in formula', p) ;
				  END ;
			  END               
			ELSE
			  BEGIN
				f := Chr(Lo(p^.line_name)) ;
				IF  Lo(p^.line_select) = Ord(' ')  THEN
				  BEGIN
					s := select (f, formula, dummy) ;
					IF  s = 0  THEN
					  BEGIN     
						g := eval (f) ;
						set_eval (g, p) ;
						dummy := remove (f) ;
					  END
					ELSE
					  display_message ('ERROR', 'Error in formula', p) ;
				  END
				ELSE
				  BEGIN
					g := eval (f) ;
					set_eval (g, p) ;
				  END ;
			  END ;
		  END ;

		show_evaluation (False) ; 

		IF  (lp = Nil)  AND  (p <> Nil)  THEN
			update_line (p) ; 
  END ;


{ ************************************************************************
  *                                                                      *
  *		    		Re-evaluate formulas already evaluated			     *	
  *                                                                      *
  ************************************************************************ }

PROCEDURE   re_evaluate_all ;
VAR
		lp:    Line_ptr ;
		x, y:  Integer ;
BEGIN
		x := x_index ;  
		y := y_index ;  
		WND_Save_Cursor (False) ;

		lp := scratchpad_list_head.line_next ;
		WHILE  lp^.line_end <> 255  DO
		  BEGIN
			IF  Lo(lp^.line_eval) <> Ord(' ') THEN
			  evaluate_formula (lp) ;
			lp := lp^.line_next ;
		  END ;

		To_Screen (screen_lines[min_y_index], min_y_index, max_y_index) ;
		x_index := x ;
		y_index := y ;
		WND_Restore_Cursor ;
END ;


{ ************************************************************************
  *                                                                      *
  *                       Evaluate all formulas                          *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  do_eval_all ;
BEGIN
		apply_to_marked_or_all (evaluate_formula) ;
END ;


{ ************************************************************************
  *                                                                      *
  *               			Assign name to formula                       *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  name_formula ;
VAR
		lp, p:  Line_ptr ;
		x, y:   Integer ;
		c:      Char ; 
		k:      Word ;
BEGIN
		lp := screen_lines [y_index] ;
		IF  lp <> Nil  THEN
		  WITH  lp^  DO
			BEGIN
				IF  line_parent = @empty_line  THEN
				  Exit ;
				IF  Lo(line_select) <> Ord(' ')  THEN
						remove_formula (Nil) ;
				x := x_index ;
				y := y_index ;
				Video_Char (Ord(' '), line_name) ;
				update_line(lp) ;
				gotoXY (x_name , y_index) ;    
				IF  tablet_save <> 0  THEN
				  BEGIN
					REPEAT UNTIL KeyPressed OR MS_LeftPressed OR MS_RightPressed ;
					IF  KeyPressed  THEN
					  get_key 
					ELSE
					  BEGIN
						IF  MS_RightPressed  THEN
						  BEGIN
							key := KB_Space ;
							REPEAT UNTIL NOT MS_RightDown ;
						  END
						ELSE
						  BEGIN
							IF  MS_WhereX > 15  THEN
							  BEGIN
								key := MemW [video_buffer:
									2 * (80 * (MS_WhereY-1) + (MS_WhereX-1))] ;
								IF  Ord(Lo(key)) > 128  THEN
							  		key := (key AND $ff00) OR (Ord(Lo(key)) - 128) ;
								REPEAT UNTIL NOT MS_LeftDown ;
							  END ;
						  END ;
					  END ;
				  END 
				ELSE
				  get_key ;
				c := Chr(Lo(key)) ;
				IF  special_key  OR  NOT (c IN ['a'..'u','w'..'z',' '])  THEN
						Video_Char (Ord(' '), line_name) 
				ELSE     
				  IF  c = ' '  THEN
						Video_Char (Ord(' '), line_name) 
				  ELSE
					BEGIN
						c := Chr(Lo(key) + 128) ;
						Video_Char (Ord(c), line_name) ;
				    END ;        

				gotoXY (x, y) ;                                                  
				update_line (lp) ;
			END ;
END ;


{ ************************************************************************
  *                                                                      *
  *               			Remove formula name		                     *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  clear_name (lp: Line_ptr) ;  FAR ;
VAR
		p:      Line_ptr ;
BEGIN
		p := lp ;
		IF  lp = Nil  THEN
				p := screen_lines [y_index] ;
		IF  p <> Nil  THEN
		  WITH  p^  DO
			BEGIN
			  IF  Lo(line_select) <> Ord(' ')  THEN
				  remove_formula (p) ;
			  Video_char (Ord(' '), line_name) ;
			END ;       

		IF  (lp = Nil)  AND (p <> Nil)  THEN
				update_line (p) ; 
END ;


{ ************************************************************************
  *                                                                      *
  *               	Remove name from all formulas                        *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  clear_name_all ;
BEGIN
		apply_to_marked_or_all (clear_name) ;
END ;


{ ************************************************************************
  *                                                                      *
  *                       Clear the scratchpad                           *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  clear_scratchpad ;
VAR
		i:        Integer ;
		lp, next: Line_ptr ;
BEGIN
		IF  tablet_save <> 0  THEN
		  close_tablet ;
		
		mark_first := Nil ;
		remove_all ;

		FOR  i := first_line TO max_lines  DO
			screen_lines [i] := Nil ;

		lp := scratchpad_list_head.line_next ;
		WHILE  lp^.line_end <> 255  DO
		  BEGIN
			next := lp^.line_next ;
			unlink_line (lp) ;
			dispose (lp) ;
			lp := next ;
		  END ;
		origin ;
		To_Screen (scratchpad_list_head.line_next, min_y_index, max_y_index) ;
		scratchpad_modified := False ;
END ;   


{ ****************************************************************************
  *                                                                          *
  *                               Open file                                  *
  *                                                                          *
  **************************************************************************** }

FUNCTION  open_file (option: Char) : Boolean ;
VAR
		i:          Integer ;   
		file_name:  String ;
		title:      String ;
		w:			Integer ;
BEGIN
		IF  option = 'r' THEN
			title := 'Reading a file into the scratchpad' 
		ELSE
			title := 'Writing the scratchpad to a file' ;

		file_name := current_file_name ;

		{ open a blank window to hide the change of font on slow machines }
		w := 1 ;
		WND_Open_Window (3, 1, 25, 80,
						 black_color, black_color,
						 WND_no_shadow, WND_no_frame,
						 '',
						 w) ;

		restore_video (False) ;
		FS_open ('.', '*.qne', '', title, 'n', file_name) ;
		WND_Close_Window (w) ;

		set_font_and_underline ;
		write_heading ;
		gotoXY (x_index, y_index) ;

		IF  file_name = ''  THEN
			open_file := False 
		ELSE
		  BEGIN
			Assign (file_var, file_name) ;                 

{$i-}                                                              
			IF option = 'w' THEN
				Rewrite (file_var)                                 
			ELSE
				Reset (file_var) ;                              
{$i+}                                                              
			IF  IOResult <> 0  THEN                                
				open_file := False 
			ELSE
			  BEGIN
				open_file := True ;
				current_file_name := file_name ;
			  END ;     
		  END ;
		
END ;


{ ************************************************************************
  *                                                                      *
  *                    	Write scratchpad to file                         *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  write_file ;
LABEL
		1 ;
VAR
		lp_start,
		lp_end:     Line_ptr ;
		i, j, k, l,	true_end:  Integer ;                          
BEGIN
		IF  NOT  open_file ('w')  THEN
			Exit ;
		IF  mark_first <> Nil  THEN
		  BEGIN
			lp_start := mark_first ;
			lp_end := mark_last ;
			IF  precedes(mark_last, mark_first)  THEN
			  BEGIN
				lp_start := mark_last ;
				lp_end := mark_first ;
			  END ;
			lp_end := lp_end^.line_next ;
		  END   
		ELSE
		  BEGIN
			lp_start := scratchpad_list_head.line_next ;
			lp_end := scratchpad_list_head.line_prev ;
			WHILE line_is_blank (lp_end)  DO
				lp_end := lp_end^.line_prev ;
			lp_end := lp_end^.line_next ;
		  END ;

		WHILE  (lp_start^.line_end <> 255)  AND  (lp_start <> lp_end)  DO
		  BEGIN

			true_end := lp_start^.line_end ;
			WHILE  (true_end > 0)  AND
				   (lp_start^.line_data[true_end] = blank)  DO
			  Dec (true_end) ;

			IF  true_end  < 1  THEN
			  GOTO 1 ;

			formula [1] := Chr(Lo(lp_start^.line_eval)) ;

			IF  Lo(lp_start^.line_name) <> Ord(' ') THEN
				formula [2] := Chr ( Lo(lp_start^.line_name) - 128) 
			ELSE
				formula [2] := ' ' ;

			IF  Lo(lp_start^.line_select) <> Ord(' ') THEN
				formula [3] := ':'  
			ELSE
				formula [3] := ' ' ;

			j := 4 ;

			FOR  k := 1  TO  true_end  DO       
			  BEGIN
				IF  Overline(lp_start^.line_data[k])  THEN
				  BEGIN                 
					formula [j] := escape_char ;
					j := j + 1 ;
					formula [j] := Chr (over_not) ;
					j := j + 1 ;
				  END ;

				IF  Lo(lp_start^.line_data[k]) = Ord(escape_char)  THEN
				  BEGIN                 
					formula [j] := escape_char ;
					j := j + 1 ;
				  END ;
						
				IF  Lo(lp_start^.line_data[k])  IN [ ext_not_symbol,
											   ext_and_symbol,
											   ext_or_symbol,
											   ext_impl_symbol,
											   ext_equiv_symbol,
											   alpha..zeta ]  THEN
				  BEGIN
					formula [j] := escape_char ;
					j := j + 1 ;
					formula [j] := Chr ( Lo(lp_start^.line_data[k]) - 128) ;
				  END
				ELSE    
				  formula [j] := Chr (Lo(lp_start^.line_data[k])) ;

				j := j + 1 ;

			  END { FOR };

			FOR  l := 1  TO  (j - 1)  DO                        
				Write (file_var, formula[l]) ;    
1:
			Write (file_var, Chr (end_of_record));

			lp_start := lp_start^.line_next ;

		  END { WHILE } ;

		  Close (file_var) ;
		  scratchpad_modified := False ;

END ;


{ ************************************************************************
  *                                                                      *
  *                    Read file into scratchpad                         *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  read_file ( lp: Line_ptr; ask: Boolean) ;
LABEL
		1, 2 ;
VAR
		i, j:      Integer ;
		nl:        Line_ptr ;
		character: Char ;
		error:	   Boolean ;	
		set_found: Boolean ;
BEGIN
		IF  (ask) AND (NOT  open_file ('r'))  THEN
		  BEGIN
			Exit ;
		  END ;

		error := False ;

		WHILE  (NOT eof(file_var)) AND (NOT error)  DO
		  BEGIN
			i := 1;
			REPEAT
				Read (file_var, character) ;
				IF  (character = Chr (line_feed))  AND  (NOT eof(file_var)) THEN
					Read (file_var, character) ;
				formula [i] := character ;
				i := i + 1 ;
			UNTIL  ( character = Chr (end_of_record) ) OR
				   ( character = Chr (line_feed)) OR
				   ( i > SizeOf(Formula_Buffer) );

			IF  i > SizeOf(Formula_Buffer) THEN
			  BEGIN
				display_message ('ERROR','Invalid data in file', Nil) ;
				error := True ;
				GOTO 2 ;
			  END ;

			nl := get_line ;
			IF nl = Nil THEN
			  BEGIN
				error := True ;
				GOTO 2 ;
			  END ;

			link_after (nl, lp) ;
			lp := nl ;

			IF  formula [1] = Chr (end_of_record)  THEN
				GOTO 2 ;

			WITH  nl^  DO
			  BEGIN
				j := 4 ;
				i := 1 ;
				WHILE  formula [j] <> Chr (end_of_record)  DO
				  BEGIN
					IF  i > line_data_length  THEN
					  BEGIN
						display_message ('ERROR','Invalid data in file', Nil) ;
						error := True ;
						GOTO 2 ;
					  END ;
					IF  formula [j] = escape_char THEN 
					  BEGIN
						Inc(j) ;

						CASE  formula[j]  OF
						  escape_char:
									 BEGIN
									   Video_char (Ord(escape_char) ,
												   line_data[i]) ;
									   Inc(i) ;
									 END ;
						  Chr(over_not) :
									   Video_fg (char_normal_overline_color,
												 line_data[i] );
					    ELSE                                                         
						  BEGIN
							Video_char (Ord (formula[j]) + 128, line_data[i]) ;
							IF  (formula[j] >= 'a') AND 
							    (formula[j] <= 'z') AND
							    (formula[j] <> 'v') THEN
							  BEGIN
								IF  Overline (line_data[i])  THEN
									Video_fg (char_greek_overline_color,
											  line_data[i]) 
								ELSE
									Video_fg (char_greek_color, line_data[i]) ;
							  END
							ELSE
								Video_fg (char_operator_color, line_data[i]) ;
							Inc(i) ;
						  END ;
						END { CASE };
					  END 
					ELSE
					  BEGIN
						Video_char ( Ord(formula [j]), line_data[i]) ;
						Inc(i) ;
					  END { IF } ;

					Inc(j) ;
		   
				END { WHILE } ;

				line_end := i - 1 ;

				{ check whether parent line of expanded formulas }

				nl^.line_parent := Nil ;
				set_found := False ;
				j := 4 ;
				WHILE  formula [j] <> Chr (end_of_record)  DO
				  BEGIN
					IF  formula [j] = '<' THEN
				  		set_found := True
					ELSE
					  IF  (formula [j] = '>')  AND	(formula [j-1] <> '\')	THEN
						BEGIN
					 	  IF  set_found  THEN
						  	BEGIN
							  nl^.line_parent := @empty_line ;
						  	  GOTO 1 ;
						  	END	;
						END
					  ELSE
						  IF  (formula [j] = '-')  AND
						  	  (formula [j-1] = '-')	THEN
						  GOTO 1 ;	
					Inc(j) ;
				  END ;
1:

				IF  formula [1] <> ' '  THEN
					Video_char ( Ord(formula [1]), line_eval) ;

				IF  formula [2] <> ' '  THEN
					Video_char ( Ord(formula[2]) + 128, line_name) ;        

				IF  formula [3] = ':' THEN
					select_formula (nl) ;   { destroys 'formula' }

			  END { WITH } ;
2:

		  END { WHILE } ;

		Close (file_var) ;
		IF  error  AND (nl <> Nil) THEN
		  BEGIN
		  	unlink_line (nl) ;
			dispose (nl) ;
		  END ;
		to_screen (scratchpad_list_head.line_next, min_y_index, max_y_index) ;
		origin ;


  END ;


{ ************************************************************************
  *                                                                      *
  *               		Build command bar menus                          *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  build_menus ;
BEGIN
		scratchpad_menu := MN_new_menu  ('',
										 WND_Single_frame,
										 WND_Transparent_shadow,
										 20,
										 white_color,
										 black_color,       
										 emphasis_bg_color,
										 button_special_fg_color,
										 button_bg_color,
										 intense_white_color,
										 emphasis_color) ;

		MN_menu_line (scratchpad_menu,KB_CtrlC ,10900, 'C\elear', 'Ctrl-C') ;
		MN_menu_line (scratchpad_menu,KB_AltR ,11000, '\eRead file', 'Alt-R') ;
		MN_menu_line (scratchpad_menu,KB_AltW ,11000, '\eWrite file', 'Alt-W') ;

		edit_menu := MN_new_menu ('',
							   	  WND_Single_frame,
							   	  WND_Transparent_shadow,
							      20,
							      white_color,
							      black_color,       
							   	  emphasis_bg_color,
							      button_special_fg_color,
							      button_bg_color,
							      intense_white_color,
							      emphasis_color) ;

		MN_menu_line (edit_menu,KB_GrayMinus ,10400, '\eCut ', 'Gray -') ;
		MN_menu_line (edit_menu,KB_GrayPlus ,10400, 'Cop\ey ', 'Gray +') ;
		MN_menu_line (edit_menu,KB_AltP ,10400, '\ePaste ', 'Ins') ;
		MN_menu_line (edit_menu,KB_Del ,10400, '\eDelete ', 'Del') ;
		MN_menu_line (edit_menu,0 ,0, '', '') ;
		MN_menu_line (edit_menu,KB_AltM ,10500, '\eMark ', 'Gray .') ;
		MN_menu_line (edit_menu,0 ,0, '', '') ;
		MN_menu_line (edit_menu,KB_AltI ,10400, '\eInsert line ', 'Alt-I') ;
		MN_menu_line (edit_menu,0 ,0, '', '') ;
		MN_menu_line (edit_menu,KB_F2 ,10200, '\eExpand ', 'F2') ;
		MN_menu_line (edit_menu,KB_AltF2 ,10200, '\eUndo Expand ', 'Alt-F2') ;

		formula_menu := MN_new_menu ( '',
									  WND_Single_frame,
									  WND_Transparent_shadow,
									  24,
									  white_color,
									  black_color,       
									  emphasis_bg_color,
									  button_special_fg_color,
									  button_bg_color,
									  intense_white_color,
									  emphasis_color) ;

		MN_menu_line (formula_menu,KB_F3 ,10600, '\eName', 'F3') ;
		MN_menu_line (formula_menu,KB_F4 ,10600, '\eRemove name', 'F4') ;
		MN_menu_line (formula_menu,KB_AltF4 ,10600, 'Remove all', 'Alt-F4') ;
		MN_menu_line (formula_menu,0 ,0, '', '') ;
		MN_menu_line (formula_menu,KB_F5 ,10700, '\eSelect', 'F5') ;
		MN_menu_line (formula_menu,KB_F6 ,10700, '\eDeselect', 'F6') ;
		MN_menu_line (formula_menu,KB_AltF5 ,10700, 'Select all', 'Alt-F5') ;
		MN_menu_line (formula_menu,KB_AltF6 ,10700, 'Deselect all', 'Alt-F6') ;
		MN_menu_line (formula_menu,0 ,0, '', '') ;
		MN_menu_line (formula_menu,KB_F7 ,10800, '\eEvaluate ', 'F7') ;
		MN_menu_line (formula_menu,KB_F8 ,10800, '\eClear evaluate', 'F8') ;
		MN_menu_line (formula_menu,KB_AltF7 ,10800, 'Evaluate all', 'Alt-F7') ;
		MN_menu_line (formula_menu,KB_AltF8 ,10800, 'Clear all ', 'Alt-F8') ;

		color_set_menu := MN_new_menu ( '', 
									   WND_Single_frame,
									   WND_Transparent_shadow,
									   24,
									   white_color,
									   black_color,      
									   emphasis_bg_color,
									   button_special_fg_color,
									   button_bg_color,
									   intense_white_color,
									   emphasis_color) ;

		MN_menu_line (color_set_menu, KB_Alt1, 0,
					 'Green on black  ', 'Alt-1') ;
		MN_menu_line (color_set_menu, KB_Alt2, 0,
					 'White on blue  ', 'Alt-2') ;
		MN_menu_line (color_set_menu, KB_Alt3, 0,
					 'White on green  ', 'Alt-3') ;
		MN_menu_line (color_set_menu, KB_Alt4, 0,
					 'White on cyan  ', 'Alt-4') ;
		MN_menu_line (color_set_menu, KB_Alt5, 0,
					 'Mono gray-scale', 'Alt-5') ;

		options_menu := MN_new_menu ( '', 
									  WND_Single_frame,
									  WND_Transparent_shadow,
									  27,
									  white_color,
									  black_color,      
									  emphasis_bg_color,
									  button_special_fg_color,
									  button_bg_color,
									  intense_white_color,
									  emphasis_color) ;

		IF  warning_on_exit  THEN
			MN_menu_line (options_menu,10 ,11100, 'Disable warning on <Exit>', '') 
		ELSE
			MN_menu_line (options_menu,10 ,11100, 'Enable warning on <Exit>', '') ;
		IF  warning_on_clear  THEN
			MN_menu_line (options_menu,11 ,11100, 'Disable warning on <Clear>', '') 
		ELSE
			MN_menu_line (options_menu,11 ,11100, 'Enable warning on <Clear>', '') ;
		MN_menu_line (options_menu,0 ,0, '', '') ;
		MN_menu_line (options_menu, color_set_menu,11100,  'Colors', '') ;

		help_menu := MN_new_menu ( '', 
								   WND_Single_frame,
								   WND_Transparent_shadow,
								   24,
								   white_color,
								   black_color,      
								   emphasis_bg_color,
								   button_special_fg_color,
								   button_bg_color,
								   intense_white_color,
								   emphasis_color) ;

		MN_menu_line (help_menu,1 ,0, 'Help on \eQUINE', '') ;
		MN_menu_line (help_menu,0 ,0, '', '') ;
		MN_menu_line (help_menu,2 ,0, '\eTruth-functional Logic', '') ;
		MN_menu_line (help_menu,0 ,0, '', '') ;
		MN_menu_line (help_menu,3 ,0, '\eLicense', '') ;
		MN_menu_line (help_menu,4 ,0, '\eDisclaimer', '') ;
		MN_menu_line (help_menu,0 ,0, '', '') ;
		MN_menu_line (help_menu,5 ,0, 'Help on \eHelp', '') ;
						
		about_bar :=  MN_new_menu_bar ( button_bg_color,
										button_emphasis_fg_color,
										button_emphasis_fg_color,
										button_bg_color, white_color,
										button_emphasis_fg_color ) ;

		MN_menu_bar_entry (about_bar, ' QUINE ', KB_AltQ, Command_entry) ;

		menu_bar :=  MN_new_menu_bar ( white_color, black_color,
									   emphasis_bg_color,
									   button_bg_color,
									   intense_white_color,
									   emphasis_color) ;

		MN_menu_bar_entry (menu_bar, '\eScratchpad', scratchpad_menu,
						   Menu_entry) ;
		MN_menu_bar_entry (menu_bar, '\eEdit',edit_menu,
						   Menu_entry) ;
		MN_menu_bar_entry (menu_bar, '\eFormula', formula_menu,
						   Menu_entry) ;
		MN_menu_bar_entry (menu_bar, '\eOptions',options_menu,
						   Menu_entry) ;
		MN_menu_bar_entry (menu_bar, '\eHelp',help_menu,
						   Menu_entry) ;
		MN_menu_bar_entry (menu_bar, 'E\exit', KB_AltX, Command_entry) ;


		scroll_up_page_bar := MN_new_menu_bar ( button_bg_color,
												button_emphasis_fg_color,
												button_emphasis_fg_color,
												button_emphasis_fg_color,
												button_emphasis_fg_color,
												button_emphasis_fg_color ) ;

		MN_menu_bar_entry (scroll_up_page_bar, '', KB_PgUp,
						   Command_entry) ;

		scroll_down_page_bar := MN_new_menu_bar ( button_bg_color,
												  button_emphasis_fg_color,
												  button_emphasis_fg_color,
												  button_emphasis_fg_color,
												  button_emphasis_fg_color,
												  button_emphasis_fg_color ) ;

		MN_menu_bar_entry (scroll_down_page_bar, '', KB_PgDown,
						   Command_entry) ;

		scroll_up_bar := MN_new_menu_bar ( button_bg_color,
										   button_emphasis_fg_color,
										   button_emphasis_fg_color,
										   button_emphasis_fg_color,
										   button_emphasis_fg_color,
										   button_emphasis_fg_color ) ;

		MN_menu_bar_entry (scroll_up_bar, '', KB_AltU,
						   Repeating_Command_entry ) ;

		scroll_down_bar := MN_new_menu_bar ( button_bg_color,
											 button_emphasis_fg_color,
											 button_emphasis_fg_color,
											 button_emphasis_fg_color,
											 button_emphasis_fg_color,
											 button_emphasis_fg_color ) ;

		MN_menu_bar_entry (scroll_down_bar, '', KB_AltD,
						   Repeating_Command_entry ) ;


END ;


{ ************************************************************************
  *                                                                      *
  *                       Display about...                               *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  display_about ;
VAR
		no_save:       Integer ;
		dummy:         Char ;   
BEGIN
		WND_Save_Cursor (False) ;
		about_save := 1 ;
		WND_Open_Window (4, 10, 15, 71,
						 button_emphasis_fg_color, button_bg_color,
						 WND_no_shadow, WND_no_frame,
						 '',
						 about_save) ;
  
		no_save := 0 ;
		WND_Open_Window (4, 11, 15, 70,
						 white_color, button_bg_color,
						 WND_no_shadow, WND_single_frame,
						 '',
						 no_save) ;

		gotoXY (37,6) ;
		TextColor (button_emphasis_fg_color) ;
		write ('Q U I N E') ;
		gotoXY (22,8) ;
		write ('A Scratchpad for Truth-functional Logic') ;
		gotoXY (35,10) ;
		TextColor (white_color) ;
		write ('Version  2.0') ;
		gotoXY (20,12) ;
		write ('Copyright (c) 1997  Gianfranco Boggio-Togna') ;
		gotoXY (30,13) ;
		write ('e-mail: gbt@computer.org') ;

		aboutc_save := 1 ;
		WND_Open_Window (18, 10, 23, 71,
						 button_emphasis_fg_color, button_bg_color,
						 WND_no_shadow, WND_no_frame,
						 '',
						 about_save) ;

		WND_Open_Window (18, 11, 23, 70,
						 white_color, button_bg_color,
						 WND_no_shadow, WND_single_frame,
						 '',
						 no_save) ;

		TextColor (button_emphasis_fg_color) ;
		gotoXY (14,19) ;
		write ('QUINE') ;
		TextColor (white_color) ;
		gotoXY (19,19) ;
		write (' comes with ') ; 
		TextColor (intense_white_color) ;
		write ('absolutely no warranty');
		TextColor (white_color) ;
		write ('. This is free') ;
		gotoXY (14,20) ;
		write ('software and you are welcome to redistribute it under') ;
		gotoXY (14,21) ;
		write ('certain conditions.  For details, please refer to the ') ;
		gotoXY (14,22) ;
		write ('"Disclaimer"  and  "License" entries in the Help menu.') ;

		MS_Show ;
		REPEAT  UNTIL  Keypressed  OR  MS_LeftPressed OR MS_RightPressed ;

		IF  about_save = 0  THEN
			fatal_error ('Internal error 2') ;
		WND_Close_window (about_save) ;
		WND_Close_window (aboutc_save) ;
		about_save := 0 ;
		WND_Restore_Cursor ;
END ;


{ ************************************************************************
  *                                                                      *
  *                       	Display a message                            *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  display_message (title, message: String; lp: Line_ptr) ;
VAR
		n, m, msg_save:  Integer ;
		no_save:	     Integer ;
		y:            	 Integer ; 
		ws:           	 String ;  
		old_lp:			 Line_ptr ;
BEGIN
		WND_Save_Cursor (False) ;

		ws := message ;
		WHILE (Length(ws)>0) AND (ws[Length(ws)] = ' ')  DO
			Dec(ws[0]) ;

		n := (Length(ws) + Length(ws) MOD 2) DIV 2 ;
		msg_save := 1 ;
		IF  (lp = Nil)  THEN
		  y := 9 
		ELSE  IF lp^.line_screen = 0  THEN
		  BEGIN
			m := screenful DIV 2 ;
			old_lp := lp ;
			WHILE  (lp^.line_end <> 255)  AND  (m > 0)  DO
			  BEGIN
				lp := lp^.line_prev ;
				Dec(m) ;
			  END ;
		    to_screen (lp, min_y_index, max_y_index) ;
		    y := old_lp^.line_screen + 1;
		  END
		ELSE
		  BEGIN
		    y := lp^.line_screen ;
		    IF  y < (max_y_index - 4)  THEN
		      Inc (y) 
			ELSE
			  Dec (y,5) ;
		  END ;

		WND_Open_Window (y, 40-n-4, y+4, 40+n+4,
						 intense_white_color, emphasis_bg_color,
						 WND_transparent_shadow, WND_no_frame,
						 '',
						 msg_save) ;
		IF  msg_save = 0  THEN
			fatal_error ('Internal error 3') ;

		no_save := 0 ;
		WND_Open_Window (y, 40-n-3, y+4, 40+n+3,
						 white_color, emphasis_bg_color,
						 WND_no_shadow, WND_double_frame,
						 title,
						 no_save) ;

		gotoXY (40-n+1, y+2) ;
		TextColor(intense_white_color) ;
		write (ws) ;

		REPEAT  UNTIL  KeyPressed  OR  MS_LeftPressed  OR  MS_RightPressed ;

		IF  KeyPressed  THEN
		  get_key ;
		IF  MS_LeftPressed  THEN
		  REPEAT UNTIL NOT  MS_LeftDown ;
		IF  MS_RightPressed THEN
		  REPEAT UNTIL NOT  MS_RightDown ;

		WND_Close_window (msg_save) ;
		WND_Restore_Cursor ;
END ;


{ ************************************************************************
  *                                                                      *
  *         Display warning when the scratchpad has been modified        *
  *                                                                      *
  ************************************************************************ }

FUNCTION  check_modified (default_key: Word) : Word  ;
CONST
	    hex_digit: String = '0123456789abcdef' ;
		labels: ARRAY [1..2, 1..2] OF String =
								( ('y\xes', 'N\xO') , ('Y\xES', 'n\xo') ) ;
VAR
		msg_save:  Integer ;
		no_save:   Integer ;
		i:		   Integer ;	
LABEL	
		1 ;
BEGIN
		check_modified := default_key ;  

		IF  NOT  scratchpad_modified  THEN
			Exit ;

		IF  (key = KB_AltX)  AND  NOT warning_on_exit  THEN
			Exit ;

		IF  (key = KB_CtrlC)  AND  NOT warning_on_clear  THEN
			Exit ;

		WND_Save_Cursor (False) ;

		WND_Open_Window (8, 10, 17, 71,
						 button_emphasis_fg_color, button_bg_color,
						 WND_no_shadow, WND_no_frame,
						 '',
						 msg_save) ;

		IF  msg_save = 0  THEN
			fatal_error ('Internal error 4') ;
		no_save := 0 ;
		WND_Open_Window (8, 11, 17, 70,
						 button_emphasis_fg_color, button_bg_color,
						 WND_no_shadow, WND_single_frame,
						 '',
						 no_save) ;

		gotoXY (16, 10) ;
		TextColor(white_color) ;
		write ('The contents of the scratchpad have not been saved.') ;
		IF  key = KB_AltX  THEN
		  BEGIN
			gotoXY (28, 12) ;
			write ('Do you really want to exit?') ;
		  END
		ELSE
		  BEGIN
			gotoXY (26, 12) ;
			write ('Do you really want to clear it?') ;
		  END ;

		i := 1 ;
1:
		WND_open_window (15, 32, 15, 38,
				 white_color, emphasis_bg_color,
				 WND_narrow_shadow, WND_no_frame,
				 '\' + hex_digit [button_emphasis_fg_color + 1] + labels [i,1],
				 no_save) ;

		WND_open_window (15, 44, 15, 49,
				 white_color, emphasis_bg_color,
				 WND_narrow_shadow, WND_no_frame,
				 '\' + hex_digit [button_emphasis_fg_color + 1] + labels [i,2],
				 no_save) ;

		REPEAT  UNTIL  KeyPressed  OR  MS_LeftPressed  OR  MS_RightPressed ;

		IF  KeyPressed  THEN
		  BEGIN
			get_key ;
			IF  (key = KB_Tab)  OR  (key = KB_ShiftTab)  THEN
			  BEGIN
				IF  i = 1  THEN	 Inc (i)  ELSE	Dec (i) ;
				GOTO 1 ;
			  END ;
			IF  (key = KB_Y)  OR  (key = KB_YY)	 OR
				( (key = KB_Enter)  AND  (i = 2) )	 THEN
				check_modified := default_key
			ELSE
				check_modified := KB_Esc ;
		  END
		ELSE
			IF  MS_LeftPressed  THEN
			  BEGIN
			  	REPEAT UNTIL NOT  MS_LeftDown ;
				IF  (MS_WhereX >= 32)  AND  (MS_WhereX <= 38)  AND
					(MS_WhereY = 15)  THEN
					check_modified := default_key
				ELSE
					check_modified := KB_Esc ;
			  END 
			ELSE
				IF  MS_RightPressed THEN
				  BEGIN
					  REPEAT UNTIL NOT  MS_RightDown ;
					  check_modified := KB_Esc ;
				  END ;	
		WND_Close_window (msg_save) ;
		WND_Restore_Cursor ;
END ;


{ ************************************************************************
  *                                                                      *
  *					Change 'exit' and 'clear' options				     *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  change_exit_option ;
VAR
		s: String ;
BEGIN
		IF  warning_on_exit  THEN
		  BEGIN
			s := 'Enable warning on <Exit>' ;
			warning_on_exit := False ;
		  END
		ELSE
		  BEGIN
			s := 'Disable warning on <Exit>' ;
			warning_on_exit := True ;
		  END ;
		MN_menu_line_new_text (options_menu, 1 ,s,  '') ;
END	;

PROCEDURE  change_clear_option ;
VAR
		s: String ;
BEGIN
		IF  warning_on_clear  THEN
		  BEGIN
			s := 'Enable warning on <Clear>' ;
			warning_on_clear := False ;
		  END
		ELSE
		  BEGIN
			s := 'Disable warning on <Clear>' ;
			warning_on_clear := True ;
		  END ;
		MN_menu_line_new_text (options_menu, 2 ,s,  '') ;
END	;


{ ************************************************************************
  *                                                                      *
  *						Interpret key pressed						     *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  interpret_key  ;
VAR
		s: String ;
BEGIN

		CASE  key  OF
		
		{ Editing }

		KB_Up:                  move_cursor (Up, 1) ;
		KB_Left:                move_cursor (Left, 1) ;
		KB_Down:                move_cursor (Down, 1) ;
		KB_Right:               move_cursor (Right, 1) ;
		KB_Tab:                 move_cursor (Right, 8 - (x_index MOD 8)) ;
		KB_ShiftTab:            IF  (x_index MOD 8) <> 0  THEN          
								  move_cursor (Left, x_index MOD 8) 
								ELSE
								  move_cursor (Left, 8) ;
		KB_Del:                 delete_char_or_block ;
		KB_Backspace :          BEGIN
								  IF  x_index = min_x_index  THEN
									escape := False     
								  ELSE
									BEGIN
										move_cursor (left, 1) ;
										delete_char_or_block ;
									END ;
								END ;
		KB_GrayIns,
		KB_Enter :              down_a_line ;

		KB_Home:                to_start_of_line ;
		KB_End:                 to_end_of_line ;
		KB_CtrlEnd:             delete_to_end_of_line ;
		KB_CtrlHome:            delete_from_start_of_line ;

		KB_CtrlK:               clear_line ; 

		KB_AltD:                scroll_screen (Down, 1) ;
		KB_AltU:                scroll_screen (Up, 1) ;
		KB_PgUp:                scroll_screen (Up, screenful) ;
		KB_PgDown:              scroll_screen (Down, screenful) ;

		KB_AltM,  
		KB_GrayPeriod:          mark_lines ;

		KB_AltC,                
		KB_GrayMinus:           process_block (True, True) ;

		KB_AltY,                
		KB_GrayPlus:            process_block (False, True) ; 

		KB_AltP,
		KB_Ins:                 paste_lines ;

		KB_CtrlEnter,
		KB_AltI:                insert_blank_line ;


		KB_F1:					IF  tablet_save <> 0  THEN
								   HLP_display_section (1944,'')
								ELSE
								   HLP_display_area (1, 'Help on QUINE') ;

		{ File handling }

		KB_AltR:                IF  screen_lines[y_index] = Nil  THEN
								  read_file (@scratchpad_list_head, True)   
								ELSE
								  read_file (screen_lines[y_index], True) ;

		KB_AltW:                write_file ;

		KB_CtrlC:               BEGIN
									key := check_modified (KB_CtrlC) ;
									IF  key = KB_CtrlC  THEN
										clear_scratchpad ;
								END ;

		KB_F2:                  expand_formula (Nil);

		KB_AltF2:               remove_expansion (Nil);


		{ Formula handling }

		KB_F3:                  name_formula ;

		KB_F4:                  IF  mark_first = Nil  THEN
								  clear_name(Nil) 
								ELSE
								  clear_name_all ;
		KB_AltF4:               clear_name_all ;

		KB_F5:                  IF  mark_first = Nil  THEN
								  select_formula(Nil) 
								ELSE
								  select_all ;
		KB_AltF5:               select_all ;  

		KB_F6:                  IF  mark_first = Nil  THEN
								  remove_formula (Nil) 
								ELSE
								  remove_all ;
		KB_AltF6:               remove_all ;

		KB_F7:                  IF  mark_first = Nil  THEN
								  evaluate_formula(Nil) 
								ELSE
								  do_eval_all ;
		KB_AltF7:               do_eval_all ;

		KB_F8:                  IF  mark_first = Nil  THEN
								  clear_eval (Nil) 
								ELSE
								  clear_eval_all ;
		KB_AltF8:               clear_eval_all ;        

		KB_CtrlM:               BEGIN
								  Str( MemAvail, s) ;
								  display_message ('', 'Available memory: '+s, Nil) ;
								END ;   

		KB_CtrlN:               BEGIN
								  Str( MaxAvail, s) ;
								  display_message ('', 'Max available block: '+s, Nil) ;
								END ;   

		KB_AltQ:                display_about  ;

		{ Options }

		KB_Alt1:				set_colors (1) ;
		KB_Alt2:				set_colors (2) ;
		KB_Alt3:				set_colors (3) ;
		KB_Alt4:				set_colors (4) ;
		KB_Alt5:				set_colors (5) ;


		{ Exit }

		KB_AltX:				key := check_modified (KB_AltX);


		ELSE                                      
		  IF  key = 1  THEN
				HLP_display_area (1, 'Help on QUINE') ;
		  IF  key = 2  THEN
				HLP_display_area (2, 'Truth-functional Logic') ;
		  IF  key = 3  THEN
				HLP_display_area (3, 'License') ;
		  IF  key = 4  THEN
				HLP_display_area (4, 'Disclaimer') ;
		  IF  key = 5  THEN
				HLP_display_area (5, 'Help on Help') ;
		  IF  key = 10  THEN
				change_exit_option ;
		  IF  key = 11  THEN
				change_clear_option ;

		END ;

END ;


{ ************************************************************************
  *                                                                      *
  *							Check tablet							     *
  *                                                                      *
  ************************************************************************ }

FUNCTION  check_tablet : Word ;
VAR
		c:      Word ;
		ch:     Word ;
		x, y:   Integer ;
		p:      Line_ptr ;
BEGIN
		check_tablet := 0 ;
		IF  tablet_save = 0  THEN
		  Exit ;        


		IF  MS_RightPressed  THEN
		  BEGIN
			close_tablet ;
			check_tablet := KB_Esc ;
			REPEAT UNTIL NOT  MS_RightDown ;
			Exit ;
		  END ;


		x := MS_WhereX ;
		y := MS_WhereY ;

		IF  (y > y_index)  AND  (y <= y_index + 1 + tablet_depth)  THEN 
		  BEGIN

			IF  y = y_index + 1  THEN  				   { top frame }
			  BEGIN
				check_tablet := KB_Up ;
				REPEAT UNTIL NOT  MS_LeftDown ;
				IF  y > min_y_index + 1  THEN
					MS_GotoXY (x, y-1) ; 
			  END ;

			IF  y = y_index + 1 + tablet_depth  THEN   { bottom frame }
			  BEGIN
			    check_tablet := KB_Down ;
				REPEAT UNTIL NOT  MS_LeftDown ;
				IF  y < max_y_index THEN
					MS_GotoXY (x, y+1) ; 
			  END ;

			IF  (y >= y_index+2)  AND (y <= y_index+6)  THEN { active area }
			  BEGIN
				IF  (x >= 5)  AND  (x <= 15)  THEN    { actions }
				  BEGIN
					p := screen_lines [y_index] ;
					IF  y = y_index + 2  THEN
					  BEGIN
						IF  x < 10  THEN
						  	check_tablet := KB_F3
						ELSE
						  	check_tablet := KB_F4 ;
					  END
					ELSE
					  IF  y = y_index + 4  THEN
						BEGIN
						  IF  x < 10  THEN
							check_tablet := KB_F5 
						  ELSE
							check_tablet := KB_F6 ;
						END
					  ELSE
						IF  y = y_index + 6  THEN
						  BEGIN
							IF  x < 10  THEN
							  check_tablet := KB_F7 
							ELSE
							  check_tablet := KB_F8 ;
						  END ;
					REPEAT UNTIL NOT  MS_LeftDown ;
				  END 
				ELSE  IF  (x > 15)  AND  (x < 78)  THEN
				  BEGIN
					c := MemW[video_buffer: 2 * (80 * (y - 1) + (x - 1))] ;
					ch := Ord(Lo(c)) ;
					IF  ch = Ord(' ')  THEN
						REPEAT UNTIL NOT  MS_LeftDown 
					ELSE IF  ch = 27 THEN
					  BEGIN
						check_tablet := KB_Left	;
						REPEAT UNTIL NOT  MS_LeftDown ;
					  END
					ELSE IF  ch = 26 THEN
					  BEGIN
						check_tablet := KB_Right ;
						REPEAT UNTIL NOT  MS_LeftDown ;
					  END
					ELSE IF  ch = 29 THEN
					  BEGIN
						check_tablet := KB_Space ;
						REPEAT UNTIL NOT  MS_LeftDown ;
					  END
					ELSE IF  ch = 16 THEN
					  BEGIN
						check_tablet := KB_Del ;
						REPEAT UNTIL NOT  MS_LeftDown ;
					  END
					ELSE IF  ch = 17 THEN
					  BEGIN
						check_tablet := KB_Backspace ;
						REPEAT UNTIL NOT  MS_LeftDown ;
					  END
					ELSE
					  BEGIN
						IF  ch >= 128  THEN
						  BEGIN
							escape := True ;
							c := ch - 128 ;
						  END ;
						process_text_character (c) ;
						REPEAT UNTIL NOT  MS_LeftDown ;
						check_tablet := KB_Esc ;
					  END ;
				  END 
				ELSE
				  BEGIN
					  IF  x = 2  THEN
				   		check_tablet := KB_Home ;
					  IF  x = 79  THEN
				   		check_tablet := KB_End ;
					  REPEAT UNTIL NOT  MS_LeftDown ;
				  END ;
			  END ;
		  END

		ELSE  { outside tablet }
		  BEGIN
			IF  y = 1 THEN    { on menu bar }
				  close_tablet 
			ELSE
			  BEGIN
				IF  y = y_index  THEN
				  BEGIN	
		  			x_index := MS_WhereX ;
		  			GotoXY (x_index , y_index) ;
				  END 
				ELSE
				 	close_tablet ;
			  END ;
		  END ;
END ;


{ ************************************************************************
  *                                                                      *
  *							Initialize units						     *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  initialize_units ;
CONST
	help_palette: HLP_palette =	
				  (0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15) ;
BEGIN

	{ Help unit }

	HLP_window_BG	:= white_color ;
	HLP_window_FG	:= Black ;

	HLP_text_BG 	:= background_color ;
	HLP_text_FG 	:= char_normal_color ;

	HLP_reference_FG :=	char_operator_color ;
	HLP_reference_BG := HLP_text_BG ;

	HLP_popup_reference_FG := char_greek_color ;
	HLP_popup_reference_BG := HLP_text_BG ;

	HLP_highlighted_reference_FG :=	intense_white_color ;
	HLP_highlighted_reference_BG := button_special_fg_color ;

 	HLP_popup_FG	:= button_bg_color ;
	HLP_popup_BG	:= white_color ;

	HLP_popup_text_BG := white_color ;
	HLP_popup_text_FG := black_color ;

	HLP_popup_title_line_FG := emphasis_bg_color ;
	HLP_popup_title_line_BG := white_color ;

	HLP_text_scroll_bar_BG			:=	button_special_fg_color;
	HLP_text_scroll_bar_button_BG	:=	button_bg_color ;
	HLP_text_scroll_bar_button_FG	:=	White ;

	HLP_button_BG 	   := button_bg_color ;
	HLP_button_FG 	   := white_color ;
	HLP_button_high_FG := button_emphasis_fg_color ;

	HLP_title_line_FG  := emphasis_bg_color ;
	HLP_title_line_BG  := HLP_window_BG	;

	HLP_first_line	   := 3 ;	
	HLP_last_line	   := 25 ;	

	IF  HLP_init (help_file_name, help_palette) = 0  THEN
		fatal_error ('Help file not found') ;

	{ File Selection unit }

	FS_text_BG            := white_color ;
	FS_text_FG            := black_color ;

	FS_file_list_BG       := black_color ;
	FS_file_list_FG       := char_normal_color ;

	FS_sel_file_BG        := button_special_fg_color ;
	FS_sel_file_FG        := intense_white_color ;

	FS_button_BG          := button_bg_color ;
	FS_button_FG          := white_color ;
	FS_button_special_BG  := emphasis_bg_color ;
	FS_button_high_FG     := button_emphasis_fg_color ;
	FS_button_special_FG  := button_special_fg_color ;

	FS_first_line 		  := 3 ;

	{ Windows unit }

	WND_Transparent_shadow_palette_number :=  dark_gray_color ;

END ;


{ ************************************************************************
  *                                                                      *
  *					  Initialize global variables 					     *
  *                                                                      *
  ************************************************************************ }

PROCEDURE	initialize_variables ;
VAR
		i, n:	Integer ;
BEGIN
		video_initialized := False ;
		scratchpad_modified :=  False ;
		current_file_name := '' ;
		tablet_save := 0 ;
		blank := video_char_attr ( Ord(' '),
								   char_normal_color,
								   background_color ) ;       
		end_marker := video_char_attr ( Ord(''),
								   		button_emphasis_fg_color,
								   		background_color ) ;       
		end_marker_line := 0 ;
		empty_line.line_next     := Nil ;
		empty_line.line_prev     := Nil ;
		empty_line.line_parent   := Nil ;
		empty_line.line_end      := 0 ;
		empty_line.line_screen   := 0 ;
		empty_line.line_eval     := video_char_attr (Ord(' '), 
													 char_eval_color,
													 background_color ) ;       
		empty_line.line_space1   := video_char_attr (Ord(' '),
													 char_normal_color, 
													 background_color ) ;       
		empty_line.line_name     := video_char_attr (Ord(' '),
													 char_greek_color,  
													 background_color ) ;       
		empty_line.line_select   := video_char_attr (Ord(' '),
													 char_greek_color,  
													 background_color ) ;       
		empty_line.line_mark     := video_char_attr (Ord(' '),
													 button_emphasis_fg_color,  
													 background_color ) ;       
		FOR  i := 1 TO line_data_length  DO
		  empty_line.line_data[i]  :=  blank ;
	
		Move (empty_line, scratchpad_list_head, Sizeof(Line)) ;
		scratchpad_list_head.line_next := @ scratchpad_list_head ;
		scratchpad_list_head.line_prev := @ scratchpad_list_head ;
		scratchpad_list_head.line_end := 255 ;
	
		Move (empty_line, paste_list_head, Sizeof(Line)) ;
		paste_list_head.line_next := @ paste_list_head ;
		paste_list_head.line_prev := @ paste_list_head ;
		paste_list_head.line_end := 255 ;

		mark_first := Nil ;
		mark_last := Nil ;

		n := 0 ;
		FOR  i := 1 TO max_lines  DO
		  BEGIN
			screen_lines [i] := Nil ;
			buffer_offset [i] := n ;
			Inc (n, 160) ;
		  END ;
	
		about_save := 0 ;
		color_set := 1 ;
		warning_on_exit := True ;
		warning_on_clear := True ;
END ;


{ ************************************************************************
  *                                                                      *
  *					   Process the command line						     *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  process_command_line ;
VAR
		parm:		  Integer ;
		parm_string:  String ;
		i:			  Integer ;
BEGIN
		parm := 1 ;

		WHILE  parm <= ParamCount  DO
		  BEGIN
			parm_string := ParamStr(parm) ;
			IF  (parm_string[1] = '-')  OR  (parm_string[1] = '/')  THEN
			   CASE	 parm_string[2]	OF
			   'c', 'C':	color_set := Ord(parm_string[3]) - Ord('0') ;
			   'w',	'W':	FOR  i := 3  TO Length(parm_string)  DO
				   				IF  UpCase(parm_string[i]) = 'E' THEN
					   				warning_on_exit := False 
								ELSE
						   			IF  UpCase(parm_string[i]) = 'C' THEN
						   				warning_on_clear := False ;
			   END 
			ELSE
			  BEGIN
				current_file_name := parm_string ;
				IF  Pos('.', current_file_name) = 0  THEN
			  		current_file_name := current_file_name + '.qne' ;
			  END ;
			  Inc (parm) ;
		  END ;
END ;


{ ************************************************************************
  *                                                                      *
  *					   		Set up the scratchpad 					     *
  *                                                                      *
  ************************************************************************ }

PROCEDURE  set_up_scratchpad ;

BEGIN
		write_heading;
		origin ;
		To_Screen (@ scratchpad_list_head, min_y_index, max_y_index) ;

		IF  current_file_name <> ''  THEN
		  BEGIN
			Assign (file_var, current_file_name) ;                 
			{$i-}                                                              
			Reset (file_var) ;                          
			{$i+}                                                              
			IF  IOResult = 0  THEN                                 
				read_file (@scratchpad_list_head, False)   
			ELSE
			  BEGIN
				Display_message ('ERROR','Cannot open '+ current_file_name, Nil) ;
				current_file_name := '' ;
			  END ;
		  END
		ELSE
			clear_scratchpad ;

		escape := False ;
		
		MS_Init ;
		MS_GotoXY (9, 1) ;                          
		MS_Hide ;
		origin ;
		WND_Save_Cursor (True) ;
		MS_GotoXY (min_x_index, min_y_index) ;                      
		MS_Show ;

		display_about ;

END ;


{ ************************************************************************
  *                                                                      *
  *				Check for mouse action in the main area				     *
  *                                                                      *
  ************************************************************************ }

FUNCTION  check_scratchpad_area : Word ;
VAR
		x, y:	Integer ; 
BEGIN
		check_scratchpad_area := 0 ;
		x := MS_WhereX ;
		y := MS_WhereY ;
		IF  MS_LeftPressed  THEN
	  	  BEGIN
	  	    IF  x < min_x_index  THEN
	  		  BEGIN
	  		    IF  y >= min_y_index  THEN
	  			  BEGIN
	  			    IF  y_index < y  THEN
	  				  move_cursor (Down, y - y_index ) ;
	  			    IF  y_index > y  THEN 
	  				  move_cursor (Up, y_index - y ) ;
	  			    IF  x_index < min_x_index  THEN
	  				  move_cursor (Right, min_x_index - x_index ) ;
	  			    IF  x_index > min_x_index  THEN 
	  				  move_cursor (Left, x_index - min_x_index ) ;
	  			    IF  mark_first = Nil  THEN
	  				  check_scratchpad_area := KB_GrayPeriod ;
	  			  END ;
	  		  END
	        ELSE
			  BEGIN
			    IF  y = y_index  THEN
				  BEGIN
				    IF  (screen_lines[y] = Nil)  OR
					    (screen_lines[y]^.line_end = 0)  THEN
				  	  x := min_x_index ;
				    IF  x_index < x  THEN
				  	  move_cursor (Right, x - x_index ) ;
				    IF  x_index > x  THEN 
				  	  move_cursor (Left, x_index - x ) ;
				    open_tablet ;
			   	  END
			    ELSE
				  IF  (x >= min_x_index)  AND  (y >= min_y_index) THEN
				    BEGIN
					  x_index := x ; 
					  IF  y_index < y  THEN
					    move_cursor (Down, y - y_index ) ;
					  IF  y_index > y  THEN 
					    move_cursor (Up, y_index - y ) ;
				    END ;
			  END ;
			  REPEAT UNTIL NOT  MS_LeftDown ;
		  END ;

		  IF  MS_RightPressed  THEN
		    BEGIN
			  IF  (x < min_x_index)    AND
			      (y >= min_y_index)   AND
			      (mark_first <> Nil)  THEN
			    check_scratchpad_area := KB_GrayPeriod 
			 ELSE
				check_scratchpad_area := KB_Esc ;
		    END ;
END ;


{ **************************************************************************
  * ********************************************************************** *
  * *                                                                    * *
  * *                     T H E    P R O G R A M			             * *
  * *                                                                    * *
  * ********************************************************************** *
  ************************************************************************** }

BEGIN
		initialize_variables ;
		process_command_line ;
        initialize_video ;
		initialize_units ;
		build_menus ;

		init { the truth-functional evaluation machine in tfeval.pas } ;

		set_up_scratchpad ;

		{ the main loop }

		REPEAT

		  show_evaluation_pos := 1 ;
		  MS_Show ;

		  key := 0 ;
		  action := 0 ;

		  REPEAT
		  UNTIL  KeyPressed  OR  MS_LeftPressed  OR  MS_RightPressed ; 

		  IF  KeyPressed  THEN
			  get_key ;

		  IF  key = 0  THEN
			  action := check_tablet 
		  ELSE
			  close_tablet ;

		  IF  action = 0  THEN
		  	action := MN_check_menu_bar (menu_bar, key) ;
		  IF  action = 0  THEN
		  	action := MN_check_menu_bar (about_bar, key) ;
		  IF  action = 0  THEN
		  	action := MN_check_menu_bar (scroll_up_bar, key) ;
		  IF  action = 0  THEN
		  	action := MN_check_menu_bar (scroll_down_bar, key) ;
		  IF  action = 0  THEN
		  	action := MN_check_menu_bar (scroll_up_page_bar, key) ;
		  IF  action = 0  THEN
		  	action := MN_check_menu_bar (scroll_down_page_bar, key) ;
		  IF  action = 0  THEN
		  	action := check_scratchpad_area ;

		  IF  action <> 0  THEN
		  	key := action ;

		  IF  special_key  THEN
			BEGIN
			  escape := False ;
			  interpret_key ;
			END 
		  ELSE 
			BEGIN
			  IF  (key = KB_BackSlash)  AND  (escape = False) THEN
				escape := True
			  ELSE
				process_text_character (key) ;
			END	;

		UNTIL  special_key  AND  (key = KB_AltX)  ;

		restore_video (True) ;
END.


