
/*
 * The Real SoundTracker - Basic 32bit integers mixer. Probably the
 * worst which you can come up with.
 *
 * Copyright (C) 1998-1999 Michael Krause
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include <string.h>
#include <stdlib.h>
#include <math.h>

#include "mixer.h"
#include "integer32-asm.h"

static int num_channels, mixfreq, amp = 8;
static gint32 *mixbuf = NULL;
static int mixbufsize = 0, clipflag;
static int stereo;

/* Reverb control variables */
#define REVERBERATION 110000L
static int RVc[8];
static int Ridx[8];
static gint32 *RVbufL[8];
static gint32 *RVbufR[8];
static guint32 md_reverb=0;

typedef struct integer32_channel {
    st_mixer_sample_info *sample;

    void *data;                 /* copy of sample->data */
    guint32 length;             /* length of sample (converted) */
    int type;                   /* 16 or 8 bits */

    int running;                /* this channel is active */
    guint32 current;            /* current playback position in sample (converted) */
    guint32 speed;              /* sample playback speed (converted) */

    guint32 loopstart;          /* loop start (converted) */
    guint32 loopend;            /* loop end (converted) */
    int loopflags;              /* 0 none, 1 forward, 2 pingpong */
    int direction;              /* current pingpong direction (+1 forward, -1 backward) */

    int volume;                 /* 0..64 */
    float panning;              /* -1.0 .. +1.0 */
} integer32_channel;

static integer32_channel channels[32];

#define ACCURACY           12         /* accuracy of the fixed point stuff, ALSO HARDCODED in the assembly routines!! */

#define MAX_SAMPLE_LENGTH  ((1 << (32 - ACCURACY)) - 1)

static void
integer32_setnumch (int n)
{
    g_assert(n >= 1 && n <= 32);

    num_channels = n;
}

static void
integer32_updatesample (st_mixer_sample_info *si)
{
    int i;
    integer32_channel *c;

    for(i = 0; i < 32; i++) {
	c = &channels[i];
	if(c->sample != si || !c->running) {
	    continue;
	}

	if(c->data != si->data
	   || c->length != MIN(si->length, MAX_SAMPLE_LENGTH) << ACCURACY
	   || c->loopflags != si->looptype) {
	    c->running = 0;
	}
	 
	/* No relevant data has changed. Don't stop the sample, but update
	   our local loop data instead. */
	c->loopstart = MIN(si->loopstart, MAX_SAMPLE_LENGTH) << ACCURACY;
	c->loopend = MIN(si->loopend, MAX_SAMPLE_LENGTH) << ACCURACY;
	c->loopflags = si->looptype;
	if(c->loopflags != ST_MIXER_SAMPLE_LOOPTYPE_NONE) {
	    // we can be more clever here...
	    c->current = c->loopstart;
	    c->direction = 1;
	}
    }
}

static gboolean
integer32_setmixformat (int format)
{
    if(format != 16)
	return FALSE;

    return TRUE;
}

static gboolean
integer32_setstereo (int on)
{
    stereo = on;
    return TRUE;
}

static void
integer32_setmixfreq (guint16 frequency)
{
    mixfreq = frequency;
}

static void
integer32_setampfactor (float amplification)
{
    amp = 8 * amplification;
}

static gboolean
integer32_getclipflag (void)
{
    return clipflag;
}

static void
integer32_reset (void)
{
    int i;

    memset(channels, 0, sizeof(channels));

    RVc[0] = (5000L * 44100) / REVERBERATION;
    RVc[1] = (5078L * 44100) / REVERBERATION;
    RVc[2] = (5313L * 44100) / REVERBERATION;
    RVc[3] = (5703L * 44100) / REVERBERATION;
    RVc[4] = (6250L * 44100) / REVERBERATION;
    RVc[5] = (6953L * 44100) / REVERBERATION;
    RVc[6] = (7813L * 44100) / REVERBERATION;
    RVc[7] = (8828L * 44100) / REVERBERATION;
   
    for(i = 0; i <= 7; i++) {
	Ridx[i] = 0;
    }

    for(i = 0; i <= 7; i++) {
	if(!RVbufL[i])
	    if(!(RVbufL[i] = (gint32*) calloc((RVc[i]+1),sizeof(gint32))))
		exit(0);
	memset(RVbufL[i], 0, (RVc[i]+1)*sizeof(gint32));
    }
   
    for(i = 0; i <= 7; i++) {
	if(!RVbufR[i])
	    if(!(RVbufR[i] = (gint32*) calloc((RVc[i]+1),sizeof(gint32))))
		exit(0);
	memset(RVbufR[i], 0, (RVc[i]+1)*sizeof(gint32));
    }
}

static void
integer32_setreverb (int r)
{
    md_reverb = r;
    return;
}

static int
integer32_getreverb (void)
{
    return md_reverb;
}

static void
integer32_startnote (int channel,
		  st_mixer_sample_info *s)
{
    integer32_channel *c = &channels[channel];

    c->sample = s;
    c->data = s->data;
    c->type = s->type;
    c->length = MIN(s->length, MAX_SAMPLE_LENGTH) << ACCURACY;
    c->running = 1;
    c->speed = 1;
    c->current = 0;
    c->loopstart = MIN(s->loopstart, MAX_SAMPLE_LENGTH) << ACCURACY;
    c->loopend = MIN(s->loopend, MAX_SAMPLE_LENGTH) << ACCURACY;
    c->loopflags = s->looptype;
    c->direction = 1;
}

static void
integer32_stopnote (int channel)
{
    integer32_channel *c = &channels[channel];

    c->running = 0;
}

static void
integer32_setsmplpos (int channel,
		   guint32 offset)
{
    integer32_channel *c = &channels[channel];

    if(offset < c->length >> ACCURACY) {
	c->current = offset << ACCURACY;
	c->direction = 1;
    } else {
	c->running = 0;
    }
}

static void
integer32_setfreq (int channel,
		float frequency)
{
    integer32_channel *c = &channels[channel];

    if(frequency > (0x7fffffff >> ACCURACY)) {
	frequency = (0x7fffffff >> ACCURACY);
    }

    c->speed = frequency * (1 << ACCURACY) / mixfreq;
}

static void
integer32_setvolume (int channel,
		  float volume)
{
    integer32_channel *c = &channels[channel];

    c->volume = 64 * volume;
}

static void
integer32_setpanning (int channel,
		   float panning)
{
    integer32_channel *c = &channels[channel];

    c->panning = panning;
}

static void *
integer32_mix (void *dest,
	    guint32 count,
	    gint8 *scopebufs[],
	    int scopebuf_offset)
{
    int todo;
    int i, j, t, *m, v;
    integer32_channel *c;
    int done, s;
    int val;
    int offs2end, oflcnt, looplen;
    gint16 *sndbuf;
    gint8 *scopedata = NULL;
    int vl = 0;
    int vr = 0;
    gint32 speedup;
    gint32 ReverbPct;

    if((stereo + 1) * count > mixbufsize) {
	g_free(mixbuf);
	mixbuf = g_new(gint32, (stereo + 1) * count);
	mixbufsize = (stereo + 1) * count;
    }
    memset(mixbuf, 0, (stereo + 1) * 4 * count);

    for(i = 0; i < num_channels; i++) {
	c = &channels[i];
	t = count;
	m = mixbuf;
	v = c->volume;

	if(scopebufs)
	    scopedata = scopebufs[i] + scopebuf_offset;

	if(!c->running) {
	    if(scopebufs)
		memset(scopedata, 0, count);
	    continue;
	}

	g_assert(c->sample->lock);
	g_mutex_lock(c->sample->lock);

	while(t) {
	    /* Check how much of the sample we can fill in one run */
	    if(c->loopflags) {
		looplen = c->loopend - c->loopstart;
		g_assert(looplen > 0);
		if(c->loopflags == ST_MIXER_SAMPLE_LOOPTYPE_AMIGA) {
		    offs2end = c->loopend - c->current;
		    if(offs2end <= 0) {
			oflcnt = - offs2end / looplen;
			offs2end += oflcnt * looplen;
			c->current = c->loopstart - offs2end;
			offs2end = c->loopend - c->current;
		    }			
		} else /* if(c->loopflags == ST_MIXER_SAMPLE_LOOPTYPE_PINGPONG) */ {
		    if(c->direction == 1)
			offs2end = c->loopend - c->current;
		    else 
			offs2end = c->current - c->loopstart;
		    
		    if(offs2end <= 0) {
			oflcnt = - offs2end / looplen;
			offs2end += oflcnt * looplen;
			if((oflcnt && 1) ^ (c->direction == -1)) {
			    c->current = c->loopstart - offs2end;
			    offs2end = c->loopend - c->current;
			    c->direction = 1;
			} else {
			    c->current = c->loopend + offs2end;
			    if(c->current == c->loopend)
				c->current--;
			    offs2end = c->current - c->loopstart;
			    c->direction = -1;
			}
		    }
		}
		g_assert(offs2end >= 0);
		g_assert(c->speed != 0);
		done = offs2end / c->speed + 1;
	    } else /* if(c->loopflags == LOOP_NO) */ {
		done = (c->length - c->current) / c->speed;
		if(!done) {
		    c->running = 0;
		    break;
		}
	    }

	    g_assert(done > 0);

	    if(done > t)
		done = t;
	    t -= done;

	    g_assert(c->current >= 0 && (c->current >> ACCURACY) < c->length);

	    if(stereo) {
		vl = 64 - ((c->panning + 1.0) * 32);
		vr = (c->panning + 1.0) * 32;
	    }

	    /* This one does the actual mixing */
	    /* One of the few occasions where templates would be really useful :) */
	    if(c->type == ST_MIXER_SAMPLE_TYPE_16) {
		gint16 *data = c->data;
		if(scopebufs) {
		    if(stereo) {
#if defined(__i386__)
			j = mixerasm_stereo_16_scopes(c->current, c->speed * c->direction,
						      data, m, scopedata,
						      v, vl, vr,
						      done);
 
			m += 2 * done;
			scopedata += done;
#else
			for(j = c->current, s = c->speed * c->direction; done; done--, j += s) {
			    val = v * data[j >> ACCURACY];
			    *m++ += vl * val >> 6;
			    *m++ += vr * val >> 6;
			    *scopedata++ = val >> 8 >> 6;
			}
#endif
		    } else {
#if defined(__i386__)
			j = mixerasm_mono_16_scopes(c->current, c->speed * c->direction,
						    data, m, scopedata,
						    v,
						    done);
 
			m += done;
			scopedata += done;
#else
			for(j = c->current, s = c->speed * c->direction; done; done--, j += s) {
			    val = v * data[j >> ACCURACY];
			    *m++ += val;
			    *scopedata++ = val >> 8 >> 6;
			}
#endif
		    }
		} else {
		    if(stereo) {
			vl *= v;
			vr *= v;
#if defined(__i386__)
			j = mixerasm_stereo_16(c->current, c->speed * c->direction,
					       data, m,
					       vl, vr,
					       done);
 
			m += 2 * done;
			scopedata += done;
#else
			for(j = c->current, s = c->speed * c->direction; done; done--, j += s) {
			    val = data[j >> ACCURACY];
			    *m++ += vl * val >> 6;
			    *m++ += vr * val >> 6;
			}
#endif
		    } else {
#if defined(__i386__)
			j = mixerasm_mono_16(c->current, c->speed * c->direction,
					     data, m,
					     v,
					     done);
 
			m += done;
			scopedata += done;
#else
			for(j = c->current, s = c->speed * c->direction; done; done--, j += s) {
			    val = v * data[j >> ACCURACY];
			    *m++ += val;
			}
#endif
		    }
		}
	    } else {
		/* I've not done assembly routines for 8 bit mixing, sorry */
		gint8 *data = c->data;
		if(scopebufs) {
		    if(stereo) {
			for(j = c->current, s = c->speed * c->direction; done; done--, j += s) {
			    val = v * (data[j >> ACCURACY] << 8);
			    *m++ += vl * val >> 6;
			    *m++ += vr * val >> 6;
			    *scopedata++ = val >> 8 >> 6;
			}
		    } else {
			for(j = c->current, s = c->speed * c->direction; done; done--, j += s) {
			    val = v * (data[j >> ACCURACY] << 8);
			    *m++ += val;
			    *scopedata++ = val >> 8 >> 6;
			}
		    }
		} else {
		    if(stereo) {
			for(j = c->current, s = c->speed * c->direction; done; done--, j += s) {
			    val = v * (data[j >> ACCURACY] << 8);
			    *m++ += vl * val >> 6;
			    *m++ += vr * val >> 6;
			}
		    } else {
			for(j = c->current, s = c->speed * c->direction; done; done--, j += s) {
			    val = v * (data[j >> ACCURACY] << 8);
			    *m++ += val;
			}
		    }
		}
	    }

	    c->current = j;
	}

	g_mutex_unlock(c->sample->lock);
    }

/*
  Reverb code incorporated from libmikmod by Giles Constant.
  This is absolutely not the right place for this, and we must think
  about a general plug-in interface with per-channel effects some time.
 */

#define COMPUTE_LECHO(n) RVbufL[n-1] [Ridx[n-1] ]=speedup+((ReverbPct*RVbufL[n-1] [Ridx[n-1] ])>>7)
#define COMPUTE_RECHO(n) RVbufR[n-1] [Ridx[n-1] ]=speedup+((ReverbPct*RVbufR[n-1] [Ridx[n-1] ])>>7)
#define INCR_LOOP(n) Ridx[n-1] ++; if (Ridx[n-1] >= RVc[n-1] ) Ridx[n-1] = 0;

    if(md_reverb != 0) {
	ReverbPct=92+(md_reverb<<1);

	if (stereo) {
	    for(todo=0; todo < (count*2); todo+=2) {
      
		/* Compute the left channel echo buffers */
		speedup = mixbuf[todo]>>3;
		COMPUTE_LECHO(1); COMPUTE_LECHO(2); COMPUTE_LECHO(3); COMPUTE_LECHO(4);
		COMPUTE_LECHO(5); COMPUTE_LECHO(6); COMPUTE_LECHO(7); COMPUTE_LECHO(8);
	 
		speedup = mixbuf[todo+1]>>3;
		COMPUTE_RECHO(1); COMPUTE_RECHO(2); COMPUTE_RECHO(3); COMPUTE_RECHO(4);
		COMPUTE_RECHO(5); COMPUTE_RECHO(6); COMPUTE_RECHO(7); COMPUTE_RECHO(8);
	 
		INCR_LOOP(1); INCR_LOOP(2); INCR_LOOP(3); INCR_LOOP(4);
		INCR_LOOP(5); INCR_LOOP(6); INCR_LOOP(7); INCR_LOOP(8);
	 
		mixbuf[todo] += RVbufL[0][Ridx[0]]-RVbufL[1][Ridx[1]]+RVbufL[2][Ridx[2]]-RVbufL[3][Ridx[3]]+
		    RVbufL[4][Ridx[4]]-RVbufL[5][Ridx[5]]+RVbufL[6][Ridx[6]]-RVbufL[7][Ridx[7]];
		mixbuf[todo+1] += RVbufR[0][Ridx[0]]-RVbufR[1][Ridx[1]]+RVbufR[2][Ridx[2]]-RVbufR[3][Ridx[3]]+
		    RVbufR[4][Ridx[4]]-RVbufR[5][Ridx[5]]+RVbufR[6][Ridx[6]]-RVbufR[7][Ridx[7]];
	    }
	} else { // mono
	    for(todo=0; todo < (count); todo++) {
		/* Compute the left channel echo buffers */
		speedup = mixbuf[todo]>>3;
		COMPUTE_LECHO(1); COMPUTE_LECHO(2); COMPUTE_LECHO(3); COMPUTE_LECHO(4);
		COMPUTE_LECHO(5); COMPUTE_LECHO(6); COMPUTE_LECHO(7); COMPUTE_LECHO(8);
	       
		INCR_LOOP(1); INCR_LOOP(2); INCR_LOOP(3); INCR_LOOP(4);
		INCR_LOOP(5); INCR_LOOP(6); INCR_LOOP(7); INCR_LOOP(8);

		mixbuf[todo] += RVbufL[0][Ridx[0]]-RVbufL[1][Ridx[1]]+RVbufL[2][Ridx[2]]-RVbufL[3][Ridx[3]]+
		    RVbufL[4][Ridx[4]]-RVbufL[5][Ridx[5]]+RVbufL[6][Ridx[6]]-RVbufL[7][Ridx[7]];
	    }
	}
    }
   
    /* modules with many channels get additional amplification here */
    t = (4 * log(num_channels) / log(4)) * 64 * 8;

    for(sndbuf = dest, clipflag = 0, todo = 0; todo < (stereo + 1) * count; todo++) {
	gint32 a, b;

	a = mixbuf[todo];
	a *= amp;                  /* amplify */
	a /= t;

	b = CLAMP(a, -32768, 32767);
	if(a != b) {
	    clipflag = 1;
	}

	*sndbuf++ = b;
    }

    return dest + (stereo + 1) * 2 * count;
}

void
integer32_dumpstatus (st_mixer_channel_status array[])
{
    int i;

    for(i = 0; i < 32; i++) {
	if(channels[i].running) {
	    array[i].current_sample = channels[i].sample;
	    array[i].current_position = channels[i].current >> ACCURACY;
	} else { 
	    array[i].current_sample = NULL;
	}
    }
}

st_mixer mixer_integer32 = {
    "integer32",
    "Basic 32bit integers mixer",

    integer32_setnumch,
    integer32_updatesample,
    integer32_setmixformat,
    integer32_setstereo,
    integer32_setmixfreq,
    integer32_setampfactor,
    integer32_setreverb,
    integer32_getreverb,
    integer32_getclipflag,
    integer32_reset,
    integer32_startnote,
    integer32_stopnote,
    integer32_setsmplpos,
    integer32_setfreq,
    integer32_setvolume,
    integer32_setpanning,
    integer32_mix,
    integer32_dumpstatus,

    MAX_SAMPLE_LENGTH,

    NULL
};
