#include "cthugha.h"
#include "display.h"
#include "disp-sys.h"
#include "cth_buffer.h"
#include "interface.h"
#include "CthughaDisplay.h"
#include "SoundAnalyze.h"
#include "pcx.h"
#include "CthughaBuffer.h"

#include <GL/glut.h>

#include <math.h>


#ifndef M_PI2
#define M_PI2  6.2831385307
#endif


//
// Option to control how fine to make the mesh
//
class OptionS : public OptionInt {
    void setH() {
	h  = 1.0 / double(value);
	h2 = 2.0 / double(value);
    }
public:
    double h;
    double h2;

    OptionS() : OptionInt("mesh-size", 64, 128, 4) {
	setH();
    }

    virtual void change(int by) {
	OptionInt::change(by);
	setH();
    }
    virtual void change(const char * to) {
	OptionInt::change(to);
	setH();
    }
} S;

Option & MeshSize = S;


//
// Option to specify texture quality
//
class OptionTexture : public OptionInt {
public:
    OptionTexture() : OptionInt("texture-quality", 1, 3) {}

    virtual const char * text() const {
	switch(value) {
	case 0:
	    return "low";
	case 1:
	    return "medium";
	case 2:
	    return "high";
	}
	return "unknown quality";
    }
    virtual void change(const char * to) {
	if( strncasecmp(to, "low", 3) == 0) {
	    value = 0;
	} else if (strncasecmp(to, "medium", 6) == 0) {
	    value = 1;
	} else if (strncasecmp(to, "high", 4) == 0) {
	    value = 2;
	} else
	    OptionInt::change(to);
    }

    int needMipmaps() {
	return value >= 2;
    }
    void setTexParameters() {
	switch(value) {
	case 0:
	    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	    break;
	case 1:
	    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	    break;
	case 2:
	    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
	    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	    break;
	}
    }
} textureQuality;

Option & TextureQuality = textureQuality;



/* possible display-function */
class ScreenEntryGL : public CoreOptionEntry {
public:
    int (*screen)();
    int nBuffers;

    ScreenEntryGL(int (*f)(), const char * name, const char * desc, int nB=1) :
	CoreOptionEntry(name, desc), screen(f), nBuffers(nB) {
    }

    int operator()() {
	CthughaBuffer::nBuffers = nBuffers;
	return (*screen)();
    }
};


int screen_plate();
int screen_2plates();
int screen_plane();
int screen_2planes();
int screen_wave1();
int screen_wave();
int screen_2001();
int screen_height();
int screen_cube();
int screen_cwave();
int screen_nearRing();

static CoreOptionEntry * _screens[] = {
    new ScreenEntryGL(screen_plate,	"Plate",	"Plate"),
    new ScreenEntryGL(screen_2plates,	"2Plates",	"", 2),
    new ScreenEntryGL(screen_plane,	"Plane",	"" ),
    new ScreenEntryGL(screen_2planes,	"2Planes",	"", 2),
    new ScreenEntryGL(screen_wave1,	"Wave1",	"Wave 1" ),
    new ScreenEntryGL(screen_wave,	"Wave",		"Full Wave" ),
    new ScreenEntryGL(screen_2001,	"2001",		"Flight", 2),
    new ScreenEntryGL(screen_height,	"Height",	"Height field"),
    new ScreenEntryGL(screen_cube,	"Cube",		"Cube", 3),
    new ScreenEntryGL(screen_cwave,	"CWave",	"Centered Wave"),
    new ScreenEntryGL(screen_nearRing,	"NearRing",	"Almost a Ring"),
};
CoreOption screen(-1, "display", _screens, sizeof(_screens) / sizeof(CoreOptionEntry*));







//
// some helpers
//

void setPalette(Palette pal, int P) {
    //
    // for glide we must use an RGBA palette
    //
    static char rgba_pal[6][256][4];

    if( CthughaBuffer::buffers[P].palChanged) {
	int avg[3] = {0,0,0};
	for(int i=0; i < 256; i++) {
	    for(int j=0; j < 3; j++) {
		rgba_pal[P][i][j] = pal[i][j];
		avg[j] += (int)(pal[i][j]);
	    }
	    rgba_pal[P][i][3] = 0;
	}
	for(int j=0; j < 3; j++) 
	    rgba_pal[P][0][j] = avg[j] / 256;
    }

#if defined(GL_EXT_shared_texture_palette) && defined(SHARED_PALETTE) 
    glColorTableEXT(GL_SHARED_TEXTURE_PALETTE_EXT,	/* target */
		    GL_RGBA,				/* internal format */
		    256,				/* table size */
		    GL_RGBA,				/* table format */
		    GL_UNSIGNED_BYTE,			/* table type */
		    rgba_pal[P]);
    glEnable(GL_SHARED_TEXTURE_PALETTE_EXT);
#else
    glColorTableEXT(GL_TEXTURE_2D,			/* target */
		    GL_RGBA,				/* internal format */
		    256,				/* table size */
		    GL_RGBA,				/* table format */
		    GL_UNSIGNED_BYTE,			/* table type */
		    rgba_pal[P]);			/* the color table */
#endif
}

void setTexture(int P) {

    setPalette( CthughaBuffer::buffers[P].currentPalette, P);

    if( textureQuality.needMipmaps() ) {
	gluBuild2DMipmaps(GL_TEXTURE_2D,
			  GL_COLOR_INDEX8_EXT,
			  BUFF_WIDTH, BUFF_HEIGHT,
			  GL_COLOR_INDEX,
			  GL_UNSIGNED_BYTE,
			  CthughaBuffer::buffers[P].passiveBuffer);
    } else {
	glTexImage2D(GL_TEXTURE_2D, 0,
		     GL_COLOR_INDEX8_EXT,
		     BUFF_WIDTH, BUFF_HEIGHT, 0,
		     GL_COLOR_INDEX,
		     GL_UNSIGNED_BYTE,
		     CthughaBuffer::buffers[P].passiveBuffer);
    }
    textureQuality.setTexParameters();
    
    glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
    glEnable(GL_TEXTURE_2D);
}

void setPcxTexture(int i) {

    PCXEntry * pcxE = (PCXEntry*)CthughaBuffer::buffers[i].pcx.current();
    if( (pcxE == NULL) || (pcxE->data == NULL)) {	// no PCX, use buffer instead
	setTexture(i);
	return;
    }
 
    setPalette( CthughaBuffer::buffers[i].getPalette( pcxE->pal), i);

    
    gluBuild2DMipmaps(GL_TEXTURE_2D,
		      GL_COLOR_INDEX8_EXT,
		      pcxE->width, pcxE->height,
		      GL_COLOR_INDEX,
		      GL_UNSIGNED_BYTE,
		      pcxE->data);
    textureQuality.setTexParameters();

    glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
    glEnable(GL_TEXTURE_2D);
}


//
// Camera Positions
//
void CameraObject() {
    double camera[3];
    double t = getTime();

    camera[0] = 2.0 * sin( t * 0.3 + 1.01);
    camera[1] = 1.7 * sin( t * 0.5 + 1.10);
    camera[2] = 1.5 + 1.0 * sin( t * 0.4 + 0.20);
    
    gluLookAt(camera[0], camera[1], camera[2],   0, 0, 0,    0, 0, 1);
}

void CameraRotate() {
    double t = getTime();
    double a = 3.0 * sin( t * 0.3 + 1.01) + 1.7 * sin( t * 0.5 + 1.10); 
    gluLookAt(0, 0, 0,   0, 0, 2,   sin(a), cos(a), 0);
}

void CameraFixed() {
    gluLookAt(0, 0, 0,   0, 0, 2,   0, 1, 0);
}


//
// Map buffer to an arbitrary 3D-mesh
//
typedef float* (*meshPos)(int i, int j, double alpha);

#define tx(i)		(double(i) * S.h)
#define ty(i)		(double(i) * S.h)

void screen_mesh( meshPos pos, double alpha) {
    for(int j=0; j < S; j++) {
	glBegin( GL_TRIANGLE_STRIP);
	
	glTexCoord2f(tx(j),   ty(0)); glVertex3fv( pos(0,  j, alpha) );
	glTexCoord2f(tx(j+1), ty(0)); glVertex3fv( pos(0,j+1, alpha) );
	for(int i=1; i <= S; i++) {
	    glTexCoord2f(tx(j),   ty(i));     glVertex3fv( pos(i,  j, alpha) );
	    glTexCoord2f(tx(j+1), ty(i));     glVertex3fv( pos(i,j+1, alpha) );
	}
	glEnd();
    }
}



int screen_plate() {

    CameraObject();

    setTexture(0);

    glBegin( GL_POLYGON);
    glTexCoord2f(0.0, 0.0); glVertex2f(-1.0, -1.0);
    glTexCoord2f(1.0, 0.0); glVertex2f( 1.0, -1.0);
    glTexCoord2f(1.0, 1.0); glVertex2f( 1.0,  1.0);
    glTexCoord2f(0.0, 1.0); glVertex2f(-1.0,  1.0);
    glEnd();

#if 0
    CameraFixed();
    setPcxTexture(0);
    glBegin( GL_POLYGON);
    glTexCoord2f(0.0, 0.0); glVertex3f(-2.0, -2.0, 2.0);
    glTexCoord2f(1.0, 0.0); glVertex3f( 2.0, -2.0, 2.0);
    glTexCoord2f(1.0, 1.0); glVertex3f( 2.0,  2.0, 2.0);
    glTexCoord2f(0.0, 1.0); glVertex3f(-2.0,  2.0, 2.0);
    glEnd();
#endif

    return 0;
}

int screen_2plates() {

    CameraObject();

    setTexture(0);

    glBegin( GL_POLYGON);
    glTexCoord2f(0.0, 0.0); glVertex2f(-1.0, -1.0);
    glTexCoord2f(1.0, 0.0); glVertex2f( 1.0, -1.0);
    glTexCoord2f(1.0, 1.0); glVertex2f( 1.0,  1.0);
    glTexCoord2f(0.0, 1.0); glVertex2f(-1.0,  1.0);
    glEnd();

    glPushMatrix();

    setTexture(1);
    double t = getTime();
    double alpha = 520.0 * sin( t * 0.3 + 1.01) + 200.0 * sin( t * 0.5 + 1.10); 
    glRotatef(alpha, 0,0,1);


    glBegin( GL_POLYGON);
    glTexCoord2f(0.0, 0.0); glVertex3f(-0.8, -0.8, 0.25);
    glTexCoord2f(1.0, 0.0); glVertex3f( 0.8, -0.8, 0.25);
    glTexCoord2f(1.0, 1.0); glVertex3f( 0.8,  0.8, 0.25);
    glTexCoord2f(0.0, 1.0); glVertex3f(-0.8,  0.8, 0.25);
    glEnd();

    glPopMatrix();

    return 0;
}

int screen_plane() {
    CameraRotate();

    setTexture(0);

    glBegin( GL_POLYGON);
    glTexCoord2f( 0.0,  0.0); glVertex3f(-40.0, 2.0,  0.0);
    glTexCoord2f(10.0,  0.0); glVertex3f( 40.0, 2.0,  0.0);
    glTexCoord2f(10.0, 10.0); glVertex3f( 40.0, 2.0, 40.0);
    glTexCoord2f( 0.0, 10.0); glVertex3f(-40.0, 2.0, 40.0);
    glEnd();

    return 0;
}

int screen_2planes() {

    CameraRotate();

    setTexture(0);

    glBegin( GL_POLYGON);
    glTexCoord2f(  0.0,  0.0); glVertex3f(-40.0, 2.0, 0.0);
    glTexCoord2f(10.0,  0.0); glVertex3f( 40.0, 2.0, 0.0);
    glTexCoord2f(10.0, 10.0); glVertex3f( 40.0, 2.0,  40.0);
    glTexCoord2f(  0.0, 10.0); glVertex3f(-40.0, 2.0,  40.0);
    glEnd();

    setTexture(1);

    glBegin( GL_POLYGON);
    glTexCoord2f( 0.0,   0.0); glVertex3f(-40.0, -2.0, 0.0);
    glTexCoord2f(10.0,   0.0); glVertex3f( 40.0, -2.0, 0.0);
    glTexCoord2f(10.0, 10.0); glVertex3f( 40.0, -2.0,  40.0);
    glTexCoord2f( 0.0, 10.0); glVertex3f(-40.0, -2.0,  40.0);
    glEnd();

    return 0;
}


int screen_2001() {

    CameraFixed();

    setTexture(0);

    double alpha = fmod(getTime(), 10.0);

    glBegin( GL_POLYGON);
    glTexCoord2f( 0.0,  0.0+alpha); glVertex3f(2,-40.0, 0.0);
    glTexCoord2f(10.0,  0.0+alpha); glVertex3f(2, 40.0, 0.0);
    glTexCoord2f(10.0, 10.0+alpha); glVertex3f(2, 40.0, 40.0);
    glTexCoord2f( 0.0, 10.0+alpha); glVertex3f(2,-40.0, 40.0);
    glEnd();

    setTexture(1);

    glBegin( GL_POLYGON);
    glTexCoord2f( 0.0,  0.0+alpha); glVertex3f(-2,-40.0,  0.0);
    glTexCoord2f(10.0,  0.0+alpha); glVertex3f(-2, 40.0,  0.0);
    glTexCoord2f(10.0, 10.0+alpha); glVertex3f(-2, 40.0, 40.0);
    glTexCoord2f( 0.0, 10.0+alpha); glVertex3f(-2,-40.0, 40.0);
    glEnd();


    return 0;
}


int screen_wave1() {
    glPushMatrix();

    CameraObject();
    double alpha = fmod(getTime(), M_PI2);

    setTexture(0);

    glBegin( GL_TRIANGLE_STRIP);

    glTexCoord2f(0.0, 0.0); glVertex3f(-1.0, -1.0, soundAnalyze.intensity*sin(alpha));
    glTexCoord2f(1.0, 0.0); glVertex3f( 1.0, -1.0, soundAnalyze.intensity*sin(alpha));
    for(int i=1; i <= S; i++) {
	glTexCoord2f(0.0, double(i)/S); 
	glVertex3f(-1.0, 
		   -1.0 + 2.0*double(i) * S.h, 
		   soundAnalyze.intensity*sin( 7.0*double(i) * S.h + alpha) );

	glTexCoord2f(1.0, double(i)/S); 
	glVertex3f( 1.0, 
		   -1.0 + 2.0*double(i) * S.h, 
		    soundAnalyze.intensity*sin( 7.0*double(i) * S.h + alpha) );
    }
    glEnd();

    glPopMatrix();

    return 0;
}



float * wave(int i, int j, double alpha) {
    static float p[3];
    p[0] = (-1.0 + double(i) * S.h2);
    p[1] = (-1.0 + double(j) * S.h2);
    p[2] = soundAnalyze.intensity * sin(8.0*double(i) * S.h + alpha) * 
	sin(8.0*double(j) * S.h + alpha) ;

    return p;
}
int screen_wave() {
    CameraObject();
    double alpha = fmod(getTime(), M_PI2);

    setTexture(0);
    screen_mesh( wave, alpha);

    return 0;
}


static int sc_x = 1;
static int sc_y = 1;
float * height1(int x, int y, double alpha) {
    static float p[3];
    p[0] = (-1.0 + double(x) * S.h2);
    p[1] = (-1.0 + double(y) * S.h2);
    p[2] = 1.0 / 256.0 * double(passive_buffer[y * sc_y + x * sc_x]);
    return p;
}
int screen_height() {
    CameraObject();

    sc_x = (BUFF_WIDTH / S);
    sc_y = (BUFF_HEIGHT / S) * BUFF_WIDTH;
    
    setTexture(0);
    screen_mesh( height1, 0);

    return 0;
}



int screen_cube() {

    glPushMatrix();

    CameraObject();

    setTexture(0);
    glBegin( GL_POLYGON);
    glTexCoord2f(0.0, 0.0); glVertex3f(-1.0,-1.0,-1.0);
    glTexCoord2f(1.0, 0.0); glVertex3f(+1.0,-1.0,-1.0);
    glTexCoord2f(1.0, 1.0); glVertex3f(+1.0,+1.0,-1.0);
    glTexCoord2f(0.0, 1.0); glVertex3f(-1.0,+1.0,-1.0);
    glEnd();

    setTexture(1);
    glBegin( GL_POLYGON);
    glTexCoord2f(0.0, 0.0); glVertex3f(+1.0,-1.0,-1.0);
    glTexCoord2f(1.0, 0.0); glVertex3f(+1.0,-1.0,+1.0);
    glTexCoord2f(1.0, 1.0); glVertex3f(+1.0,+1.0,+1.0);
    glTexCoord2f(0.0, 1.0); glVertex3f(+1.0,+1.0,-1.0);
    glEnd();


    setTexture(2);
    glBegin( GL_POLYGON);
    glTexCoord2f(0.0, 0.0); glVertex3f(+1.0,-1.0,-1.0);
    glTexCoord2f(1.0, 0.0); glVertex3f(+1.0,-1.0,+1.0);
    glTexCoord2f(1.0, 1.0); glVertex3f(-1.0,-1.0,+1.0);
    glTexCoord2f(0.0, 1.0); glVertex3f(-1.0,-1.0,-1.0);
    glEnd();

    glPopMatrix();

    return 0;
}




float * cwave(int i, int j, double alpha) {
    const double x = double(i) * S.h - 0.5;
    const double y = double(j) * S.h - 0.5;

    const double r = x*x + y*y;
    static float p[3];
    p[0] = (-1.0 + double(i) * S.h2);
    p[1] = (-1.0 + double(j) * S.h2);
    p[2] = soundAnalyze.intensity * 0.2 *  sin( 25.0 * r - alpha) / (r + 0.5);

    return p;
}
int screen_cwave() {

    CameraObject();
    double alpha = fmod(getTime(), M_PI2);

    setTexture(0);
    screen_mesh(cwave, alpha);

    return 0;
}


float * nearRing(int i, int j, double /* alpha */) {
    static float XYZ[3];
    double F = 0.25 * (1.0 + (double(j) * S.h));
    double alpha = 8.0 * double(j) * S.h + getTime() * 0.5;
    double beta =  M_PI2 * double(i) * S.h;

    XYZ[0] = sin(alpha) * (1.0 + F * sin(beta) );
    XYZ[1] = F * cos(beta);
    XYZ[2] = cos(alpha) * (1.0 + F * sin(beta) );

    return XYZ;
}

int screen_nearRing() {
    CameraObject();

    setTexture(0);
    screen_mesh(nearRing, 0);

    return 0;
}

