/* Target-vector operations for controlling go32 processes, for GDB.
   Copyright 1994 Free Software Foundation, Inc.
   Contributed by DJ Delorie.
   Modified for V2 by CW Sandmann.

This file is part of GDB.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <fcntl.h>

#include "defs.h"
#include "frame.h"  /* required by inferior.h */
#include "inferior.h"
#include "target.h"
#include "wait.h"
#include "gdbcore.h"
#include "command.h"

extern char **environ;

/* Forward declaration */
extern struct target_ops go32_ops;

/* ͻ
     Go32's external debugger interface routines			
   ͼ */

#define SOME_PID 42

int prog_has_started = 0;

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <debug/v2load.h>
#include <debug/dbgcom.h>

#define r_ofs(x) ((int)(&(((TSS *)0)->x)))
static struct {
  int tss_ofs;
  int size;
} regno_mapping[] = {
  r_ofs(tss_eax), 4,
  r_ofs(tss_ecx), 4,
  r_ofs(tss_edx), 4,
  r_ofs(tss_ebx), 4,
  r_ofs(tss_esp), 4,
  r_ofs(tss_ebp), 4,
  r_ofs(tss_esi), 4,
  r_ofs(tss_edi), 4,
  r_ofs(tss_eip), 4,
  r_ofs(tss_eflags), 4,
  r_ofs(tss_cs), 2,
  r_ofs(tss_ss), 2,
  r_ofs(tss_ds), 2,
  r_ofs(tss_es), 2,
  r_ofs(tss_fs), 2,
  r_ofs(tss_gs), 2
};

static struct {
  int go32_sig;
  int gdb_sig;
} sig_map[] = {
  0, TARGET_SIGNAL_BUS,
  1, TARGET_SIGNAL_TRAP,
  2, TARGET_SIGNAL_UNKNOWN,
  3, TARGET_SIGNAL_TRAP,
  4, TARGET_SIGNAL_FPE,
  5, TARGET_SIGNAL_SEGV,
  6, TARGET_SIGNAL_ILL,
  7, TARGET_SIGNAL_FPE,
  8, TARGET_SIGNAL_BUS,
  9, TARGET_SIGNAL_FPE,
  10, TARGET_SIGNAL_BUS,
  11, TARGET_SIGNAL_SEGV,
  12, TARGET_SIGNAL_SEGV,
  13, TARGET_SIGNAL_SEGV,
  14, TARGET_SIGNAL_SEGV,
  16, TARGET_SIGNAL_FPE,
  31, TARGET_SIGNAL_ILL,
  -1,-1
};

void
init_go32_extdebug(int *argc, char ***argvx)
{
  int i;
  char **argv;
  char cmdline[128];
}

/*  */

static void
go32_open(char *name, int from_tty)
{
  printf("Use the `run' command to run go32 programs\n");
}

/*  */

static void go32_close(int quitting)
{
  printf("go32_close called\n");
}

/*  */

static void
go32_attach(char *args, int from_tty)
{
  printf("Use the `run' command to run go32 programs\n");
}

/*  */

static void
go32_detach(char *args, int from_tty)
{
  printf("go32_detach called\n");
}

/*  */

static int resume_is_step;

static void
go32_resume(int pid, int step, enum target_signal siggnal)
{
  resume_is_step = step;
}

/*  */

static int
go32_wait(int pid, struct target_waitstatus *status)
{
/*  printf("go32_wait %d\n", pid); */
  if (resume_is_step)
    a_tss.tss_eflags |= 0x0100;
  else
    a_tss.tss_eflags &= 0xfeff;
  run_child();
  if (a_tss.tss_irqn == 0x21)
  {
    status->kind = TARGET_WAITKIND_EXITED;
    status->value.integer = a_tss.tss_eax & 0xff;
  }
  else
  {
    int i;
    status->value.sig = TARGET_SIGNAL_UNKNOWN;
    for (i=0; sig_map[i].go32_sig != -1; i++)
      if (a_tss.tss_irqn == sig_map[i].go32_sig)
      {
        status->value.sig = sig_map[i].gdb_sig;
        break;
      }
    status->kind = TARGET_WAITKIND_STOPPED;
  }
  return SOME_PID;
}

/*  */

static void
go32_fetch_registers(int regno)
{
/*JHW*/
  int end_reg=regno+1;	/*just one reg initially*/

  if (regno < 0) {	/*do the all registers*/
     regno=0;	/*start at first register*/
     end_reg=sizeof(regno_mapping)/sizeof(regno_mapping[0]);	/*# regs in table*/
  }
  for (; regno<end_reg; regno++) {
     switch (regno_mapping[regno].size)
     {
       case 4:
       case 2:
         supply_register(regno, (char *)&a_tss + regno_mapping[regno].tss_ofs);
         break;
      default:	/*unknown register size*/
         printf("Invalid register size %d bytes in go32_fetch_register()", regno_mapping[regno].size);
         exit(1);
     }
  }
}

/*  */

static void store_register(int regno)
{
  char *rp = (char *)&a_tss + regno_mapping[regno].tss_ofs;
  int v = *(int *)(&registers[REGISTER_BYTE(regno)]);
  switch (regno_mapping[regno].size)
  {
    case 4:
      *(int *)rp = v;
      break;
    case 2:
      *(short *)rp = v;
      break;
   default:	/*unknown register size*/
      printf("Invalid register size %d bytes in store_register()", regno_mapping[regno].size);
      exit(1);
  }
}

static void
go32_store_registers(int regno)
{
  if (regno >= 0)
    store_register(regno);
  else
  {
    int r;
    for (r=0; r<sizeof(regno_mapping)/sizeof(regno_mapping[0]); r++)
      store_register(r);
  }
}

/*  */

static void
go32_prepare_to_store(void)
{
}

/*  */

static int
go32_xfer_memory(CORE_ADDR memaddr, char *myaddr, int len, int write,
		 struct target_ops *target)
{
/*  printf("go32_xfer_memory %x %x %d %d\n", memaddr, myaddr, len, write); */
  if (write)
    write_child(memaddr, myaddr, len);
  else
    read_child(memaddr, myaddr, len);
  return len;
}

/*  */

static void
go32_files_info(struct target_ops *target)
{
  printf_filtered("You are running a DJGPP V2 program\n");
}

/*  */

static void
go32_kill_inferior(void)
{
  printf("go32_kill_inferior called\n");
  /* nothing to do */
}

/*  */

extern char **environ;

static void
go32_create_inferior(char *exec_file, char *args, char **env)
{
  jmp_buf start_state;
  char *cmdline;
  char **env_save = environ;

  if (prog_has_started)
  {
    printf("The program has been started once already.  Please quit and restart gdb.\n");
    return;
  }

  cmdline = (char *)alloca(strlen(args)+4);
  cmdline[0] = strlen(args);
  strcpy(cmdline+1, args);
  cmdline[strlen(args)+1] = 13;

  environ = env;

  if (v2loadimage(exec_file, cmdline, start_state))
  {
    environ = env_save;
    printf("Load failed for image %s\n", exec_file);
    exit(1);
  }
  environ = env_save;

  edi_init(start_state);

  inferior_pid = SOME_PID;
  push_target(&go32_ops);
  clear_proceed_status();
  proceed((CORE_ADDR) -1, TARGET_SIGNAL_0, 0);
}

/*  */

static void
go32_mourn_inferior(void)
{
  unpush_target(&go32_ops);
}

/*  */

static int go32_can_run(void)
{
  return 1;
}

/*  */

static void ignore(void) {}
static void ignore2(char *a,int b) {}

/*  */

struct target_ops go32_ops = {
  "go32",			/* to_shortname */
  "go32 target process",	/* to_longname */
  "Program loaded by go32, when gdb is used as an external debugger",	/* to_doc */
  go32_open,			/* to_open */
  go32_close,			/* to_close */
  go32_attach,			/* to_attach */
  go32_detach, 			/* to_detach */
  go32_resume,			/* to_resume */
  go32_wait,			/* to_wait */
  go32_fetch_registers,		/* to_fetch_registers */
  go32_store_registers,		/* to_store_registers */
  go32_prepare_to_store,	/* to_prepare_to_store */
  go32_xfer_memory,		/* to_xfer_memory */
  go32_files_info,		/* to_files_info */
  memory_insert_breakpoint,	/* to_insert_breakpoint */
  memory_remove_breakpoint,	/* to_remove_breakpoint */
  ignore,			/* to_terminal_init */
  ignore,	 		/* to_terminal_inferior */
  ignore,			/* to_terminal_ours_for_output */
  ignore,			/* to_terminal_ours */
  ignore2,			/* to_terminal_info */
  go32_kill_inferior,		/* to_kill */
  0,				/* to_load */
  0,				/* to_lookup_symbol */
  go32_create_inferior,		/* to_create_inferior */
  go32_mourn_inferior,		/* to_mourn_inferior */
  go32_can_run,			/* to_can_run */
  0, 				/* to_notice_signals */
  0,				/* to_thread_alive */
  0,				/* to_stop */
  process_stratum,		/* to_stratum */
  0,				/* to_next */
  1,				/* to_has_all_memory */
  1,				/* to_has_memory */
  1,				/* to_has_stack */
  1,				/* to_has_registers */
  1,				/* to_has_execution */
  0,				/* sections */
  0,				/* sections_end */
  OPS_MAGIC			/* to_magic */
};

void
_initialize_inftarg ()
{
  add_target (&go32_ops);
}
