// Verwaltung der 1:1 Floppyimages

#include "pc64.h"

#define ProgCounter     *(word*)_MK_FP(wRAMSeg, 0x0800)
#define StackPtr        *(word*)_MK_FP(wRAMSeg, 0x0802)
#define XWord           *(word*)_MK_FP(wRAMSeg, 0x0804)
#define XReg            *(byte*)_MK_FP(wRAMSeg, 0x0804)
#define YWord           *(word*)_MK_FP(wRAMSeg, 0x0806)
#define YReg            *(byte*)_MK_FP(wRAMSeg, 0x0806)
#define Flags_SignZero  *(word*)_MK_FP(wRAMSeg, 0x0808)
#define Flags           *(byte*)_MK_FP(wRAMSeg, 0x0808)
#define SignZero        *(byte*)_MK_FP(wRAMSeg, 0x0809)
#define Overflow        *(byte*)_MK_FP(wRAMSeg, 0x080A)
#define Akku            *(byte*)_MK_FP(wRAMSeg, 0x080B)
#define Carry           *(byte*)_MK_FP(wRAMSeg, 0x080C)
#define hImageFile      *(int*)_MK_FP(wRAMSeg, 0x080E)
#define pbError         *(byte __far**)_MK_FP(wRAMSeg, 0x0810)

extern "C" flag __near __cdecl EmulateImage(word wRAMSeg, word wPC);
extern "C" byte __far __cdecl abROM[16384];

// Konstruktor
CImage::CImage() {
  pbBuffer = (byte*)malloc(0x0814 + 16);
  //trace("alloc %08lX pbBuffer %08lX", this, pbBuffer);
  assert(pbBuffer);
  memset(pbBuffer, 0, 0x0814 + 16);
  wRAMSeg = HIWORD(pbBuffer) + (LOWORD(pbBuffer) + 15 >> 4);
  acFile[0] = 0;
  hImageFile = 0;
  pbError = NULL;
  Reset();
}

// Destruktor
CImage::~CImage() {
  if (hImageFile) {
    _lclose(hImageFile);
  }
  if (pbError) {
    //trace("free %08lX pbError %08lX", this, pbError);
    free(pbError);
  }
  //trace("free %08lX pbBuffer %08lX", this, pbBuffer);
  free(pbBuffer);
}

// Typ liefern
word CImage::GetType() {
  return DEVIMAGE;
}

// Verzeichnis setzen
flag CImage::SetDir(char* pDir) {
  char acNew[80];
  _fullpath(acNew, pDir, 80);
  AnsiUpper(acNew);
  if (strcmp(acNew, acFile) == 0) {
    return TRUE;
  }
  int hNew = _lopen(acNew, READ_WRITE | OF_SHARE_DENY_NONE);
  if (hNew <= 0) {
    hNew = _lopen(acNew, READ | OF_SHARE_DENY_NONE);
    if (hNew <= 0) {
      return FALSE;
    }
  }
  if (hImageFile != 0) {
    _lclose(hImageFile);
  }
  hImageFile = hNew;
  strcpy(acFile, acNew);
  // Fehlercodes einlesen
  if (_filelength(hImageFile) == 683 * 256L + 683) {
    if (!pbError) {
      pbError = (byte*)malloc(683);
      //trace("alloc %08lX pbError %08lX", this, pbError);
      assert(pbError);
    }
    _llseek(hImageFile, 683 * 256L, 0);
    _lread(hImageFile, pbError, 683);
  } else {
    if (pbError) {
      //trace("free %08lX pbError %08lX", this, pbError);
      free(pbError);
      pbError = NULL;
    }
  }
  // ID einlesen
  _llseek(hImageFile, 17 * 21 * 256L + 162, 0);
  _lread(hImageFile, _MK_FP(wRAMSeg, 0x0016), 2);
  Command("I", 1);
  return TRUE;
}

// Verzeichnis liefern
void CImage::GetDir(char* pDir, word wMax) {
  strncpy(pDir, acFile, wMax);
  return;
}

// Gert zurcksetzen
word CImage::Reset() {
  StackPtr = 0x01FF;
  Akku = 0;
  XWord = 0;
  YWord = 0;
  Flags_SignZero = 0x0024;
  Overflow = 0;
  Carry = 0;
  if (!EmulateImage(wRAMSeg, 0xEAA0)) {
    return ENODEVICE;
  }
  return 0;
}

// Befehl abschicken
word CImage::Command(char* sCommand, word wLength) {
  *(byte*)_MK_FP(wRAMSeg, 0x83) = 0x0F;
  *(byte*)_MK_FP(wRAMSeg, 0x84) = 0x6F;
  for (word w = 0; w < wLength; w++) {
    *(byte*)_MK_FP(wRAMSeg, 0x85) = sCommand[w];
    if (!EmulateImage(wRAMSeg, 0xEA2F)) {
      return ENODEVICE;
    }
  }
  if (!EmulateImage(wRAMSeg, 0xEBE7)) {
    return ENODEVICE;
  }
  return 0;
}

// Fehlermeldung holen
word CImage::GetError(char* pBuffer, word wMax) {
  char* pc = pBuffer;
  word w;
  do {
    w = Get(15);
    if (wMax) {
      if ((byte)w < 32) {
        *pc = 0;
        wMax = 0;
      } else {
        *pc++ = (byte)w;
      }
    }
  } while (w < 256);
  if (wMax) {
    *pc = 0;
  }
  return atoi(pBuffer);
}

// Status in Image schreiben
flag CImage::Save(int hFile) {
  assert(hImageFile == 0);
  return _lwrite(hFile, _MK_FP(wRAMSeg, 0), 0x080E) == 0x080E;
}

// Status aus Image lesen
flag CImage::Load(int hFile) {
  assert(hImageFile == 0);
  return _lread(hFile, _MK_FP(wRAMSeg, 0), 0x080E) == 0x080E;
}

// Emulation beginnen
word CImage::Start() {
  assert(hImageFile == 0);
  hImageFile = _lopen(acFile, READ_WRITE | OF_SHARE_DENY_NONE);
  if (hImageFile <= 0) {
    hImageFile = _lopen(acFile, READ | OF_SHARE_DENY_NONE);
    if (hImageFile <= 0) {
      hImageFile = 0;
      return 0;
    }
  }
  // Fehlercodes einlesen
  if (_filelength(hImageFile) == 683 * 256L + 683) {
    if (!pbError) {
      pbError = (byte*)malloc(683);
      //trace("alloc %08lX pbError %08lX", this, pbError);
      assert(pbError);
    }
    _llseek(hImageFile, 683 * 256L, 0);
    _lread(hImageFile, pbError, 683);
  } else {
    if (pbError) {
      //trace("free %08lX pbError %08lX", this, pbError);
      free(pbError);
      pbError = NULL;
    }
  }
  // ID einlesen
  _llseek(hImageFile, 17 * 21 * 256L + 162, 0);
  _lread(hImageFile, _MK_FP(wRAMSeg, 0x0016), 2);
  return 0;
}

// Emulation beenden
word CImage::Stop() {
  if (hImageFile) {
    _lclose(hImageFile);
    hImageFile = 0;
  }
  if (pbError) {
    //trace("free %08lX pbError %08lX", this, pbError);
    free(pbError);
    pbError = NULL;
  }
  return 0;
}

// Datei ffnen
word CImage::Open(word wChannel, char* pcName, word wLength) {
  *(byte*)_MK_FP(wRAMSeg, 0x83) = (byte)wChannel;
  *(byte*)_MK_FP(wRAMSeg, 0x84) = (byte)(wChannel | 0xF0);
  for (word w = 0; w < wLength; w++) {
    *(byte*)_MK_FP(wRAMSeg, 0x85) = pcName[w];
    if (!EmulateImage(wRAMSeg, 0xEA2F)) {
      return ENODEVICE;
    }
  }
  char* pc = strchr(pcName, ',');
  if (pc && !strcmp(pc, ",D,W")) {
    abROM[0xD8A3 - 0xC000] = 0;
  }
  if (!EmulateImage(wRAMSeg, 0xEBE7)) {
    abROM[0xD8A3 - 0xC000] = 1;
    return ENODEVICE;
  }
  abROM[0xD8A3 - 0xC000] = 1;
  return 0;
}

// Zeichen schreiben
word CImage::Put(word wChannel, byte bValue) {
  *(byte*)_MK_FP(wRAMSeg, 0x83) = (byte)wChannel;
  *(byte*)_MK_FP(wRAMSeg, 0x84) = (byte)(wChannel | 0x60);
  *(byte*)_MK_FP(wRAMSeg, 0xF8) = 128;
  *(byte*)_MK_FP(wRAMSeg, 0x85) = bValue;
  if (!EmulateImage(wRAMSeg, 0xEA2F)) {
    return ENODEVICE;
  }
  return 0;
}

// Ende Schreiben
word CImage::Unlisten() {
  *(byte*)_MK_FP(wRAMSeg, 0xF8) = 0;
  if (!EmulateImage(wRAMSeg, 0xEBE7)) {
    return ENODEVICE;
  }
  return 0;
}

// Zeichen lesen
word CImage::Get(word wChannel) {
  *(byte*)_MK_FP(wRAMSeg, 0x83) = (byte)wChannel;
  *(byte*)_MK_FP(wRAMSeg, 0x84) = (byte)(wChannel | 0x60);
  if (!EmulateImage(wRAMSeg, 0xE909)) {
    return ENODEVICE;
  }
  word wReturn = *(byte*)_MK_FP(wRAMSeg, 0x85);
  byte bFlags = *(byte*)_MK_FP(wRAMSeg, 0x7A);
  if (!(bFlags & 0x80)) {
    wReturn |= EINPUT;
  }
  if (!(bFlags & 0x08)) {
    wReturn |= ENDOFFILE;
  }
  return wReturn;
}

/*
// Zeichen lesen
word CImage::Get(word wChannel) {
  *(byte*)_MK_FP(wRAMSeg, 0x83) = (byte)wChannel;
  *(byte*)_MK_FP(wRAMSeg, 0x84) = (byte)(wChannel | 0x60);
  if (!EmulateImage(wRAMSeg, 0xE909)) {
    return ENODEVICE;
  }
  byte bCh = *(byte*)_MK_FP(wRAMSeg, 0x82);
  word wReturn = *(byte*)_MK_FP(wRAMSeg, 0x023E + bCh);
  byte b = *(byte*)_MK_FP(wRAMSeg, 0xF2 + bCh);
  if (!(b & 0x80)) {
    wReturn |= EINPUT;
  }
  if (!(b & 0x08)) {
    wReturn |= ENDOFFILE;
  }
  return wReturn;
}
*/

// Block schreiben
word CImage::Write(word wChannel, byte* pbBuffer, word* pwCount) {
  *(byte*)_MK_FP(wRAMSeg, 0x83) = (byte)wChannel;
  *(byte*)_MK_FP(wRAMSeg, 0x84) = (byte)(wChannel | 0x60);
  *(byte*)_MK_FP(wRAMSeg, 0xF8) = 128;
  for (word w = *pwCount; w; w--) {
    *(byte*)_MK_FP(wRAMSeg, 0x85) = *pbBuffer++;
    if (!EmulateImage(wRAMSeg, 0xEA2F)) {
      *pwCount -= w;
      return ENODEVICE;
    }
  }
  *(byte*)_MK_FP(wRAMSeg, 0xF8) = 0;
  if (!EmulateImage(wRAMSeg, 0xEBE7)) {
    return ENODEVICE;
  }
  return 0;
}

// Block lesen
word CImage::Read(word wChannel, byte* pbBuffer, word* pwCount) {
  *(byte*)_MK_FP(wRAMSeg, 0x83) = (byte)wChannel;
  *(byte*)_MK_FP(wRAMSeg, 0x84) = (byte)(wChannel | 0x60);
  for (word w = *pwCount; w; w--) {
    if (!EmulateImage(wRAMSeg, 0xE909)) {
      *pwCount -= w;
      return ENODEVICE;
    }
    *pbBuffer++ = *(byte*)_MK_FP(wRAMSeg, 0x85);
    if (!(*(byte*)_MK_FP(wRAMSeg, 0x7A) & 8)) {
      *pwCount -= w - 1;
      return ENDOFFILE;
    }
  }
  return 0;
}

// Datei schlieen
word CImage::Close(word wChannel) {
  *(byte*)_MK_FP(wRAMSeg, 0x83) = (byte)wChannel;
  *(byte*)_MK_FP(wRAMSeg, 0x84) = (byte)(wChannel | 0xE0);
  if (!EmulateImage(wRAMSeg, 0xE8CE)) {
    return ENODEVICE;
  }
  return 0;
}

// Satzlnge einer relativen Datei liefern
byte CImage::GetRelLength(word wChannel) {
  word wIndex = *(byte*)_MK_FP(wRAMSeg, 0x022B + wChannel) & 15;
  return *(byte*)_MK_FP(wRAMSeg, 0x00C7 + wIndex);
}

// Disketten kopieren
flag CImage::BeginTrack(flag fWrite) {
  assert(hImageFile != 0);
  if (!fWrite) {
    Command("I", 1);
    char acError[40];
    if (GetError(acError, 40)) {
      return ErrorBox(acFile, acError);
    }
    assert(gpbError);
    if (pbError) {
      memcpy(gpbError, pbError, 683);
    }
  }
  _llseek(hImageFile, 0, 0);
  fWriteTrack = fWrite;
  wTrack = 1;
  return TRUE;
}

word CImage::ReadNextTrack(byte* pbBuffer) {
  assert(pbBuffer);
  assert(hImageFile > 0);
  assert(!fWriteTrack);
  assert(wTrack < 36);
  word wCount;
  if (wTrack < 25) {
    if (wTrack < 18) {
      wCount = 21 * 256;
    } else {
      wCount = 19 * 256;
    }
  } else {
    if (wTrack < 31) {
      wCount = 18 * 256;
    } else {
      wCount = 17 * 256;
    }
  }
  SetError(EREAD);
  if (_lread(hImageFile, pbBuffer, wCount) != (int)wCount) {
    return ErrorBox(acFile, NULL);
  }
  wTrack++;
  return wCount;
}

flag CImage::WriteNextTrack(byte* pbBuffer, word wLength) {
  assert(pbBuffer);
  assert(wLength == 17 * 256 || wLength == 18 * 256 || wLength == 19 * 256 || wLength == 21 * 256);
  assert(hImageFile > 0);
  assert(fWriteTrack);
  assert(wTrack < 36);
  SetError(EWRITE);
  if (_lwrite(hImageFile, pbBuffer, wLength) != (int)wLength) {
    return ErrorBox(acFile, NULL);
  }
  wTrack++;
  return TRUE;
}

flag CImage::EndTrack() {
  Command("I", 1);
  char acError[40];
  if (GetError(acError, 40)) {
    return ErrorBox(acFile, acError);
  }
  if (fWriteTrack) {
    assert(hImageFile > 0);
    for (int i = 0; i < 683; i++) {
      if (gpbError[i] != 1) {
        _llseek(hImageFile, 683 * 256L, 0);
        SetError(EWRITE);
        if (_lwrite(hImageFile, gpbError, 683) != 683) {
          return ErrorBox(acFile, NULL);
        }
        return TRUE;
      }
    }
    _chsize(hImageFile, 683 * 256L);
  }
  return TRUE;
}

// Datum holen
dword CImage::GetDateAndTime(word) {
  unsigned uDate;
  unsigned uTime;
  if (_dos_getftime(hImageFile, &uDate, &uTime) == -1) {
    return 0;
  }
  return ((dword)uDate << 16) | uTime;
}
