;Die Befehle des 6510-Prozessors

include pc64.inc

;Importierte Variablen und Funktionnen
const
extrn ReadTable:word
data
extrn awError:word
extrn c bModule:byte
extrn c acLoad:byte
data?
extrn ProgCounter:word
extrn Clock:word
extrn SaveDX:word
extrn StackPtr:word
extrn XReg:byte
extrn XWord:word
extrn YReg:byte
extrn YWord:word
extrn Flags_SignZero:word
extrn Flags:byte
extrn SignZero:byte
extrn Overflow:byte
extrn Akku:byte
extrn Events:byte
extrn QueryState:byte
code
IMP ReadIO,near
IMP WriteIO,near
IMP JumpToIO,near
IMP WriteCPU,near
IMP Update,near
IMP Stop,near
IMP ReadKey,near
extrn c Listen:far
extrn c Talk:far
extrn c ChListen:far
extrn c ChTalk:far
extrn c IECIn:far
extrn c IECOut:far
extrn c Unlisten:far
extrn c Untalk:far
KERNEL segment para public use16 'BSS'
KERNEL ends
RAM segment para public use16 'BSS'
RAM ends

ifdef DEBUG

const
extrn HexTable:word
data
extrn c wRunDebug:word
data?
extrn lpLine:dword
extrn fnContinue:word
extrn lReadAdr:dword
extrn lWriteAdr:dword
extrn wReadData:word
extrn wWriteData:word
extrn wParam:word
code
extrn NextCmd:near
extrn DumpText:far

else

code 1
ResetCPU proc near
  public ResetCPU
  mov AX,KERNEL                         ;Reset-Vektor holen
  mov ES,AX
  mov AX,ES:[1FFCh]
  mov ProgCounter,AX                    ;PC setzen
  xor AX,AX                             ;Register auf 0
  mov Akku,AL
  mov XWord,AX
  mov YWord,AX
  mov SignZero,AL                       ;Flags auf 0
  mov Overflow,AL
  mov byte ptr SaveDX,AL
  mov Flags,00100100b                   ;auer 1F und IF
  mov StackPtr,01FFh                    ;SP ans obere Stackende
  mov Clock,AX                          ;Taktzyklenzhler zurcksetzen
  mov Events,AL                         ;Keine besonderen Vorkommnisse
  mov QueryState,55h                    ;Zwischen 55h und AAh umschalten
  ret
ResetCPU endp

endif

code 1
EXP StartCPU,near
StartCPU proc near
  mov DX,SaveDX
  mov BP,Clock
  mov AX,RAM
  mov GS,AX
  mov SI,ProgCounter
  shld BX,SI,5                          ;Segmentbasis nach FS
  and BX,0000000000011110b
  mov FS,ReadTable[BX]
  ret
StartCPU endp

code 1
EXP StopCPU,near
StopCPU proc near
  mov SaveDX,DX
  mov Clock,BP
  mov ProgCounter,SI
  ret
StopCPU endp

;Adressierung implied
GETn macro
  READ equ <>
  WRITE equ <>
  DATA equ <Akku>
endm

;Adressierung implied bei 2 Befehlsbyte
GETn2 macro
  READ equ <>
  WRITE equ <>
  DATA equ <>
  ifdef DEBUG
    movzx CX,byte ptr FS:[SI]           ;Nchstes Byte holen
    mov CX,HexTable[ECX*2]
    mov ES:[DI+7],CX
  endif
  inc SI
endm

;Adressierung implied bei 3 Befehlsbyte
GETn3 macro
  READ equ <>
  WRITE equ <>
  DATA equ <>
  ifdef DEBUG
    movzx CX,byte ptr FS:[SI]           ;Erstes Byte holen
    mov CX,HexTable[ECX*2]
    mov ES:[DI+7],CX
    movzx CX,byte ptr FS:[SI+1]         ;Zweites Byte holen
    mov CX,HexTable[ECX*2]
    mov ES:[DI+9],CX
  endif
  add SI,2
endm

;Adressierung Byte (immediate)
GETb macro
  READ equ <>
  WRITE equ <>
  DATA equ <AL>
  mov AL,FS:[SI]                        ;Nchstes Byte nach AL holen
  inc SI
  ifdef DEBUG
    movzx CX,AL
    mov CX,HexTable[ECX*2]
    mov ES:[DI+7],CX
    mov byte ptr ES:[DI+17],'#'
    mov ES:[DI+18],CX
  endif
endm

;Adressierung Zeropage
GETz macro
  READ equ <READz>
  WRITE equ <WRITEz>
  DATA equ <AL>
  mov BL,FS:[SI]                        ;Adresse nach BX holen
  xor BH,BH
  inc SI                                ;PC weiterzhlen
  ifdef DEBUG
    mov AX,HexTable[EBX*2]
    mov ES:[DI+7],AX
    mov ES:[DI+17],AX
    mov ES:[DI+28],AX
    mov word ptr ES:[DI+26],'00'
  endif
endm

;Adressierung Zeropage,X
GETzx macro
  READ equ <READz>
  WRITE equ <WRITEz>
  DATA equ <AL>
  mov BL,FS:[SI]                        ;Adresse nach BX holen
  xor BH,BH
  inc SI                                ;PC weiterzhlen
  ifdef DEBUG
    mov AX,HexTable[EBX*2]
    mov ES:[DI+7],AX
    mov ES:[DI+17],AX
    mov word ptr ES:[DI+19],'X,'
  endif
  add BL,XReg                           ;X-Register addieren
  ifdef DEBUG
    mov AX,HexTable[EBX*2]
    mov ES:[DI+28],AX
    mov word ptr ES:[DI+26],'00'
  endif
endm

;Adressierung Zeropage,Y
GETzy macro
  READ equ <READz>
  WRITE equ <WRITEz>
  DATA equ <AL>
  mov BL,FS:[SI]                        ;Adresse nach BX holen
  xor BH,BH
  inc SI                                ;PC weiterzhlen
  ifdef DEBUG
    mov AX,HexTable[EBX*2]
    mov ES:[DI+7],AX
    mov ES:[DI+17],AX
    mov word ptr ES:[DI+19],'Y,'
  endif
  add BL,YReg                           ;Y-Register addieren
  ifdef DEBUG
    mov AX,HexTable[EBX*2]
    mov ES:[DI+28],AX
    mov word ptr ES:[DI+26],'00'
  endif
endm

;Adressierung absolute
GETa macro
  READ equ <READa>
  WRITE equ <WRITEa>
  DATA equ <AL>
  mov BX,FS:[SI]                        ;Adresse nach BX holen
  add SI,2                              ;PC weiterzhlen
  ifdef DEBUG
    movzx CX,BL
    mov AX,HexTable[ECX*2]
    mov ES:[DI+7],AX
    mov ES:[DI+19],AX
    mov ES:[DI+28],AX
    mov CL,BH
    mov AX,HexTable[ECX*2]
    mov ES:[DI+9],AX
    mov ES:[DI+17],AX
    mov ES:[DI+26],AX
  endif
endm

;Adressierung absolute,X
GETax macro
  READ equ <READa>
  WRITE equ <WRITEa>
  DATA equ <AL>
  mov BX,FS:[SI]                        ;Basisadresse nach BX holen
  add SI,2                              ;PC weiterzhlen
  ifdef DEBUG
    movzx CX,BL
    mov AX,HexTable[ECX*2]
    mov ES:[DI+7],AX
    mov ES:[DI+19],AX
    mov CL,BH
    mov AX,HexTable[ECX*2]
    mov ES:[DI+9],AX
    mov ES:[DI+17],AX
    mov word ptr ES:[DI+21],'X,'
  endif
  add BX,XWord                          ;X-Register addieren
  ifdef DEBUG
    mov CL,BL
    mov AX,HexTable[ECX*2]
    mov ES:[DI+28],AX
    mov CL,BH
    mov AX,HexTable[ECX*2]
    mov ES:[DI+26],AX
  endif
endm

;Adressierung absolute,Y
GETay macro
  READ equ <READa>
  WRITE equ <WRITEa>
  DATA equ <AL>
  mov BX,FS:[SI]                        ;Basisadresse nach BX holen
  add SI,2                              ;PC weiterzhlen
  ifdef DEBUG
    movzx CX,BL
    mov AX,HexTable[ECX*2]
    mov ES:[DI+7],AX
    mov ES:[DI+19],AX
    mov CL,BH
    mov AX,HexTable[ECX*2]
    mov ES:[DI+9],AX
    mov ES:[DI+17],AX
    mov word ptr ES:[DI+21],'Y,'
  endif
  add BX,YWord                          ;Y-Register addieren
  ifdef DEBUG
    mov CL,BL
    mov AX,HexTable[ECX*2]
    mov ES:[DI+28],AX
    mov CL,BH
    mov AX,HexTable[ECX*2]
    mov ES:[DI+26],AX
  endif
endm

;Adressierung indirect
GETi macro
  READ equ <READa>
  WRITE equ <WRITEa>
  DATA equ <AL>
  mov BX,FS:[SI]                        ;Zeiger nach BX holen
  add SI,2                              ;PC weiterzhlen
  ifdef DEBUG
    movzx CX,BL
    mov AX,HexTable[ECX*2]
    mov ES:[DI+7],AX
    mov ES:[DI+20],AX
    mov CL,BH
    mov AX,HexTable[ECX*2]
    mov ES:[DI+9],AX
    mov ES:[DI+18],AX
    mov byte ptr ES:[DI+17],'('
    mov byte ptr ES:[DI+22],')'
  endif
  shld CX,BX,5                          ;Speicherbasis nach ES
  and CX,0000000000011110b
  mov ES,ReadTable[ECX]
  mov AL,ES:[BX]                        ;Adresse nach BX holen
  inc BL
  mov BH,ES:[BX]
  mov BL,AL
  ifdef DEBUG
    mov ES,word ptr lpLine[2]
    mov CL,BL
    mov AX,HexTable[ECX*2]
    mov ES:[DI+28],AX
    mov CL,BH
    mov AX,HexTable[ECX*2]
    mov ES:[DI+26],AX
  endif
endm

;Adressierung indirect,X
GETix macro
  READ equ <READa>
  WRITE equ <WRITEa>
  DATA equ <AL>
  mov BL,FS:[SI]                        ;Zeiger nach BX holen
  xor BH,BH
  inc SI                                ;PC weiterzhlen
  ifdef DEBUG
    mov AX,HexTable[EBX*2]
    mov ES:[DI+7],AX
    mov byte ptr ES:[DI+17],'('
    mov ES:[DI+18],AX
    mov dword ptr ES:[DI+20],' )X,'
  endif
  add BL,XReg                           ;X-Register addieren
  mov BX,GS:[BX]                        ;Adresse nach BX holen
  ifdef DEBUG
    movzx CX,BL
    mov AX,HexTable[ECX*2]
    mov ES:[DI+28],AX
    mov CL,BH
    mov AX,HexTable[ECX*2]
    mov ES:[DI+26],AX
  endif
endm

;Adressierung indirect,Y
GETiy macro
  READ equ <READa>
  WRITE equ <WRITEa>
  DATA equ <AL>
  mov BL,FS:[SI]                        ;Zeiger nach BX holen
  xor BH,BH
  inc SI                                ;PC weiterzhlen
  ifdef DEBUG
    mov AX,HexTable[EBX*2]
    mov ES:[DI+7],AX
    mov byte ptr ES:[DI+17],'('
    mov ES:[DI+18],AX
    mov dword ptr ES:[DI+20],' Y,)'
  endif
  mov BX,GS:[BX]                        ;Basisadresse nach BX
  add BX,YWord                          ;Y-Register addieren
  ifdef DEBUG
    movzx CX,BL
    mov AX,HexTable[ECX*2]
    mov ES:[DI+28],AX
    mov CL,BH
    mov AX,HexTable[ECX*2]
    mov ES:[DI+26],AX
  endif
endm

;Adressierung relative
GETr macro
  READ equ <>
  WRITE equ <>
  DATA equ <>
  movsx AX,byte ptr FS:[SI]             ;Sprungdistanz nach AX holen
  inc SI                                ;PC weiterzhlen
  ifdef DEBUG
    shrd BX,AX,15
    and BX,0000000111111110b
    mov BX,HexTable[BX]
    mov ES:[DI+7],BX
    mov CX,SI
    add CX,AX
    shld BX,CX,9
    and BX,0000000111111110b
    mov BX,HexTable[BX]
    mov ES:[DI+17],BX
    shrd BX,CX,15
    and BX,0000000111111110b
    mov BX,HexTable[BX]
    mov ES:[DI+19],BX
  endif
endm

;Branch-Befehle anhand von ZF ausfhren
BRANCH macro Cond
  local Skip
  ifidni <Cond>,<set>
    je Skip                             ;Sprung nicht ausfhren
  elseifidni <Cond>,<clear>
    jne Skip                            ;Sprung nicht ausfhren
  else
    .err <BRANCH condition must be SET or CLEAR>
  endif
  add SI,AX                             ;Distanz zu PC addieren
  dec BP                                ;Ausfhrungszeit um 1 Takt erhhen
  ifdef DEBUG
    mov word ptr ES:[DI+26],'>-'
    mov byte ptr ES:[DI+56],'3'
  endif
Skip:
endm

;Byte an Adresse BX lesen (ndert AX,CX,DI,ES)
READa macro
  local Continue
  mov CX,offset Continue                ;Rcksprung fr IO
  shld DI,BX,5                          ;Speicherbasis nach DI
  and DI,0000000000011110b
  mov DI,ReadTable[DI]
  and DI,DI                             ;Ist das Segment 0?
  je ReadIO                             ;Ja, dann IO lesen
  mov ES,DI                             ;Speicherbasis nach ES
  mov AL,ES:[BX]                        ;Datenwert holen
Continue:
  ifdef DEBUG
    les DI,lpLine
    mov lReadAdr,EBX
    and EAX,000000FFh
    mov wReadData,AX
    mov byte ptr ES:[DI+30],':'
    mov CX,HexTable[EAX*2]
    mov ES:[DI+31],CX
  endif
endm

;Byte aus Zeropage lesen
READz macro
  mov AL,GS:[BX]                        ;Direkter Speicherzugriff ohne IO
  ifdef DEBUG
    mov lReadAdr,EBX
    and EAX,000000FFh
    mov wReadData,AX
    mov byte ptr ES:[DI+30],':'
    mov CX,HexTable[EAX*2]
    mov ES:[DI+31],CX
  endif
endm

;Byte an Adresse BX schreiben (ndert AX,BX,CX,DI,ES)
WRITEa macro
  local Continue
  ifdef DEBUG
    mov lWriteAdr,EBX
    and EAX,000000FFh
    mov wWriteData,AX
    mov byte ptr ES:[DI+33],'='
    mov CX,HexTable[EAX*2]
    mov ES:[DI+34],CX
  endif
  mov CX,offset Continue                ;Rcksprung fr IO
  mov AH,BH                             ;Schreiben in IO-Bereich?
  and AH,11110000b
  cmp AH,DH
  je WriteIO                            ;Ja, dann zum IO-Verteiler
  cmp BX,2                              ;Auf CPU-Ports 0 und 1 prfen
  jb WriteCPU                           ;Ja, dann Speicher umkonfigurieren
  mov GS:[BX],AL                        ;Sonst Wert ganz normal schreiben
Continue:
endm

;Byte in Zeropage schreiben (ndert AX,BX,CX,DI,ES)
WRITEz macro
  local Continue
  ifdef DEBUG
    mov lWriteAdr,EBX
    and EAX,000000FFh
    mov wWriteData,AX
    mov byte ptr ES:[DI+33],'='
    mov CX,HexTable[EAX*2]
    mov ES:[DI+34],CX
  endif
  mov CX,offset Continue                ;Rcksprung fr CPU-Port
  cmp BX,2                              ;Auf CPU-Ports 0 und 1 prfen
  jb WriteCPU                           ;Ja, dann Speicher umkonfigurieren
  mov GS:[BX],AL                        ;Sonst Wert ganz normal schreiben
Continue:
endm

;Sign Flag und Zero Flag setzen
SETSZ macro
  lahf                                  ;Flags nach AH holen
  mov SignZero,AH                       ;SF und ZF setzen
endm

;Sign Flag, Zero Flag und Carry Flag setzen
SETSZC macro
  lahf                                  ;Flags nach AH holen
  mov SignZero,AH                       ;SF und ZF setzen
  mov DL,AH                             ;CF setzen (Bit 0)
endm

;Sign Flag, Zero Flag, Carry Flag und Overflow Flag setzen
SETSZCV macro
  lahf                                  ;Flags nach AH holen
  mov SignZero,AH                       ;SF und ZF setzen
  mov DL,AH                             ;CF setzen (Bit 0)
  seto Overflow                         ;VF setzen (Bit 0)
endm

;Befehl mit verschiedenen Adressierungsarten definieren
CMD macro Name,AdrList
  Name1 substr <Name>,1,1
  Name2 substr <Name>,2,1
  Name3 substr <Name>,3,1
  NameStr catstr <'>,Name3,Name2,Name1,< '>
  irp i,<AdrList>
    Ticks substr <i>,1,1
    TicksStr catstr <'>,Ticks,<'>
    Adr substr <i>,2
    ProcName catstr <Name>,Adr
    GetAdr catstr <GET>,Adr
    code 4
    ProcName proc near
      ifdef DEBUG
        ifdif Name1,<?>
          mov dword ptr ES:[DI+12],NameStr
        endif
        if Ticks
          mov byte ptr ES:[DI+56],TicksStr
        endif
      endif
      GetAdr
      ACTION                            ;Eigentliche Befehlsfolge ausfhren
      if Ticks
        sub BP,Ticks                    ;Bentigte Zeit weiterzhlen
        ifndef DEBUG
          js Update                     ;Periodische Arbeiten alle 64ms
        endif
      endif
      NEXTCMD                           ;Nchsten Befehl ausfhren
    ProcName endp
  endm
endm

;Adressen der emulierten Befehle
const 2
EXP CommandTable,word
CommandTable label word
  dw BRKn ,ORAix,?02n ,ASOix,SKBn2,ORAz ,ASLz ,ASOz  ;00
  dw PHPn ,ORAb ,ASLn ,ANCb ,SKWn3,ORAa ,ASLa ,ASOa
  dw BPLr ,ORAiy,?12n ,ASOiy,SKBn2,ORAzx,ASLzx,ASOzx ;10
  dw CLCn ,ORAay,?N1n ,ASOay,SKWn3,ORAax,ASLax,ASOax
  dw JSRa ,ANDix,?22n ,RLAix,BITz ,ANDz ,ROLz ,RLAz  ;20
  dw PLPn ,ANDb ,ROLn ,ANCb ,BITa ,ANDa ,ROLa ,RLAa
  dw BMIr ,ANDiy,?32n ,RLAiy,SKBn2,ANDzx,ROLzx,RLAzx ;30
  dw SECn ,ANDay,?N1n ,RLAay,SKWn3,ANDax,ROLax,RLAax
  dw RTIn ,EORix,?42n ,LSEix,SKBn2,EORz ,LSRz ,LSEz  ;40
  dw PHAn ,EORb ,LSRn ,ALRb ,JMPa ,EORa ,LSRa ,LSEa
  dw BVCr ,EORiy,?52n ,LSEiy,SKBn2,EORzx,LSRzx,LSEzx ;50
  dw CLIn ,EORay,?N1n ,LSEay,SKWn3,EORax,LSRax,LSEax
  dw RTSn ,ADCix,?62n ,RRAix,SKBn2,ADCz ,RORz ,RRAz  ;60
  dw PLAn ,ADCb ,RORn ,ARRb ,JMPi ,ADCa ,RORa ,RRAa
  dw BVSr ,ADCiy,?72n ,RRAiy,SKBn2,ADCzx,RORzx,RRAzx ;70
  dw SEIn ,ADCay,?N1n ,RRAay,SKWn3,ADCax,RORax,RRAax
  dw SKBn2,STAix,SKBn2,AXSix,STYz ,STAz ,STXz ,AXSz  ;80
  dw DEYn ,SKBn2,TXAn ,XMAb ,STYa ,STAa ,STXa ,AXSa
  dw BCCr ,STAiy,?92n ,?93iy,STYzx,STAzx,STXzy,AXSzy ;90
  dw TYAn ,STAay,TXSn ,XAAay,YASax,STAax,XASay,AXSay
  dw LDYb ,LDAix,LDXb ,LAXix,LDYz ,LDAz ,LDXz ,LAXz  ;A0
  dw TAYn ,LDAb ,TAXn ,?ABb ,LDYa ,LDAa ,LDXa ,LAXa
  dw BCSr ,LDAiy,?B2n ,LAXiy,LDYzx,LDAzx,LDXzy,LAXzy ;B0
  dw CLVn ,LDAay,TSXn ,LSAay,LDYax,LDAax,LDXay,LAXay
  dw CPYb ,CMPix,SKBn2,DCMix,CPYz ,CMPz ,DECz ,DCMz  ;C0
  dw INYn ,CMPb ,DEXn ,SAXb ,CPYa ,CMPa ,DECa ,DCMa
  dw BNEr ,CMPiy,CRPn ,DCMiy,SKBn2,CMPzx,DECzx,DCMzx ;D0
  dw CLDn ,CMPay,?N1n ,DCMay,SKWn3,CMPax,DECax,DCMax
  dw CPXb ,SBCix,SKBn2,INSix,CPXz ,SBCz ,INCz ,INSz  ;E0
  dw INXn ,SBCb ,NOPn ,?EBb ,CPXa ,SBCa ,INCa ,INSa
  dw BEQr ,SBCiy,CRPn ,INSiy,SKBn2,SBCzx,INCzx,INSzx ;F0
  dw SEDn ,SBCay,?N1n ,INSay,SKWn3,SBCax,INCax,INSax

code 4
AddDec proc near
  ifdef DEBUG
    TBEG REM
    if GERMAN
      INFO "Addition im Dezimalmodus"
    else
      INFO "Addition in Decimal mode"
    endif
    TEND
  endif
  mov BX,AX
  mov CX,AX
  and AX,0F0Fh
  and BX,0F0F0h
  ror DL,1
  adc AL,AH
  cmp AL,0Ah
  jb @@LowOK
  sub AL,0Ah
  or BX,0808h
@@LowOK:
  mov AH,00000000b
  add BL,BH
  jns @@NoSign
  mov AH,10000000b
@@NoSign:
  seto Overflow
  rcl DL,1
  adc CL,CH
  jne @@NoZero
  or AH,01000000b
@@NoZero:
  mov SignZero,AH
  test DL,00000001b
  jne @@AdjustHigh
  cmp BL,0A0h
  jb @@HighOK
@@AdjustHigh:
  sub BL,0A0h
  or DL,00000001b
@@HighOK:
  and AL,0Fh
  or AL,BL
  ret
AddDec endp

code 4
SubDec proc near
  ifdef DEBUG
    TBEG REM
    if GERMAN
      INFO "Subtraktion im Dezimalmodus"
    else
      INFO "Subtraction in Decimal mode"
    endif
    TEND
  endif
  mov BX,AX
  mov CX,AX
  and AX,0F0Fh
  and BX,0F0F0h
  not DL
  ror DL,1
  sbb AL,AH
  jnc LowOK
  sub AL,06h
  or BX,0800h
LowOK:
  mov AH,00000000b
  sub BL,BH
  jns NoSign
  mov AH,10000000b
NoSign:
  mov Overflow,0
  jno NoOverflow
  mov Overflow,1
NoOverflow:
  cmc
  rcl DL,1
  sbb CL,CH
  jne NoZero
  or AH,01000000b
NoZero:
  mov SignZero,AH
  test DL,00000001b
  jne HighOK
  sub BL,60h
  xor DL,DL
HighOK:
  and AL,0Fh
  and BL,0F0h
  or AL,BL
  ret
SubDec endp

;Addition Akkumulator mit bertrag
ACTION macro
  READ                                  ;Operand lesen
  test Flags,00001000b                  ;Auf Dezimalmodus prfen
  jne @@Dec
  shr DL,1                              ;CF holen
  adc Akku,DATA                         ;Addition ausfhren
  SETSZCV                               ;Vernderte Flags setzen
  jmp @@NoDec
align 4
@@Dec:
  mov AH,DATA
  mov AL,Akku
  call AddDec
  mov Akku,AL
@@NoDec:
  endm
CMD ADC,<2b,3z,4zx,4a,4ax,4ay,6ix,5iy>

;Und-Verknpfung Akkumulator
ACTION macro
  READ                                  ;Operand lesen
  and Akku,DATA                         ;Verknpfung ausfhren
  SETSZ                                 ;Vernderte Flags setzen
  endm
CMD AND,<2b,3z,4zx,4a,4ax,4ay,6ix,5iy>

;Schieben um eine Stelle nach links
ACTION macro
  READ                                  ;Alten Wert lesen
  shl DATA,1                            ;Verschiebung ausfhren
  SETSZC                                ;Vernderte Flags setzen
  WRITE                                 ;Neuen Wert zurckschreiben
  endm
CMD ASL,<2n,5z,6zx,6a,7ax>

;Sprung wenn Carry gelscht
ACTION macro
  test DL,00000001b                     ;CF prfen
  BRANCH clear                          ;Sprung falls Flag gelscht
  endm
CMD BCC,<2r>

;Sprung wenn Carry gesetzt
ACTION macro
  test DL,00000001b                     ;CF prfen
  BRANCH set                            ;Sprung falls Flag gesetzt
  endm
CMD BCS,<2r>

;Sprung wenn gleich
ACTION macro
  test SignZero,01000000b               ;ZF prfen
  BRANCH set                            ;Sprung falls Flag gesetzt
  endm
CMD BEQ,<2r>

;Bits testen
ACTION macro
  READ                                  ;Operand leden
  test Akku,DATA                        ;Test ausfhren
  lahf                                  ;ZF nach AH holen
  test DATA,01000000b                   ;Datenbit 6 nach VF holen
  setne Overflow
  and DATA,10000000b                    ;Datenbit 7 nach SF holen
  and AH,01000000b
  or AH,DATA
  mov SignZero,AH                       ;SF und ZF setzen
  endm
CMD BIT,<3z,4a>

;Sprung wenn negativ
ACTION macro
  test SignZero,10000000b               ;SF prfen
  BRANCH set                            ;Sprung falls Flag gesetzt
  endm
CMD BMI,<2r>

;Sprung wenn ungleich
ACTION macro
  test SignZero,01000000b               ;ZF prfen
  BRANCH clear                          ;Sprung falls Flag gelscht
  endm
CMD BNE,<2r>

;Sprung wenn positiv
ACTION macro
  test SignZero,10000000b               ;SF prfen
  BRANCH clear                          ;Sprung falls Flag gelscht
  endm
CMD BPL,<2r>

;Zum Debugger springen
ACTION macro
  inc SI                                ;Rcksprungadresse erhhen
  INTERRUPT 0FFFEh,1                    ;IRQ-Vektor aufrufen, BF gesetzt lassen
  or Flags,00000100b                    ;Weitere Interrupts untersagen
  endm
CMD BRK,<7n>

;Sprung wenn kein berlauf
ACTION macro
  test Overflow,00000001b               ;VF prfen
  BRANCH clear                          ;Sprung falls Flag gelscht
  endm
CMD BVC,<2r>

;Sprung wenn berlauf
ACTION macro
  test Overflow,00000001b               ;VF prfen
  BRANCH set                            ;Sprung falls Flag gesetzt
  endm
CMD BVS,<2r>

;CF lschen
ACTION macro
  xor DL,DL
  endm
CMD CLC,<2n>

;Dezimalmodus ausschalten
ACTION macro
  and Flags,11110111b
  endm
CMD CLD,<2n>

code 4
IMP ReadCIA113,near
RTSandCLI proc near
  mov BX,StackPtr                       ;Rcksprungadresse holen
  inc BL
  mov AL,GS:[BX]
  inc BL
  mov AH,GS:[BX]
  mov StackPtr,BX
  mov SI,AX                             ;Sprung ausfhren
  inc SI                                ;Der 6502 erhht den Wert um 1
  JUMP                                  ;Speicherbasis angleichen
  mov CX,offset CLIEntry                ;CLI bei $EDAB
  jmp ReadCIA113                        ;LDA $DC0D bei $ED9C
RTSandCLI endp

;Interrupts einschalten
ACTION macro
  local NoIRQ
CLIEntry label near
  and Flags,11111011b                   ;IF lschen
  test Events,evIRQ                     ;Liegt ein Interrupt an?
  je NoIRQ                              ;Nein, dann nichts tun
  mov AL,FS:[SI]                        ;Bei SEI, RTI und BRK verzgern
  cmp AL,78h
  je NoIRQ
  cmp AL,40h
  je NoIRQ
  cmp AL,00h
  je NoIRQ
  ifdef DEBUG
    TBEG INT
    test Events,evIRQCIA
    je @@NoIRQCIA
    if GERMAN
      INFO "IRQ von CIA 1 liegt an, Sprung zum IRQ-Vektor an Adresse FFFE"
    else
      INFO "IRQ from CIA 1 is pending, jumping to IRQ vector at FFFE"
    endif
  @@NoIRQCIA:
    test Events,evIRQVIC
    je @@NoIRQVIC
    if GERMAN
      INFO "IRQ vom VIC liegt an, Sprung zum IRQ-Vektor an Adresse FFFE"
    else
      INFO "IRQ from VIC-II is pending, jumping to IRQ vector at FFFE"
    endif
  @@NoIRQVIC:
    TEND
  endif
  INTERRUPT 0FFFEh,0                    ;IRQ-Vektor aufrufen
  or Flags,00000100b                    ;Weitere Interrupts untersagen
align 4
NoIRQ:
  endm
CMD CLI,<2n>

;VF lschen
ACTION macro
  mov Overflow,00000000b
  endm
CMD CLV,<2n>

;Akku mit Wert vergleichen
ACTION macro
  READ                                  ;Operand lesen
  cmp Akku,DATA                         ;Vergleich ausfhren
  cmc                                   ;Subtraktion 8086->6502
  SETSZC                                ;Vernderte Flags setzen
  endm
CMD CMP,<2b,3z,4zx,4a,4ax,4ay,6ix,5iy>

;X-Register mit Wert vergleichen
ACTION macro
  READ                                  ;Operand lesen
  cmp XReg,DATA                         ;Vergleich ausfhren
  cmc                                   ;Subtraktion 8086->6502
  SETSZC                                ;Vernderte Flags setzen
  endm
CMD CPX,<2b,3z,4a>

;X-Register mit Wert vergleichen
ACTION macro
  READ                                  ;Operand lesen
  cmp YReg,DATA                         ;Vergleich ausfhren
  cmc                                   ;Subtraktion 8086->6502
  SETSZC                                ;Vernderte Flags setzen
  endm
CMD CPY,<2b,3z,4a>

;Wert um 1 verringern
ACTION macro
  READ                                  ;Alten Wert lesen
  dec DATA                              ;Operation ausfhren
  SETSZ                                 ;Vernderte Flags setzen
  WRITE                                 ;Neuen Wert zurckschreiben
  endm
CMD DEC,<5z,6zx,6a,7ax>

;X-Register um 1 verringern
ACTION macro
  dec XReg                              ;Operation ausfhren
  SETSZ                                 ;Vernderte Flags setzen
  endm
CMD DEX,<2n>

;Y-Register um 1 verringern
ACTION macro
  dec YReg                              ;Operation ausfhren
  SETSZ                                 ;Vernderte Flags setzen
  endm
CMD DEY,<2n>

;Exclusiv-Oder-Verknpfung Akkumulator
ACTION macro
  READ                                  ;Operand lesen
  xor Akku,DATA                         ;Verknpfung ausfhren
  SETSZ                                 ;Vernderte Flags setzen
  endm
CMD EOR,<2b,3z,4zx,4a,4ax,4ay,6ix,5iy>

;Wert um 1 erhhen
ACTION macro
  READ                                  ;Alten Wert lesen
  inc DATA                              ;Operation ausfhren
  SETSZ                                 ;Vernderte Flags setzen
  WRITE                                 ;Neuen Wert zurckschreiben
  endm
CMD INC,<5z,6zx,6a,7ax>

;X-Register um 1 erhhen
ACTION macro
  inc XReg                              ;Operation ausfhren
  SETSZ                                 ;Vernderte Flags setzen
  endm
CMD INX,<2n>

;Y-Register um 1 erhhen
ACTION macro
  inc YReg                              ;Operation ausfhren
  SETSZ                                 ;Vernderte Flags setzen
  endm
CMD INY,<2n>

;Unbedingter Sprung
ACTION macro
  ifdef DEBUG
    mov dword ptr ES:[DI+26],'  >-'
  endif
  mov SI,BX                             ;Sprung ausfhren
  JUMP                                  ;Speicherbasis angleichen
  endm
CMD JMP,<3a,6i>

;Aufruf eines Unterprogramms
ACTION macro
  ifdef DEBUG
    mov dword ptr ES:[DI+26],'  >-'
  endif
  mov AX,SI                             ;Alte Adresse holen
  mov SI,BX                             ;Sprung ausfhren
  dec AX                                ;Der 6502 verringert den Wert um 1
  mov BX,StackPtr                       ;Alte Adresse auf Stapel
  mov GS:[BX],AH
  dec BL
  mov GS:[BX],AL
  dec BL
  mov StackPtr,BX
  JUMP                                  ;Speicherbasis angleichen
  endm
CMD JSR,<6a>

;Akkumulator laden
ACTION macro
  READ                                  ;Operand lesen
  mov Akku,DATA                         ;und bertragen
  and DATA,DATA                         ;SF und ZF holen
  SETSZ                                 ;und setzen
  endm
CMD LDA,<2b,3z,4zx,4a,4ax,4ay,6ix,5iy>

;X-Register laden
ACTION macro
  READ                                  ;Operand lesen
  mov XReg,DATA                         ;und bertragen
  and DATA,DATA                         ;SF und ZF holen
  SETSZ                                 ;und setzen
  endm
CMD LDX,<2b,3z,4zy,4a,4ay>

;Y-Register laden
ACTION macro
  READ                                  ;Operand lesen
  mov YReg,DATA                         ;und bertragen
  and DATA,DATA                         ;SF und ZF holen
  SETSZ                                 ;und setzen
  endm
CMD LDY,<2b,3z,4zx,4a,4ax>

;Schieben um 1 Stelle nach rechts
ACTION macro
  READ                                  ;Alten Wert lesen
  shr DATA,1                            ;Verschiebung ausfhren
  SETSZC                                ;Vernderte Flags setzen
  WRITE                                 ;Neuen Wert zurckschreiben
  endm
CMD LSR,<2n,5z,6zx,6a,7ax>

;Leerer Befehl
ACTION macro
  endm
CMD NOP,<2n>

;Oder-Verknpfung Akkumulator
ACTION macro
  READ                                  ;Operand lesen
  or Akku,DATA                          ;Verknpfung ausfhren
  SETSZ                                 ;Vernderte Flags setzen
  endm
CMD ORA,<2b,3z,4zx,4a,4ax,4ay,6ix,5iy>

;Push Akkumulator
ACTION macro
  mov AL,Akku                           ;Akkumulator holen
  mov BX,StackPtr                       ;und auf Stapel
  mov GS:[BX],AL
  dec BL
  mov StackPtr,BX
  endm
CMD PHA,<3n>

;Push Flags
ACTION macro
  GETFLAGS                              ;Flags nach AL holen
  mov BX,StackPtr                       ;und auf Stapel
  mov GS:[BX],AL
  dec BL
  mov StackPtr,BX
  endm
CMD PHP,<3n>

;Pull Akkumulator
ACTION macro
  mov BX,StackPtr                       ;Wert von Stapel holen
  inc BL
  mov AL,GS:[BX]
  mov StackPtr,BX
  mov Akku,AL                           ;Akkumulator setzen
  and AL,AL                             ;SF und ZF holen
  SETSZ                                 ;und setzen
  endm
CMD PLA,<4n>

;Pull Flags
ACTION macro
  local NoIRQ
  mov BX,StackPtr                       ;Wert vom Stapel holen
  inc BL
  mov AL,GS:[BX]
  mov StackPtr,BX
  SETFLAGS                              ;Flags setzen
  test Events,evIRQ                     ;Liegt ein Interrupt an?
  je NoIRQ                              ;Nein
  test Flags,00000100b                  ;Sind Interrupts zugelassen?
  jne NoIRQ                             ;Nein
  ifdef DEBUG
    TBEG INT
    test Events,evIRQCIA
    je @@NoIRQCIA
    if GERMAN
      INFO "IRQ von CIA 1 liegt an, Sprung zum IRQ-Vektor an Adresse FFFE"
    else
      INFO "IRQ from CIA 1 is pending, jumping to IRQ vector at FFFE"
    endif
  @@NoIRQCIA:
    test Events,evIRQVIC
    je @@NoIRQVIC
    if GERMAN
      INFO "IRQ vom VIC liegt an, Sprung zum IRQ-Vektor an Adresse FFFE"
    else
      INFO "IRQ from VIC-II is pending, jumping to IRQ vector at FFFE"
    endif
  @@NoIRQVIC:
    TEND
  endif
  INTERRUPT 0FFFEh,0                    ;IRQ-Vektor aufrufen
  or Flags,00000100b                    ;Weitere Interrupts untersagen
align 4
NoIRQ:
  endm
CMD PLP,<4n>

;Der 8086 verndert bei RCL/RCR das SF und ZF nicht, der 6502 hingegen schon.
;Deshalb mu der Emulator das Ergebnis kennen, damit die Flags mit AND
;gesetzt werden knnen. Aus diesem Grund wird der Akku in das AL-Register
;geholt, um ein paar Taktzyklen zu sparen

;Akkumulator ber CF um eine Stelle nach links schieben
ACTION macro
  mov AL,Akku                           ;Alten Inhalt lesen
  shr DL,1                              ;CF holen
  rcl AL,1                              ;Verschiebung ausfhren
  setc DL                               ;CF setzen
  and AL,AL                             ;SF und ZF holen
  SETSZ                                 ;und setzen
  mov Akku,AL                           ;Neuen Inhalt zurckschreiben
  endm
CMD ROL,<2n>

;Schieben ber CF um eine Stelle nach links
ACTION macro
  READ                                  ;Alten Wert lesen
  shr DL,1                              ;CF holen
  rcl DATA,1                            ;Verschiebung ausfhren
  setc DL                               ;CF setzen
  and DATA,DATA                         ;SF und ZF holen
  SETSZ                                 ;und setzen
  WRITE                                 ;Neuen Wert zurckschreiben
  endm
CMD ROL,<5z,6zx,6a,7ax>

;Akkumulator ber CF um eine Stelle nach rechts schieben
ACTION macro
  mov AL,Akku                           ;Alten Inhalt lesen
  shr DL,1                              ;CF holen
  rcr AL,1                              ;Verschiebung ausfhren
  setc DL                               ;CF setzen
  and AL,AL                             ;SF und ZF holen
  SETSZ                                 ;und setzen
  mov Akku,AL                           ;Neuen Inhalt zurckschreiben
  endm
CMD ROR,<2n>

;Schieben ber CF um eine Stelle nach rechts
ACTION macro
  READ                                  ;Alten Wert lesen
  shr DL,1                              ;CF holen
  rcr DATA,1                            ;Verschiebung ausfhren
  setc DL                               ;CF setzen
  and DATA,DATA                         ;SF und ZF holen
  SETSZ                                 ;und setzen
  WRITE                                 ;Neuen Wert zurckschreiben
  endm
CMD ROR,<5z,6zx,6a,7ax>

;Rscksprung vom Interrupt
ACTION macro
  local NoIRQ
  mov BX,StackPtr                       ;Flags vom Stack holen
  inc BL
  mov AL,GS:[BX]
  inc BL
  SETFLAGS                              ;Flags setzen
  mov AL,GS:[BX]                        ;Rcksprungadresse vom Stack holen
  inc BL
  mov AH,GS:[BX]
  mov StackPtr,BX
  mov SI,AX                             ;Sprung ausfhren
  JUMP                                  ;Speicherbasis angleichen
  test Events,evIRQ                     ;Liegt ein Interrupt an?
  je NoIRQ                              ;Nein
  test Flags,00000100b                  ;Sind Interrupts zugelassen?
  jne NoIRQ                             ;Nein
  ifdef DEBUG
    TBEG INT
    test Events,evIRQCIA
    je @@NoIRQCIA
    if GERMAN
      INFO "IRQ von CIA 1 liegt an, Sprung zum IRQ-Vektor an Adresse FFFE"
    else
      INFO "IRQ from CIA 1 is pending, jumping to IRQ vector at FFFE"
    endif
  @@NoIRQCIA:
    test Events,evIRQVIC
    je @@NoIRQVIC
    if GERMAN
      INFO "IRQ vom VIC liegt an, Sprung zum IRQ-Vektor an Adresse FFFE"
    else
      INFO "IRQ from VIC-II is pending, jumping to IRQ vector at FFFE"
    endif
  @@NoIRQVIC:
    TEND
  endif
  INTERRUPT 0FFFEh,0                    ;IRQ-Vektor aufrufen
  or Flags,00000100b                    ;Weitere Interrupts untersagen
align 4
NoIRQ:
  endm
CMD RTI,<6n>

;Rcksprung aus Unterprogramm
ACTION macro
RTS label near
  mov BX,StackPtr                       ;Rcksprungadresse holen
  inc BL
  mov AL,GS:[BX]
  inc BL
  mov AH,GS:[BX]
  mov StackPtr,BX
  mov SI,AX                             ;Sprung ausfhren
  inc SI                                ;Der 6502 erhht den Wert um 1
  JUMP                                  ;Speicherbasis angleichen
  endm
CMD RTS,<6n>

;Subtraktion Akkumulator mit Carry
ACTION macro
  READ                                  ;Operand lesen
  test Flags,00001000b                  ;Auf Dezimalmodus prfen
  jne @@Dec
  shr DL,1                              ;CF holen
  cmc                                   ;Subtraktion 6502->8086
  sbb Akku,DATA                         ;Operation ausfhren
  cmc                                   ;Subtraktion 8086->6502
  SETSZCV                               ;Vernderte Flags setzen
  jmp @@NoDec
align 4
@@Dec:
  mov AH,DATA
  mov AL,Akku
  call SubDec
  mov Akku,AL
@@NoDec:
  endm
CMD SBC,<2b,3z,4zx,4a,4ax,4ay,6ix,5iy>

;CF setzen
ACTION macro
  mov DL,0000001b
  endm
CMD SEC,<2n>

;Dezimalmodus einschalten
ACTION macro
  or Flags,00001000b
  ifdef DEBUG
    TBEG REM
      if GERMAN
        WARN "Dezimal-Flag wurde gesetzt!"
      else
        WARN "Decimal was set!"
      endif
    TEND
  endif
  endm
CMD SED,<2n>

;Interrupts sperren
ACTION macro
  or Flags,00000100b
  endm
CMD SEI,<2n>

;Akkumulator schreiben
ACTION macro
  mov DATA,Akku                         ;Akku auslesen
  WRITE                                 ;und Wert schreiben
  endm
CMD STA,<3z,4zx,4a,5ax,5ay,6ix,5iy>

;X-Register schreiben
ACTION macro
  mov DATA,XReg                         ;Register auslesen
  WRITE                                 ;und Wert schreiben
  endm
CMD STX,<3z,4zy,4a,5ay>

;Y-Register schreiben
ACTION macro
  mov DATA,YReg                         ;Register auslesen
  WRITE                                 ;und Wert schreiben
  endm
CMD STY,<3z,4zx,4a,5ax>

;Akkumulator -> X-Register
ACTION macro
  mov AL,Akku                           ;bertragung vornehmen
  mov XReg,AL
  and AL,AL                             ;SF und ZF holen
  SETSZ                                 ;und setzen
  endm
CMD TAX,<2n>

;Akkumulator -> Y-Register
ACTION macro
  mov AL,Akku                           ;bertragung vornehmen
  mov YReg,AL
  and AL,AL                             ;SF und ZF holen
  SETSZ                                 ;und setzen
  endm
CMD TAY,<2n>

;SP -> X-Register
ACTION macro
  mov AL,byte ptr StackPtr              ;bertragung vornehmen
  mov XReg,AL
  and AL,AL                             ;SF und ZF holen
  SETSZ                                 ;und setzen
  endm
CMD TSX,<2n>

;X-Register -> Akkumulator
ACTION macro
  mov AL,XReg                           ;bertragung vornehmen
  mov Akku,AL
  and AL,AL                             ;SF und ZF holen
  SETSZ                                 ;und setzen
  endm
CMD TXA,<2n>

;X-Register -> SP
ACTION macro
  mov AL,XReg                           ;bertragung vornehmen
  mov byte ptr StackPtr,AL
  endm                                  ;Keine Flags setzen!
CMD TXS,<2n>

;Y-Register -> Akkumulator
ACTION macro
  mov AL,YReg                           ;bertragung vornehmen
  mov Akku,AL
  and AL,AL                             ;SF und ZF holen
  SETSZ                                 ;und setzen
  endm
CMD TYA,<2n>

;CPU abgestrzt, Emulation beenden
code 1
Crash proc near
  dec SI
  if GERMAN
    ERROR "Der Befehl %02X an Adresse %04X hat einen Absturz der 6510-CPU bewirkt!",<AX,SI>
  else
    ERROR "The instruction %02X at address %04X has crashed the 6510 CPU!",<AX,SI>
  endif
Crash endp

;Undokumentierter Befehl
ACTION macro
  jmp Crash
  endm
CMD CRP,<0n>

;Undokumentierter Befehl 02: Absturz oder Listen
code 4
?02n proc near
  ifdef DEBUG
    mov dword ptr ES:[DI+12],'PRC '
  endif
  cmp SI,0ED0Ch+1
  je @@Listen
  cmp SI,0ED0Eh+1
  jne Crash
  test Akku,40h
  jne IsTalk
@@Listen:
  mov AL,Akku
  or AL,20h
  mov GS:[95h],AL
  push DX
  push FS
  push GS
  and AX,000Fh
  push AX
  call Listen
  add SP,2
  pop GS
  pop FS
  pop DX
  or GS:[90h],AH
  mov Akku,0
  xor DL,DL
  jmp RTSandCLI
?02n endp

;Undokumentierter Befehl 12: Absturz oder Talk
code 4
?12n proc near
  ifdef DEBUG
    mov dword ptr ES:[DI+12],'PRC '
  endif
  cmp SI,0ED09h+1
  jne Crash
IsTalk label near
  mov AL,Akku
  or AL,40h
  mov GS:[95h],AL
  push DX
  push FS
  push GS
  and AX,000Fh
  push AX
  call Talk
  add SP,2
  pop GS
  pop FS
  pop DX
  or GS:[90h],AH
  mov Akku,0
  xor DL,DL
  jmp RTSandCLI
?12n endp

;Undokumentierter Befehl 22: Absturz oder Sekundradresse Listen
code 4
?22n proc near
  ifdef DEBUG
    mov dword ptr ES:[DI+12],'PRC '
  endif
  cmp SI,0EDB9h+1
  jne Crash
  movzx AX,Akku
  mov GS:[95h],AL
  push DX
  push FS
  push GS
  push AX
  call ChListen
  add SP,2
  pop GS
  pop FS
  pop DX
  or GS:[90h],AH
  xor DL,DL
  jmp RTSandCLI
?22n endp

;Undokumentierter Befehl 32: Absturz oder Sekundradresse Talk
code 4
?32n proc near
  ifdef DEBUG
    mov dword ptr ES:[DI+12],'PRC '
  endif
  cmp SI,0EDC7h+1
  jne Crash
  mov AL,Akku
  mov GS:[95h],AL
  push DX
  push FS
  push GS
  push AX
  call ChTalk
  add SP,2
  pop GS
  pop FS
  pop DX
  or GS:[90h],AH
  xor DL,DL
  jmp RTSandCLI
?32n endp

;Undokumentierter Befehl 42: Absturz oder Byte auf IEC-Bus ausgeben
code 4
?42n proc near
  ifdef DEBUG
    mov dword ptr ES:[DI+12],'PRC '
  endif
  cmp SI,0EDDDh+1
  jne Crash
  movzx AX,Akku
  mov GS:[95h],AL
  push DX
  push FS
  push GS
  push AX
  call IECOut
  add SP,2
  pop GS
  pop FS
  pop DX
  or GS:[90h],AH
  xor DL,DL
  jmp RTSandCLI
?42n endp

;Undokumentierter Befehl 52: Absturz oder Byte vom IEC-Bus einlesen
code 4
?52n proc near
  ifdef DEBUG
    mov dword ptr ES:[DI+12],'PRC '
  endif
  cmp SI,0EE13h+1
  jne Crash
  push DX
  push FS
  push GS
  call IECIn
  pop GS
  pop FS
  pop DX
  mov GS:[0A4h],AL
  mov Akku,AL
  or GS:[90h],AH
  and AL,AL                             ;Die Flags mssen gesetzt werden!
  SETSZ
  xor DL,DL
  jmp RTSandCLI
?52n endp

;Undokumentierter Befehl 62: Absturz oder Unlisten
code 4
?62n proc near
  ifdef DEBUG
    mov dword ptr ES:[DI+12],'PRC '
  endif
  cmp SI,0EDFEh+1
  jne Crash
  mov byte ptr GS:[95h],3Fh
  push DX
  push FS
  push GS
  call Unlisten
  pop GS
  pop FS
  pop DX
  or GS:[90h],AH
  xor DL,DL
  jmp RTSandCLI
?62n endp

;Undokumentierter Befehl 72: Absturz oder Untalk
code 4
?72n proc near
  ifdef DEBUG
    mov dword ptr ES:[DI+12],'PRC '
  endif
  cmp SI,0EDEFh+1
  jne Crash
  mov byte ptr GS:[95h],5Fh
  push DX
  push FS
  push GS
  call Untalk
  pop GS
  pop FS
  pop DX
  or GS:[90h],AH
  xor DL,DL
  jmp RTSandCLI
?72n endp

;Undokumentierter Befehl 92: Absturz oder Zeichen von Tastatur lesen
code 4
?92n proc near
  ifdef DEBUG
    mov dword ptr ES:[DI+12],'PRC '
  endif
  cmp SI,0EA87h+1
  jne Crash
  ifdef DEBUG
    TBEG REM
    if GERMAN
      INFO "Pseudobefehl: Tastatur ber Int 16h lesen"
    else
      INFO "Pseudo instruction: Read the keyboard with Int 16h"
    endif
    TEND
  endif
  call ReadKey
  jmp RTS
?92n endp

;Undokumentierter Befehl B2: Absturz oder Autotype
data?
extern c gpcAutotype:word
code 4
?B2n proc near
  ifdef DEBUG
    mov dword ptr ES:[DI+12],'PRC '
  endif
  cmp SI,0E5CDh+1
  jne Crash
  ifdef DEBUG
    TBEG REM
    if GERMAN
      INFO "Pseudobefehl: Zeichen aus Autotype-String in Tastaturpuffer holen"
    else
      INFO "Pseudo instruction: Get next character from the autotype string"
    endif
    TEND
  endif
  jmp @@Next

@@Copy:
  mov DI,gpcAutotype
  mov AL,[DI]
  and AL,AL
  je @@Finished

  mov BL,GS:[00C6h]
  xor BH,BH
  mov GS:[0277h+BX],AL

  inc DI
  mov gpcAutotype,DI
  inc byte ptr GS:[00C6h]
@@Next:
  cmp byte ptr GS:[00C6h],10
  jb @@Copy

  jmp CommandTable[0A5h*2]

@@Finished:
  dec SI
  mov byte ptr FS:[SI],0A5h
  NEXTCMD                               ;Nchsten Befehl ausfhren

?B2n endp


;Keine Operation, 1 Befehlsbyte
ACTION macro
  ifdef DEBUG
    mov dword ptr ES:[DI+12],'PON '
    TBEG REM
    if GERMAN
      INFO "Undokumentierter Befehl: Entspricht NOP"
    else
      INFO "Undocumented instruction: Corresponds to NOP"
    endif
    TEND
  endif
  endm
CMD ?N1,<2n>

;Keine Operation, 2 Befehlsbyte
ACTION macro
  ifdef DEBUG
    TBEG REM
    if GERMAN
      INFO "Undokumentierter Befehl: NOP und Byte berlesen"
    else
      INFO "Undocumented instruction: NOP and skip a byte"
    endif
    TEND
  endif
  endm
CMD SKB,<3n2>

;Keine Operation, 3 Befehlsbyte
ACTION macro
  ifdef DEBUG
    TBEG REM
    if GERMAN
      INFO "Undokumentierter Befehl: NOP und Wort berlesen"
    else
      INFO "Undocumented instruction: NOP and skip a word"
    endif
    TEND
  endif
  endm
CMD SKW,<4n3>

;Schieben um eine Stelle nach links und Oder-Verknpfung
ACTION macro
  ifdef DEBUG
    TBEG REM
    if GERMAN
      INFO "Undokumentierter Befehl: ASL und ORA"
    else
      INFO "Undocumented instruction: ASL and ORA"
    endif
    TEND
  endif
  READ                                  ;Alten Wert lesen
  shl DATA,1                            ;Verschiebung ausfhren
  setc DL                               ;CF setzen
  or Akku,DATA                          ;Oder-Verknpfung ausfhren
  SETSZ                                 ;Vernderte Flags setzen
  WRITE                                 ;Neuen Wert zurckschreiben
  endm
CMD ASO,<5z,6zx,6a,7ax,7ay,8ix,7iy>

;Schieben um eine Stelle nach links und Und-Verknpfung
ACTION macro
  ifdef DEBUG
    TBEG REM
    if GERMAN
      INFO "Undokumentierter Befehl: ROL und AND"
    else
      INFO "Undocumented instruction: ROL and AND"
    endif
    TEND
  endif
  READ                                  ;Alten Wert lesen
  shr DL,1                              ;CF holen
  rcl DATA,1                            ;Verschiebung ausfhren
  setc DL                               ;CF setzen
  and Akku,DATA                         ;Und-Verknpfung ausfhren
  SETSZ                                 ;Vernderte Flags setzen
  WRITE                                 ;Neuen Wert zurckschreiben
  endm
CMD RLA,<5z,6zx,6a,7ax,7ay,8ix,7iy>

;Schieben um eine Stelle nach rechts und Xor-Verknpfung
ACTION macro
  ifdef DEBUG
    TBEG REM
    if GERMAN
      INFO "Undokumentierter Befehl: LSR und EOR"
    else
      INFO "Undocumented instruction: LSR and EOR"
    endif
    TEND
  endif
  READ                                  ;Alten Wert lesen
  shr DATA,1                            ;Verschiebung ausfhren
  setc DL                               ;CF setzen
  xor Akku,DATA                         ;Xor-Verknpfung ausfhren
  SETSZ                                 ;Vernderte Flags setzen
  WRITE                                 ;Neuen Wert zurckschreiben
  endm
CMD LSE,<5z,6zx,6a,7ax,7ay,8ix,7iy>

;Schieben um eine Stelle nach rechts und Addition
ACTION macro
  ifdef DEBUG
    TBEG REM
    if GERMAN
      INFO "Undokumentierter Befehl: ROR und ADC"
    else
      INFO "Undocumented instruction: ROR and ADC"
    endif
    TEND
  endif
  READ                                  ;Alten Wert lesen
  test Flags,00001000b                  ;Auf Dezimalmodus prfen
  jne @@Dec
  shr DL,1                              ;CF holen
  rcr DATA,1                            ;Verschiebung ausfhren
  adc Akku,DATA                         ;Addition ausfhren
  SETSZCV                               ;Vernderte Flags setzen
  jmp @@NoDec
align 4
@@Dec:
  shr DL,1                              ;CF holen
  rcr DATA,1                            ;Verschiebung ausfhren
  rcl DL,1                              ;CF setzen
  push AX
  push BX
  mov AH,DATA
  mov AL,Akku
  call AddDec
  mov Akku,AL
  pop BX
  pop AX
@@NoDec:
  WRITE                                 ;Neuen Wert zurckschreiben
  endm
CMD RRA,<5z,6zx,6a,7ax,7ay,8ix,7iy>

;Verringern um eins und mit Akkumulator vergleichen
ACTION macro
  ifdef DEBUG
    TBEG REM
    if GERMAN
      INFO "Undokumentierter Befehl: DEC und CMP"
    else
      INFO "Undocumented instruction: DEC and CMP"
    endif
    TEND
  endif
  READ                                  ;Alten Wert lesen
  dec DATA                              ;Operation ausfhren
  cmp Akku,DATA                         ;Vergleich ausfhren
  cmc                                   ;Subtraktion 8086->6502
  SETSZC                                ;Vernderte Flags setzen
  WRITE                                 ;Neuen Wert zurckschreiben
  endm
CMD DCM,<5z,6zx,6a,7ax,7ay,8ix,7iy>

;Erhhen um eins und Subtraktion ausfhren
ACTION macro
  ifdef DEBUG
    TBEG REM
    if GERMAN
      INFO "Undokumentierter Befehl: INC und SBC"
    else
      INFO "Undocumented instruction: INC and SBC"
    endif
    TEND
  endif
  READ                                  ;Alten Wert lesen
  inc DATA                              ;Operation ausfhren
  WRITE                                 ;Neuen Wert zurckschreiben
  test Flags,00001000b                  ;Auf Dezimalmodus prfen
  jne @@Dec
  shr DL,1                              ;CF holen
  cmc                                   ;Subtraktion 6502->8086
  sbb Akku,DATA                         ;Operation ausfhren
  cmc                                   ;Subtraktion 8086->6502
  SETSZCV                               ;Vernderte Flags setzen
  jmp @@NoDec
align 4
@@Dec:
  mov AH,DATA
  mov AL,Akku
  push BX
  call SubDec
  pop BX
  mov Akku,AL
@@NoDec:
  endm
CMD INS,<5z,6zx,6a,7ax,7ay,8ix,7iy>

;Akkumulator und X-Register laden
ACTION macro
  ifdef DEBUG
    TBEG REM
    if GERMAN
      INFO "Undokumentierter Befehl: Ldt Akku und XReg gleichzeitig"
    else
      INFO "Undocumented instruction: Loads accu and X register simultaneously"
    endif
    TEND
  endif
  READ                                  ;Operand lesen
  mov Akku,DATA                         ;und bertragen
  mov XReg,DATA
  and DATA,DATA                         ;SF und ZF holen
  SETSZ                                 ;und setzen
  endm
CMD LAX,<3z,4zy,4a,4ay,6ix,5iy>

;Und-Verknpfung zwischen Akku und X-Register speichern
ACTION macro
  ifdef DEBUG
    TBEG REM
    if GERMAN
      INFO "Undokumentierter Befehl: Speichert Akku & XReg"
    else
      INFO "Undocumented instruction: Stores Akku & XReg"
    endif
    TEND
  endif
  mov DATA,Akku                         ;Akku lesen
  and DATA,XReg                         ;Mit X-Register verknpfen
  WRITE                                 ;Ergebnis schreiben
  endm
CMD AXS,<3z,4zy,4a,6ix>

;Und-Verknpfung und N nach C
ACTION macro
  ifdef DEBUG
    TBEG REM
    if GERMAN
      INFO "Undokumentierter Befehl: AND, CF = NF"
    else
      INFO "Undocumented instruction: AND, CF = NF"
    endif
    TEND
  endif
  READ                                  ;Operand lesen
  and Akku,DATA                         ;Verknpfung ausfhren
  SETSZ                                 ;Vernderte Flags setzen
  shr AH,7                              ;NF nach CF kopieren
  mov DL,AH
  endm
CMD ANC,<2b>

;Und-Verknpfung und Akku nach rechts schieben
ACTION macro
  ifdef DEBUG
    TBEG REM
    if GERMAN
      INFO "Undokumentierter Befehl: AND und LSR"
    else
      INFO "Undocumented instruction: AND and LSR"
    endif
    TEND
  endif
  READ                                  ;Operand lesen
  and Akku,DATA                         ;Verknpfung ausfhren
  shr Akku,1                            ;Verschiebung ausfhren
  SETSZC                                ;Vernderte Flags setzen
  endm
CMD ALR,<2b>

;Und-Verknpfung, Akku nach rechts rollen, CF = Bit 0, VF = Bit 5 XOR Bit 6
ACTION macro
  ifdef DEBUG
    TBEG REM
    if GERMAN
      INFO "Undokumentierter Befehl: AND und ROR, VF = Bit 6, CF = Bit 5 ^ Bit 6"
    else
      INFO "Undocumented instruction: AND and ROR, VF = bit 6, CF = bit 5 ^ bit 6"
    endif
    TEND
  endif
  READ                                  ;Operand lesen
  and AL,Akku                           ;Verknpfung ausfhren
  mov AH,AL                             ;fr Dezimalkorrektur
  mov DI,AX
  shr DL,1                              ;CF holen
  rcr AL,1                              ;Verschiebung ausfhren
  and AL,AL                             ;SF und ZF holen
  SETSZ                                 ;und setzen
  mov Akku,AL                           ;Neuen Inhalt zurckschreiben
  shl AX,2                              ;CF = Bit 6
  mov DL,AH
  shr AL,7                              ;VF = Bit 5 ^ Bit 6
  xor AL,AH
  mov Overflow,AL
  test Flags,00001000b                  ;Auf Dezimalmodus prfen
  je @@NoDec
  test DI,0000000000000001b
  mov AX,DI                             ;BCD-Fixup fr Low Nibble
  and AX,0000000100001111b
  add AL,AH
  cmp AL,06h
  jb @@NoIncLow
  mov AL,Akku
  mov AH,AL
  add AL,06h
  and AX,1111000000001111b
  or AL,AH
  mov Akku,AL
@@NoIncLow:
  mov AX,DI                             ;BCD-Fixup fr High Nibble
  and AX,0001000011110000b
  add AL,AH
  jc @@IncHigh
  mov DL,0
  cmp AL,60h
  jb @@NoDec
@@IncHigh:
  mov DL,1
  mov AL,Akku
  mov AH,AL
  add AL,60h
  and AX,0000111111110000b
  or AL,AH
  mov Akku,AL
@@NoDec:
  endm
CMD ARR,<2b>

;Undokumentierter Befehl 8B
ACTION macro
  ifdef DEBUG
    TBEG REM
    if GERMAN
      INFO "Undokumentierter Befehl: Akku = XReg & Byte"
    else
      INFO "Undocumented instruction: Accu = XReg & byte"
    endif
    TEND
  endif
  READ                                  ;Operand lesen
  and DATA,XReg                         ;Verknpfung ausfhren
  mov Akku,DATA
  SETSZ                                 ;Vernderte Flags setzen
  endm
CMD XMA,<2b>

;Undokumentierter Befehl 93
ACTION macro
  ifdef DEBUG
    mov dword ptr ES:[DI+12],'SXA '
    TBEG REM
    if GERMAN
      INFO "Undokumentierter Befehl: Speichert Byte[Lo + 1] + 1 & Akku & XReg"
    else
      INFO "Undocumented instruction: Stores byte[Lo + 1] + 1 & accu & XReg"
    endif
    TEND
  endif
  push BX                               ;Schreibadresse sichern
  mov BL,FS:[SI-1]                      ;Leseadresse nach BX holen
  inc BX                                ;1 addieren
  xor BH,BH                             ;Auf Zeropage beschrnken
  mov AL,GS:[BX]                        ;Wert holen
  inc AX
  and AL,Akku                           ;Und-Verknpfungen ausfhren
  and AL,XReg
  pop BX                                ;Schreibadresse zurckholen
  WRITE
  endm
CMD ?93,<5iy>

;Undokumentierter Befehl 9B
ACTION macro
  ifdef DEBUG
    TBEG REM
    if GERMAN
      INFO "Undokumentierter Befehl: Byte[HiLo + Y] = Hi + 1 & (SPtr = Akku & XReg)"
    else
      INFO "Undocumented instruction: Byte[HiLo + Y] = Hi + 1 & (SPtr = Accu & XReg)"
    endif
    TEND
  endif
  mov AL,Akku                           ;Erste Und-Verknpfung durchfhren
  and AL,XReg
  mov byte ptr StackPtr,AL              ;Stapelzeiger setzen
  mov AH,FS:[SI-1]                      ;Wert holen
  inc AH                                ;Erhhen
  and AL,AH                             ;Zweite Und-Verknpfung durchfhren
  WRITE
  endm
CMD XAA,<5ay>

;Undokumentierter Befehl 9C
ACTION macro
  ifdef DEBUG
    TBEG REM
    if GERMAN
      INFO "Undokumentierter Befehl: Byte[HiLo + X] = Hi + 1 & YReg"
    else
      INFO "Undocumented instruction: Byte[HiLo + X] = Hi + 1 & YReg"
    endif
    TEND
  endif
  mov AL,FS:[SI-1]                      ;Wert holen
  inc AX                                ;Erhhen
  and AL,YReg                           ;Und-Verknpfung durchfhren
  WRITE
  endm
CMD YAS,<5ax>

;Undokumentierter Befehl 9E
ACTION macro
  ifdef DEBUG
    TBEG REM
    if GERMAN
      INFO "Undokumentierter Befehl: Byte[HiLo + Y] = Hi + 1 & XReg"
    else
      INFO "Undocumented instruction: Byte[HiLo + Y] = Hi + 1 & XReg"
    endif
    TEND
  endif
  mov AL,FS:[SI-1]                      ;Wert holen
  inc AX                                ;Erhhen
  and AL,XReg                           ;Und-Verknpfung durchfhren
  WRITE
  endm
CMD XAS,<5ay>

;Undokumentierter Befehl 9F
ACTION macro
  ifdef DEBUG
    TBEG REM
    if GERMAN
      INFO "Undokumentierter Befehl: Speichert HiByte + 1 & Akku & XReg"
    else
      INFO "Undocumented instruction: Stores HiByte + 1 & Accu & XReg"
    endif
    TEND
  endif
  mov AL,FS:[SI-1]                      ;Wert holen
  inc AX                                ;Erhhen
  and AL,Akku                           ;Und-Verknpfungen durchfhren
  and AL,XReg
  WRITE
  endm
CMD AXS,<5ay>

;Undokumentierter Befehl AB
ACTION macro
  ifdef DEBUG
    mov dword ptr ES:[DI+12],'??? '
    TBEG REM
    if GERMAN
      INFO "Undokumentierter Befehl: Akku = XReg = Akku & Byte & $11 | Byte & $EE"
    else
      INFO "Undocumented instruction: Akku = XReg = Akku & byte & $11 | byte & $EE"
    endif
    TEND
  endif
  READ                                  ;Wert nach AL holen
  mov AH,Akku                           ;Verknpfung durchfhren
  and AH,AL
  and AX,0001000111101110b
  or AL,AH
  SETSZ                                 ;Vernderte Flags setzen
  mov Akku,AL                           ;Ergebnis zurckschreiben
  mov XReg,AL
  endm
CMD ?AB,<2b>

;Undokumentierter Befehl BB
ACTION macro
  ifdef DEBUG
    TBEG REM
    if GERMAN
      INFO "Undokumentierter Befehl: Akku = XReg = SPtr = Byte[HiLo + Y] & SPtr"
    else
      INFO "Undocumented instruction: Accu = XReg = SPtr = Byte[HiLo + Y] & SPtr"
    endif
    TEND
  endif
  READ                                  ;Wert holen
  and AL,byte ptr StackPtr              ;Und-Verknpfung durchfhren
  SETSZ                                 ;Vernderte Flags setzen
  mov Akku,AL                           ;Ergebnis zurckschreiben
  mov XReg,AL
  mov byte ptr StackPtr,AL
  endm
CMD LSA,<4ay>

;Undokumentierter Befehl CB
ACTION macro
  ifdef DEBUG
    TBEG REM
    if GERMAN
      INFO "Undokumentierter Befehl: XReg = (Akku & XReg) - Byte"
    else
      INFO "Undocumented instruction: XReg = (Accu & XReg) - Byte"
    endif
    TEND
  endif
  READ                                  ;Operand lesen
  mov AH,Akku                           ;Und-Verknpfung ausfhren
  and AH,XReg
  sub AH,DATA                           ;Operation ausfhren
  mov XReg,AH                           ;Ergebnis zurckschreiben
  cmc                                   ;Subtraktion 8086->6502
  SETSZC                                ;Vernderte Flags setzen
  endm
CMD SAX,<2b>

;Undokumentierter Befehl EB, entspricht SBC
ACTION macro
  ifdef DEBUG
    mov dword ptr ES:[DI+12],'CBS '
    TBEG REM
    if GERMAN
      INFO "Undokumentierter Befehl: Entspricht SBC"
    else
      INFO "Undocumented instruction: Corresponds to SBC"
    endif
    TEND
  endif
  READ                                  ;Operand lesen
  test Flags,00001000b                  ;Auf Dezimalmodus prfen
  jne @@Dec
  shr DL,1                              ;CF holen
  cmc                                   ;Subtraktion 6502->8086
  sbb Akku,DATA                         ;Operation ausfhren
  cmc                                   ;Subtraktion 8086->6502
  SETSZCV                               ;Vernderte Flags setzen
  jmp @@NoDec
align 4
@@Dec:
  mov AH,DATA
  mov AL,Akku
  call SubDec
  mov Akku,AL
@@NoDec:
  endm
CMD ?EB,<2b>

ende
