// ber paralleles Kabel angeschlossenen C64 verwalten

#include "pc64.h"

flag (__far *pfnLinkAbortProc)() = DefAbortProc;

// Standard-Abbruchprozedur
int iRetries;
flag DefAbortProc() {
  if (fCtrlBreak || iRetries == 0) {
    *(long*)L64State = -1;
    return TRUE;
  }
  iRetries--;
  return FALSE;
}
flag RunAbortProc() {
  return fCtrlBreak;
}

// Pufferinhalt auf Diskette schreiben
word CLink::Flush(word wChannel) {
  // Sind berhaupt Zeichen im Puffer vorhanden?
  int iLength = awOut[wChannel] - awIn[wChannel];
  awIn[wChannel] = awOut[wChannel] = 2;
  if (!apbBuffer[wChannel] || iLength <= 0) {
    return 0;
  }
  // Gertenummer und Kanal eintragen
  apbBuffer[wChannel][0] = (byte)(wDevice | 0x10);
  apbBuffer[wChannel][1] = (byte)(wChannel | 0x60);
  // Inhalt des Puffers schreiben
  while (L64Send(wLpt, apbBuffer[wChannel], iLength + 2) < 0) {
    if ((*pfnLinkAbortProc)()) {
      return ENODEVICE;
    }
  }
  return GetState(FALSE);
}

// Status und Fehlerkanal auslesen
flag fData;
word CLink::GetState(flag fSet) {
  // Ergebnis der letzten Aktion holen
  char sResult[65];
  while (L64Receive(wLpt, (byte*)sResult, 64) < 0) {
    if ((*pfnLinkAbortProc)()) {
      sError[0] = 0;
      return ENODEVICE;
    }
  }
  if (fData) {
    memcpy(sError, sResult + 1, 64);
  } else {
    // Abbruch beim ersten Steuerzeichen
    for (int i = 1; i < L64Result; i++) {
      if (sResult[i] < 32) {
        break;
      }
    }
    sResult[i] = 0;
    // Meldung umkopieren
    i = atoi(sResult + 1);
    if (i || fSet) {
      strcpy(sError, sResult + 1);
    }
    // Fehler ins Debug-Fenster schreiben
    switch (i) {
    case 1:
    case 73:
      INFO (DEV) "%s", sResult + 1);
    case 0:
      break;
    default:
      WARNING (DEV) "%s!", sResult + 1);
    }
  }
  // Status zurckliefern
  return (byte)sResult[0] * 256;
}

// Auf Funktionskompatibilitt prfen
static flag fCompTest;
flag CLink::CanHandle(char* pcEntry, flag fSetError) {
  if (fCompTest || !acType[0] || GetPrivateProfileInt(acType, pcEntry, 0, "DRIVES.INI")) {
    return TRUE;
  }
  if (fSetError) {
    strcpy(sError, "98,DRIVE NOT 1541 COMPLIANT,00,00");
  }
  return FALSE;
}

// Konstruktor
CLink::CLink() {
  // Alle Pufferzeiger lschen
  memset(apbBuffer, 0, sizeof apbBuffer);
  // Hchste Schnittstelle suchen
  for (wLpt = 3; wLpt; wLpt--) {
    if (*(word __far *)(0x00000408L + wLpt * 2)) {
      break;
    }
  }
  // Standardgert ist erste Floppy
  wDevice = 8;
}

// Destruktor
CLink::~CLink() {
  // Puffer freigeben
  for (word w = 0; w < 15; w++) {
    if (apbBuffer[w]) {
      free(apbBuffer[w]);
      apbBuffer[w] = NULL;
    }
  }
}

// Typ zurckgeben
word CLink::GetType() {
  return DEVLINK;
}

flag CLink::SetDir(char* pDir) {
  assert(pDir);
  assert(*pDir);
  if (*pDir < '1' || *pDir > '4') {
    return FALSE;
  }
  // Meldung auf bertragungsfehler setzen
  sError[0] = 0;
  // Kontakt aufnehmen
  wLpt = *pDir - '1';
  if (iRetries < 5) {
    iRetries = 5;
  }
  do {
    char c = (char)(0x20 + def.fScreenOn);
    if (L64Send(wLpt, &c, 1) == 1) {
      // Gertenummer zwischen 0 und 15?
      word wNewDevice = atoi(pDir + 1);
      if (wNewDevice >= 16) {
        return FALSE;
      }
      wDevice = wNewDevice;
      // Gert zurcksetzen
      if (Reset()) {
        return FALSE;
      }
      // Alles in Ordnung
      return TRUE;
    }
    L64State[wLpt] = 255;
  } while (!pfnLinkAbortProc());
  // Keine Verbindung
  return FALSE;
}

// Verzeichnis zurckgeben
void CLink::GetDir(char* pDir, word wMax) {
  assert(wMax > 4);
  wsprintf(pDir, "%d%d:", wLpt + 1, wDevice);
}

// Gert zurcksetzen
word CLink::Reset() {
  // Offene Dateien schlieen, Speicher fr Puffer freigeben
  for (word w = 0; w < 15; w++) {
    if (apbBuffer[w]) {
      Close(w);
    }
  }
  // Meldung auf bertragungsfehler setzen
  sError[0] = 0;
  // Abbruch zurcksetzen
  fCtrlBreak = FALSE;
  // Floppy zurcksetzen, Drucker auf Vorhandensein prfen
  byte abBuffer[10];
  abBuffer[0] = (byte)(wDevice | 0x10);
  abBuffer[1] = 0x6F;
  abBuffer[2] = 'U';
  abBuffer[3] = 'I';
  // Synchronisation erzwingen
  L64State[wLpt] = 255;
  // Maximal drei Versuche, dann Fehler
  w = 0;
  while (L64Send(wLpt, abBuffer, wDevice < 8 ? 2 : 4) < 0) {
    if (L64Result == -1 && ++w >= 3) {
      return ENODEVICE;
    }
  }
  // Status und Fehlerkanal holen
  iRetries = 50;
  word wReturn = GetState(TRUE);
  if (wReturn == 0 && wDevice >= 8) {
    assert(sError[2] == ',');
    strncpy(acType, sError + 3, 40);
    acType[39] = ',';
    *strchr(acType, ',') = 0;
    if (!GetPrivateProfileString(acType, NULL, "", (char*)abBuffer, 5, "DRIVES.INI")) {
      extern CLink* pLink;
      extern char* pcType;
      pLink = this;
      pcType = acType;
      fCompTest = TRUE;
      DialogBox(hInst, "CompTest", hwndFrame, (FARPROC)CompTestDlgProc);
      fCompTest = FALSE;
    }
  }
  return wReturn;
}

// Befehl absetzen
word CLink::Command(char* sCommand, word wLength) {
  // Meldung auf bertragungsfehler setzen
  sError[0] = 0;
  // Abbruch zurcksetzen
  fCtrlBreak = FALSE;
  // Lnge ermitteln, wenn sie fehlt
  if (!wLength) {
    wLength = strlen(sCommand);
  }
  if (wLength > 64) {
    wLength = 64;
  }
  // Befehl umkopieren
  byte abBuffer[66];
  abBuffer[0] = (byte)(0x10 | wDevice);
  abBuffer[1] = 0x6F;
  memcpy(abBuffer + 2, sCommand, wLength);
  // Befehl abschicken
  while (L64Send(wLpt, abBuffer, wLength + 2) < 0) {
    if ((*pfnLinkAbortProc)()) {
      return ENODEVICE;
    }
  }
  // Status zurckliefern
  return GetState(TRUE);
}

// Fehlermeldung holen
static char acLinkError[] = "99,LINK ERROR,00,00";
word CLink::GetError(char* pBuffer, word wMax) {
  // Letzte Meldung holen
  if (sError[0]) {
    strncpy(pBuffer, sError, wMax);
  } else {
    strncpy(pBuffer, acLinkError, wMax);
  }
  // Meldung fr nchstes Mal lschen
  strcpy(sError, "00, OK,00,00");
  // Fehlernummer zurckgeben
  return atoi(pBuffer);
}

// Status in Image schreiben
flag CLink::Save(int hFile) {
  if (_lwrite(hFile, sError, sizeof sError) != sizeof sError) {
    return FALSE;
  }
  return TRUE;
}

// Status aus Image lesen
flag CLink::Load(int hFile) {
  if (_lread(hFile, sError, sizeof sError) != sizeof sError) {
    return FALSE;
  }
  return TRUE;
}

// Emulation beginnen
word CLink::Start() {
  return 0;
}

// Emulation beenden
word CLink::Stop() {
  return 0;
}

word CLink::NoError() {
  strcpy(sError, "00, OK,00,00");
  return 0;
}

// Datei ffnen
word CLink::Open(word wChannel, char* pcName, word wLength) {
  assert(wChannel < 16);
  // Alten Puffer schreiben, falls vorhanden
  Flush(wChannel);
  // Meldung auf bertragungsfehler setzen
  sError[0] = 0;
  // Abbruch zurcksetzen
  fCtrlBreak = FALSE;
  // Kommando zum ffnen senden
  byte abBuffer[64];
  abBuffer[0] = (byte)(wDevice | 0x10);
  abBuffer[1] = (byte)(wChannel | 0xF0);
  memcpy(abBuffer + 2, pcName, wLength);
  while (L64Send(wLpt, abBuffer, wLength + 2) < 0) {
    if ((*pfnLinkAbortProc)()) {
      return ENODEVICE;
    }
  }
  word wReturn = GetState(TRUE);
  if (!wReturn && *(word*)sError == 0x3030) {
    char* pc = strchr(pcName, ',');
    if (pc && !strcmp(pc, ",D,W")) {
      if (!CanHandle("CreateDELFile", FALSE)) {
        return 0;
      }
      fData = TRUE;
      wReturn = Command("M-R\x2B\x02\x10", 6);
      fData = FALSE;
      if (wReturn) {
        return NoError();
      }
      assert(L64Result == 18);
      byte bChannel = (byte)(sError[wChannel] & 0x0F);
      if (bChannel >= 6) {
        return NoError();
      }
      fData = TRUE;
      wReturn = Command("M-R\x60\x02\x0C", 6);
      fData = FALSE;
      if (wReturn) {
        return NoError();
      }
      assert(L64Result == 14);
      byte bSector = (byte)sError[0 + bChannel];
      byte bIndex = (byte)sError[6 + bChannel];
      if ((bIndex & 31) != 2) {
        return NoError();
      }
      fData = TRUE;
      wReturn = Command("M-R\x85\xFE\x01", 6);
      fData = FALSE;
      if (wReturn) {
        return NoError();
      }
      assert(L64Result == 3);
      byte bTrack = (byte)sError[0];
      assert(apbBuffer[9] == NULL);
      wReturn = Open(9, "#", 1);
      if (wReturn || *(word*)sError != 0x3030) {
        return NoError();
      }
      char acCmd[30];
      wReturn = Command(acCmd, wsprintf(acCmd, "U1 9 0 %d %d", bTrack, bSector));
      if (wReturn || *(word*)sError != 0x3030) {
        Close(9);
        return NoError();
      }
      wReturn = Command(acCmd, wsprintf(acCmd, "B-P 9 %d", bIndex));
      if (wReturn || *(word*)sError != 0x3030) {
        Close(9);
        return NoError();
      }
      word wCount = 30;
      wReturn = Read(9, (byte*)acCmd, &wCount);
      if (wReturn || *(word*)sError != 0x3030) {
        Close(9);
        return NoError();
      }
      assert(wCount == 30);
      word w = pc - pcName;
      if (acCmd[0] != 1 || memcmp(acCmd + 3, pcName, w)) {
        Close(9);
        return NoError();
      }
      while (w < 16) {
        if ((byte)acCmd[3 + w] != 160) {
          Close(9);
          return NoError();
        }
        w++;
      }
      wReturn = Command(acCmd, wsprintf(acCmd, "B-P 9 %d", bIndex));
      if (wReturn || *(word*)sError != 0x3030) {
        Close(9);
        return NoError();
      }
      wCount = 1;
      byte bNewType = 0;
      wReturn = Write(9, &bNewType, &wCount);
      if (wReturn || *(word*)sError != 0x3030) {
        Close(9);
        return NoError();
      }
      assert(wCount == 1);
      wReturn = Command(acCmd, wsprintf(acCmd, "U2 9 0 %d %d", bTrack, bSector));
      Close(9);
      NoError();
    }
  }
  return wReturn;
}

// Ein Zeichen schreiben
word CLink::Put(word wChannel, byte bValue) {
  // Puffer reservieren, falls noch nicht geschehen
  if (!apbBuffer[wChannel]) {
    apbBuffer[wChannel] = (byte*)malloc(1024);
    assert(apbBuffer[wChannel]);
    if (!apbBuffer[wChannel]) {
      return EOUTPUT;
    }
    awIn[wChannel] = awOut[wChannel] = 2;
  }
  // Zeichen in Puffer bertragen
  apbBuffer[wChannel][awOut[wChannel]++] = bValue;
  // Puffer schreiben, wenn voll
  if (awOut[wChannel] < 1024) {
    return 0;
  } else {
    return Flush(wChannel);
  }
}

// Ende Schreiben
word CLink::Unlisten() {
  return 0;
}

// Ein Zeichen lesen
word CLink::Get(word wChannel) {
  // Puffer reservieren, falls noch nicht geschehen
  if (!apbBuffer[wChannel]) {
    apbBuffer[wChannel] = (byte*)malloc(1024);
    assert(apbBuffer[wChannel]);
    if (!apbBuffer[wChannel]) {
      return EINPUT;
    }
    awIn[wChannel] = awOut[wChannel] = 2;
  }
  // Von Schreiben auf Lesen umschalten, oder Puffer leer
  if (awOut[wChannel] >= awIn[wChannel]) {
    Flush(wChannel);
    // Von Gert in Puffer lesen
    apbBuffer[wChannel][0] = (byte)wDevice;
    apbBuffer[wChannel][1] = (byte)(wChannel | 0x60);
    *(word*)(apbBuffer[wChannel] + 2) = 1024 - 2;
    while (L64Send(wLpt, apbBuffer[wChannel], 4) < 0) {
      if ((*pfnLinkAbortProc)()) {
        sError[0] = 0;
        return ENODEVICE;
      }
    }
    while (L64Receive(wLpt, apbBuffer[wChannel] + 2, 1024 - 2) < 0) {
      if ((*pfnLinkAbortProc)()) {
        sError[0] = 0;
        return ENODEVICE;
      }
    }
    awIn[wChannel] = 2 + L64Result;
    // Status erst beim letzten Zeichen zurckgeben
    awState[wChannel] = GetState(FALSE);
  }
  // Nchstes Zeichen aus Puffer holen
  word wReturn;
  if (awOut[wChannel] < awIn[wChannel]) {
    wReturn = apbBuffer[wChannel][awOut[wChannel]++];
  } else {
    wReturn = 0;
  }
  // Letztes Zeichen? Dann mit Status verknpfen
  if (awOut[wChannel] == awIn[wChannel]) {
    wReturn |= awState[wChannel];
  }
  return wReturn;
}

// Block schreiben
word CLink::Write(word wChannel, byte* pbBuffer, word* pwCount) {
  assert(wChannel <= 14);
  assert(awIn[wChannel] == awOut[wChannel]);
  // Daten in Puffer umkopieren
  byte* pbOut = (byte*)malloc(2 + *pwCount);
  assert(pbOut);
  pbOut[0] = (byte)(wDevice | 0x10);
  pbOut[1] = (byte)(wChannel | 0x60);
  memcpy(pbOut + 2, pbBuffer, *pwCount);
  // Inhalt des Puffers schreiben
  while (L64Send(wLpt, pbOut, 2 + *pwCount) < 0) {
    if ((*pfnLinkAbortProc)()) {
      free(pbOut);
      *pwCount = 0;
      sError[0] = 0;
      return ENODEVICE;
    }
  }
  assert((word)L64Result == 2 + *pwCount);
  free(pbOut);
  return GetState(FALSE);
}

// Block lesen
word CLink::Read(word wChannel, byte* pbBuffer, word* pwCount) {
  assert(wChannel <= 14);
  assert(awIn[wChannel] == awOut[wChannel]);
  // Von Gert in Puffer lesen
  byte abOut[4];
  abOut[0] = (byte)wDevice;
  abOut[1] = (byte)(wChannel | 0x60);
  *(word*)(abOut + 2) = *pwCount;
  *pwCount = 0;
  while (L64Send(wLpt, abOut, 4) < 0) {
    if ((*pfnLinkAbortProc)()) {
      sError[0] = 0;
      return ENODEVICE;
    }
  }
  while (L64Receive(wLpt, pbBuffer, *(word*)(abOut + 2)) < 0) {
    if ((*pfnLinkAbortProc)()) {
      sError[0] = 0;
      return ENODEVICE;
    }
  }
  *pwCount = L64Result;
  return GetState(FALSE);
}

// Datei schlieen
word CLink::Close(word wChannel) {
  // Letzten Datenblock schreiben und Puffer freigeben
  if (apbBuffer[wChannel]) {
    Flush(wChannel);
    free(apbBuffer[wChannel]);
    apbBuffer[wChannel] = NULL;
  }
  // Gert zum Schlieen auffordern
  byte abBuffer[2];
  abBuffer[0] = (byte)(wDevice | 0x10);
  abBuffer[1] = (byte)(wChannel | 0xE0);
  while (L64Send(wLpt, abBuffer, 2) < 0) {
    if ((*pfnLinkAbortProc)()) {
      sError[0] = 0;
      return ENODEVICE;
    }
  }
  return GetState(FALSE);
}

static BOOL FAR PASCAL RelLengthDlgProc(HWND hwnd, WORD wMsg, WORD wParam, LONG lParam) {
  switch (wMsg) {
  case WM_INITDIALOG:
    CenterWindow(hwnd);
    return TRUE;
  case WM_COMMAND:
    switch (wParam) {
    case IDOK:
      {
        word w = GetDlgItemInt(hwnd, IDE_RELLENGTH, NULL, FALSE);
        if (w >= 1 && w <= 255) {
          EndDialog(hwnd, w);
        }
        return TRUE;
      }
    case IDCANCEL:
      EndDialog(hwnd, 0);
      return TRUE;
    }
  }
  return FALSE;
}


// Satzlnge einer relativen Datei liefern
byte CLink::GetRelLength(word wChannel) {
  if (!CanHandle("ReadRecordLength", FALSE)) {
    return (byte)DialogBox(hInst, "RelLength", hwndFrame, (FARPROC)RelLengthDlgProc);
  }
  fData = TRUE;
  word w = Command("M-R\x2B\x02\x10", 6);
  fData = FALSE;
  if (w) {
    return 0;
  }
  assert(L64Result == 18);
  byte bChannel = (byte)(sError[wChannel] & 0x0F);
  fData = TRUE;
  w = Command("M-R\xC7\x00\x10", 6);
  fData = FALSE;
  if (w) {
    return 0;
  }
  assert(L64Result == 18);
  byte bReturn = (byte)sError[bChannel];
  strcpy(sError, "00, OK,00,00");
  return bReturn;
}

// Disketten kopieren
extern flag fAbort;
#if GERMAN
  static char acNoFastAccess[] = "Schnelle Laufwerksroutinen funktionieren nicht!";
#else
  static char acNoFastAccess[] = "Fast loader does not work correctly!";
#endif

flag CLink::CompliantImage(char* pcError, int iLength) {
  assert(pcError);
  assert(iLength > 10);
  assert(apbBuffer[9] == NULL);
  Open(9, "#", 1);
  if (GetError(pcError, iLength)) {
    return FALSE;
  }
  Command("U1 9 0 18 0", 11);
  if (GetError(pcError, iLength)) {
    Close(9);
    return FALSE;
  }
  word wCount = 4;
  Read(9, (byte*)pcError, &wCount);
  flag fReturn = pcError[0] == 18 && pcError[1] == 1 && pcError[2] == 'A';
  if (GetError(pcError, iLength)) {
    Close(9);
    return FALSE;
  }
  assert(wCount == 4);
  Close(9);
  pcError[0] = 0;
  return fReturn;
}

flag CLink::BeginTrack(flag fWrite) {
  char acError[40];
  if (!CanHandle("CompliantImage", TRUE)) {
    GetError(acError, 40);
    return ErrorBox(NULL, acError);
  }
  assert(gpbError);
  pbError = gpbError;
  Command("I", 1);
  word wError = GetError(acError, 40);
  if (fWrite && (wError == 20 || wError == 21)) {
    MessageBeep(MB_ICONQUESTION);
    if (MessageBox(hwndFrame, acWantFormat, acConfirm, MB_ICONQUESTION | MB_OKCANCEL) != IDOK) {
      return FALSE;
    }
    Command("N: ,00", 6);
    wError = GetError(acError, 40);
  }
  if (wError) {
    if (!fAbort) {
      ErrorBox(NULL, acError);
    }
    return FALSE;
  }
  extern flag gfCopySlow;
  iFastAccess = !gfCopySlow && CanHandle("UseL64FastAccess", FALSE);
  if (iFastAccess) {
    iFastAccess = -1;
    if (!SetFastAccess(TRUE)) {
      return FALSE;
    }
  } else {
    Open(2, "#", 1);
    if (GetError(acError, 40)) {
      if (!fAbort) {
        ErrorBox(NULL, acError);
      }
      return FALSE;
    }
  }
  fWriteTrack = fWrite;
  wTrack = 1;
  return TRUE;
}

flag CLink::SetFastAccess(flag fOn) {
  L64State[wLpt] = 255;
  int iNew = fOn ? 1 : -1;
  if (iFastAccess == iNew) {
    return TRUE;
  }
  byte ab[2];
  ab[0] = (byte)(fOn ? 0x28 : 0x29);
  ab[1] = (byte)wDevice;
  if (iRetries < 5) {
    iRetries = 5;
  }
  while (L64Send(wLpt, ab, 2) < 0) {
    L64State[wLpt] = 255;
    if ((*pfnLinkAbortProc)()) {
      if (!fAbort) {
        ErrorBox(NULL, acLinkError);
      }
      return FALSE;
    }
  }
  iFastAccess = iNew;
  L64State[wLpt] = 255;
  return TRUE;
}

int __cdecl CLink::AskUser(word wButtons, char* pcFormat, ...) {
  if (iFastAccess && !SetFastAccess(FALSE)) {
    return IDCANCEL;
  }
  char ac[256];
  wvsprintf(ac, pcFormat, (char*)(&pcFormat + 1));
  MessageBeep(MB_ICONEXCLAMATION);
  return MessageBox(hwndFrame, ac, acError, MB_ICONEXCLAMATION | wButtons);
}

int CLink::AskJobError(byte bError, word wTrack, word wSector) {
  const int iButtons = MB_ABORTRETRYIGNORE | MB_DEFBUTTON2;
  switch (bError & 0x0F) {
  case 2:
    return AskUser(iButtons, "20,READ ERROR,%02d,%02d\n\nHeader nicht gefunden!", wTrack, wSector);
  case 3:
    return AskUser(iButtons, "21,READ ERROR,%02d,%02d\n\nSynchronmarkierung nicht gefunden!", wTrack, wSector);
  case 4:
    return AskUser(iButtons, "22,READ ERROR,%02d,%02d\n\nPrfsummenfehler im Header!", wTrack, wSector);
  case 5:
    return AskUser(iButtons, "23,READ ERROR,%02d,%02d\n\nPrfsummenfehler in den Daten!", wTrack, wSector);
  case 0:
  case 6:
    return AskUser(iButtons, "24,READ ERROR,%02d,%02d\n\nPrfsummenfehler im Header oder im Datenblock!", wTrack, wSector);
  case 7:
    return AskUser(iButtons, "25,WRITE ERROR,%02d,%02d\n\nBlock wurde nicht richtig geschrieben!", wTrack, wSector);
  case 8:
    return AskUser(iButtons, "26,WRITE PROTECT ON,%02d,%02d\n\nDiskette ist schreibgeschtzt!", wTrack, wSector);
  case 9:
    return AskUser(iButtons, "27,READ ERROR,%02d,%02d\n\nPrfsummenfehler im Header!", wTrack, wSector);
  case 10:
    return AskUser(iButtons, "28,WRITE ERROR,%02d,%02d\n\nNachfolgende Synchronmarkierung nicht gefunden!", wTrack, wSector);
  case 11:
    return AskUser(iButtons, "29,DISK ID MISMATCH,%02d,%02d\n\nFalsche Diskettennummer im Block!", wTrack, wSector);
  case 15:
    return AskUser(iButtons, "74,DRIVE NOT READY,%02d,%02d\n\nKeine Diskette im Laufwerk!", wTrack, wSector);
  default:
    #if GERMAN
      return AskUser(iButtons, "Unbekannter Job-Fehlercode $%02X auf Spur %d Sektor %d!", bError, wTrack, wSector);
    #else
      return AskUser(iButtons, "Unknown job error code $%02X on track %d sector %d!", bError, wTrack, wSector);
    #endif
  }
}

byte JobError(int iError) {
  if (iError >= 20 && iError <= 29) {
    return (byte)(2 + iError - 20);
  } else {
    return 15;
  }
}

word CLink::ReadNextTrack(byte* pbBuffer) {
  assert(pbBuffer);
  assert(!fWriteTrack);
  assert(wTrack < 36);
  word wSectors;
  if (wTrack < 25) {
    if (wTrack < 18) {
      wSectors = 21;
    } else {
      wSectors = 19;
    }
  } else {
    if (wTrack < 31) {
      wSectors = 18;
    } else {
      wSectors = 17;
    }
  }
  if (iFastAccess) {
    if (iFastAccess < 0 && !SetFastAccess(TRUE)) {
      return FALSE;
    }
    pbBuffer[0] = 0x2A;
    pbBuffer[1] = (byte)wTrack;
    pbBuffer[2] = 1;
    while (L64Send(wLpt, pbBuffer, 3) < 0) {
      if ((*pfnLinkAbortProc)()) {
        if (!fAbort) {
          ErrorBox(NULL, acLinkError);
        }
        return FALSE;
      }
    }
    word wSize = 1 + wSectors * 259;
    byte* pb = (byte*)malloc(wSize);
    assert(pb);
    while (L64Receive(wLpt, pb, wSize) < 0) {
      if ((*pfnLinkAbortProc)()) {
        free(pb);
        if (!fAbort) {
          ErrorBox(NULL, acLinkError);
        }
        return FALSE;
      }
    }
    if ((word)L64Result != wSize || pb[0] != wSectors) {
      free(pb);
      return ErrorBox(NULL, acNoFastAccess);
    }
    memset(pbBuffer, 1, wSectors * 256);
    for (word w = 0; w < wSectors; w++) {
      pbBuffer[w * 256] = 0x4B;
    }
    for (w = 0; w < wSectors; w++) {
      word wOffset = 1 + w * 259;
      word wSector = pb[wOffset + 2];
      int iError = 0;
      assert(wSector < wSectors);
      if (pb[wOffset] != 1) {
        if (!SetFastAccess(FALSE)) {
          free(pb);
          return FALSE;
        }
        Open(2, "#", 1);
        char ac[40];
        if (GetError(ac, 40)) {
          if (!fAbort) {
            ErrorBox(NULL, ac);
          }
          free(pb);
          return FALSE;
        }
        Command(ac, wsprintf(ac, "U1 2 0 %u %u", wTrack, wSector));
        iError = GetError(ac, 40);
        if (iError) {
          pbError[wSector] = JobError(iError);
        }
        Command("B-P 2 0", 7);
        word wCount = 256;
        Read(2, pb + wOffset + 3, &wCount);
        Close(2);
        if (GetError(ac, 40)) {
          if (!fAbort) {
            ErrorBox(NULL, ac);
          }
          free(pb);
          return FALSE;
        }
      }
      if (iError != 20 && iError != 21 && iError != 27) {
        memcpy(pbBuffer + wSector * 256, pb + wOffset + 3, 256);
      }
    }
    free(pb);
  } else {
    memset(pbBuffer, 0x01, wSectors * 256);
    for (word wSector = 0; wSector < wSectors; wSector++) {
      pbBuffer[wSector * 256] = 0x4B;
      char ac[40];
      Command(ac, wsprintf(ac, "U1 2 0 %u %u", wTrack, wSector));
      int iError = GetError(ac, 40);
      if (iError) {
        pbError[wSector] = JobError(iError);
      }
      word wCount = 256;
      Read(2, pbBuffer + wSector * 256, &wCount);
      if (GetError(ac, 40)) {
        if (!fAbort) {
          ErrorBox(NULL, ac);
        }
        return FALSE;
      }
    }
  }
  wTrack++;
  pbError += wSectors;
  return wSectors * 256;
}

flag CLink::WriteNextTrack(byte* pbBuffer, word wLength) {
  assert(pbBuffer);
  assert(wLength == 17 * 256 || wLength == 18 * 256 || wLength == 19 * 256 || wLength == 21 * 256);
  assert(fWriteTrack);
  assert(wTrack < 36);
  char ac[259];
  if (iFastAccess) {
    for (word wSector = 0; wLength; wSector++) {
    Retry:
      if (iFastAccess < 0 && !SetFastAccess(TRUE)) {
        return FALSE;
      }
      int iTry = 0;
      for (;;) {
        ac[0] = 0x2D;
        ac[1] = (byte)wTrack;
        ac[2] = (byte)wSector;
        memcpy(ac + 3, pbBuffer + wSector * 256, 256);
        while (L64Send(wLpt, (byte*)ac, 3 + 256) < 0) {
          if ((*pfnLinkAbortProc)()) {
            if (!fAbort) {
              ErrorBox(NULL, acLinkError);
            }
            return FALSE;
          }
        }
        while (L64Receive(wLpt, (byte*)ac, 1) < 0) {
          if ((*pfnLinkAbortProc)()) {
            if (!fAbort) {
              ErrorBox(NULL, acLinkError);
            }
            return FALSE;
          }
        }
        if (ac[0] == 1 || ac[0] >= 16) {
          break;
        }
        iTry++;
        if (iTry >= 3) {
          switch (AskJobError(ac[0], wTrack, wSector)) {
          case IDABORT:
          case IDCANCEL:
            return FALSE;
          case IDRETRY:
            goto Retry;
          }
          break;
        }
      }
      wLength -= 256;
    }
  } else {
    Command("B-P 2 0", 7);
    if (GetError(ac, 40)) {
      if (!fAbort) {
        ErrorBox(NULL, ac);
      }
      return FALSE;
    }
    for (word wSector = 0; wLength; wSector++) {
      word wCount = 256;
      Write(2, pbBuffer + wSector * 256, &wCount);
      if (GetError(ac, 40)) {
        if (!fAbort) {
          ErrorBox(NULL, ac);
        }
        return FALSE;
      }
      Command(ac, wsprintf(ac, "U2 2 0 %u %u", wTrack, wSector));
      if (GetError(ac, 40)) {
        if (!fAbort) {
          ErrorBox(NULL, ac);
        }
        return FALSE;
      }
      wLength -= 256;
    }
  }
  wTrack++;
  return TRUE;
}

flag CLink::EndTrack() {
  char acError[40];
  if (iFastAccess) {
    if (iFastAccess > 0 && !SetFastAccess(FALSE)) {
      return FALSE;
    }
  } else {
    Close(2);
  }
  Command("I", 1);
  if (GetError(acError, 40)) {
    if (!fAbort) {
      ErrorBox(NULL, acError);
    }
    return FALSE;
  }
  return TRUE;
}
