// Show directory tree and let the user select

#include "pc64.h"

#pragma warning(disable: 4135)

const int MAXDEPTH = 32;
const int MAXNAMES = 1024;
const int NAMESIZE = 14;

typedef char NAMES[MAXNAMES][NAMESIZE];

struct DIRTREE {
  HANDLE ah[MAXDEPTH];
  short asCount[MAXDEPTH];
  short asIndex[MAXDEPTH];
  short asLength[MAXDEPTH];
  short sDepth;
  short sLine;
  short sWidth;
  short sHeight;
  short sIndent;
  short sCursorX;
  short sCursorY;
};

#ifdef PC64
static word wDrives = 0;
word ScanForC64Drives() {
  static word awFlags[4];
  word wFound = 0;
  assert(hDrives);
  NAMES* pn = (NAMES*)GlobalLock(hDrives);
  assert(pn);
  for (word wLpt = 0; wLpt < 4; wLpt++) {
    byte ab[40];
    L64State[wLpt] = 255;
    ab[0] = (byte)(0x20 | def.fScreenOn);
    if (L64Send(wLpt, ab, 1) < 0) {
      continue;
    }
    ab[0] = 0x22;
    if (L64Send(wLpt, ab, 1) < 0 || L64Receive(wLpt, ab, 6) < 0) {
      continue;
    }
    if (*(word*)ab < L64HEXVER) {
      ErrorBox(NULL,
               #if GERMAN
                 "Sie verwenden eine veraltete Version von L64!\n\nKopieren Sie bitte die aktuelle Version "L64VERSION" ber den Manager auf den externen C64.");
               #else
                 "You are using an old L64 version!\n\nPleas copy the current version "L64VERSION" with <Manager / Copy> to the external C64.");
               #endif
    }
    for (int i = 0; i < 16; i++) {
      if (!(awFlags[wLpt] & 1 << i)) {
        ab[0] = (byte)(0x10 | i);
        ab[1] = 0x6F;
        for (int iTry = 0; iTry < 3; iTry++) {
          if (L64Send(wLpt, ab, 2) > 0) {
            break;
          }
        }
        if (L64Result < 0) {
          break;
        }
        for (iTry = 0; iTry < 3; iTry++) {
          if (L64Receive(wLpt, ab, 40) > 0) {
            break;
          }
        }
        if (L64Result > 1 && !(ab[0] & ~0x40)) {
          CLink link;
          wsprintf((*pn)[wDrives], "%d%d:", wLpt + 1, i);
          link.SetDir((*pn)[wDrives]); // Wozu SetDir? Bildschirm aus und
          wDrives++;                   // Kompatibilittstest
          wFound++;
          awFlags[wLpt] |= 1 << i;
        }
      }
    }
  }
  GlobalUnlock(hDrives);
  if (wFound) {
    WinDrawAllWindows();
  }
  return wFound;
}
#endif // PC64

static void Free(DIRTREE* pdt, short sDepth) {
  int i;
  #ifdef _DEBUG
    assert(sDepth >= 0 && sDepth < MAXDEPTH);
    for (i = pdt->sDepth + 1; i < MAXDEPTH; i++) assert(!pdt->ah[i]);
  #endif
  for (i = pdt->sDepth; i >= sDepth; i--) if (pdt->ah[i]) {
    #ifdef PC64
      if (i) GlobalFree(pdt->ah[i]);
    #else
      GlobalFree(pdt->ah[i]);
    #endif
    pdt->ah[i] = NULL;
    pdt->asCount[i] = pdt->asIndex[i] = pdt->asLength[i] = 0;
  }
  pdt->sDepth = sDepth - 1;
}

static void Update(HWND hwnd) {
  InvalidateRect(hwnd, NULL, TRUE);
  if (GetWindowLong(hwnd, GWL_STYLE) & DTS_NOTIFY) {
    SendMessage(GetParent(hwnd), WM_COMMAND, GetWindowWord(hwnd, GWW_ID), MAKELONG(hwnd, DTN_SELCHANGE));
  }
}

static void OutOfMemory(HWND hwnd) {
  if (GetWindowLong(hwnd, GWL_STYLE) & DTS_NOTIFY) {
    SendMessage(GetParent(hwnd), WM_COMMAND, GetWindowWord(hwnd, GWW_ID), MAKELONG(hwnd, DTN_ERRSPACE));
  }
}

static flag ReadDir(DIRTREE* pdt, HWND hwnd) {
  char acPath[80];
  NAMES* pn;
  uint uFind;
  _find_t find;
  int i;
  short sDepth;
  short sLength;
  pn = (NAMES*)malloc(sizeof NAMES);
  assert(pn);
  if (!pn) {
    OutOfMemory(hwnd);
    return FALSE;
  }
  sDepth = pdt->sDepth + 1;
  assert(sDepth < MAXDEPTH);
  acPath[0] = 0;
  for (i = 0; i <= pdt->sDepth; i++) {
    NAMES* pnTemp = (NAMES*)GlobalLock(pdt->ah[i]);
    assert(pnTemp);
    strcat(acPath, (*pnTemp)[pdt->asIndex[i]]);
    strcat(acPath, "\\");
    GlobalUnlock(pdt->ah[i]);
  }
  char acSubPath[80];
  char* pcSubPath;
  strcpy(acSubPath, acPath);
  pcSubPath = strend(acSubPath);
  strcat(acPath, "*.*");
  assert(strlen(acPath) < 80);
  pdt->asCount[sDepth] = 0;
  pdt->asIndex[sDepth] = 0;
  pdt->asLength[sDepth] = 0;
  i = 0;
  uFind = _dos_findfirst(acPath, _A_SUBDIR, &find);
  while (!uFind && i < MAXNAMES) {
    flag fDir = (find.attrib & _A_SUBDIR) && find.name[0] != '.';
    flag fImage = FALSE;
    if (!(find.attrib & _A_SUBDIR)) {
      char* pcExt = strchr(find.name, '.');
      if (pcExt && !strcmp(pcExt, ".D64")) {
        fImage = TRUE;
      }
    }
    if (fDir || fImage) {
      sLength = strlen(find.name);
      assert(sLength < NAMESIZE - 1);
      memcpy((*pn)[i], find.name, sLength + 1);
      if (pdt->asLength[sDepth] < sLength) {
        pdt->asLength[sDepth] = sLength;
      }
      (*pn)[i][NAMESIZE - 1] = '';
      if (fDir) {
        strcpy(pcSubPath, find.name);
        strcat(pcSubPath, "\\*.*");
        assert(strlen(acSubPath) < 80);
        _find_t subs;
        uint uSubs = _dos_findfirst(acSubPath, _A_SUBDIR, &subs);
        while (!uSubs) {
          if (subs.attrib & _A_SUBDIR) {
            if (subs.name[0] != '.') {
              break;
            }
          } else {
            char* pcExt = strchr(subs.name, '.');
            if (pcExt && !strcmp(pcExt, ".D64")) {
              break;
            }
          }
          uSubs = _dos_findnext(&subs);
        }
        if (uSubs == 0) {
          (*pn)[i][NAMESIZE - 1] = '+';
        }
      }
      i++;
    }
    uFind = _dos_findnext(&find);
  }
  if (!i) {
    free(pn);
    return FALSE;
  }
  if (GetWindowLong(hwnd, GWL_STYLE) & DTS_SORT) {
    qsort(pn, i, NAMESIZE, (int(cdecl*)(const void*, const void*))strcmp);
  }
  HANDLE hMem = GlobalAlloc(GMEM_MOVEABLE, i * NAMESIZE);
  assert(hMem);
  if (!hMem) {
    free(pn);
    OutOfMemory(hwnd);
    return FALSE;
  }
  NAMES* pnTemp = (NAMES*)GlobalLock(hMem);
  assert(pnTemp);
  memcpy(pnTemp, pn, i * NAMESIZE);
  GlobalUnlock(hMem);
  free(pn);
  pdt->ah[sDepth] = hMem;
  pdt->asCount[sDepth] = i;
  pdt->sDepth++;
  return TRUE;
}

LONG __export FAR PASCAL DirTreeWndProc(HWND hwnd, WORD wMsg, WORD wParam, LONG lParam) {
  DIRTREE* pdt;
  char acDir[80];
  char* pc;
  int i, iLine;
  HDC hdc;
  PAINTSTRUCT ps;
  short asIndex[MAXDEPTH];
  short sDepth;
  short sScroll;
  COLORREF crBack, crFore;
  char cLetter;
  word wLength;
  if (wMsg == WM_CREATE) {
    pdt = (DIRTREE*)malloc(sizeof DIRTREE);
    assert(pdt);
    SetWindowLong(hwnd, 0, (LONG)pdt);
    if (!pdt) {
      OutOfMemory(hwnd);
      goto Default;
    }
    memset(pdt, 0, sizeof DIRTREE);
    #ifdef PC64
      pdt->ah[0] = hDrives;
      pdt->asCount[0] = wDrives;
      pdt->asLength[0] = 4;
    #else
      NAMES* pn = (NAMES*)malloc(sizeof NAMES);
      assert(pn);
      if (!pn) {
        OutOfMemory(hwnd);
        goto Default;
      }
      for (i = 0; i < 26; i++) {
        if (GetDriveType(i)) wsprintf((*pn)[pdt->asCount[0]++], "%c:", 'A' + i);
      }
      assert(pdt->asCount[0]);
      HANDLE hMem = GlobalAlloc(GMEM_MOVEABLE, pdt->asCount[0] * NAMESIZE);
      assert(hMem);
      if (!hMem) {
        free(pn);
        OutOfMemory(hwnd);
        goto Default;
      }
      NAMES* pnTemp = (NAMES*)GlobalLock(hMem);
      assert(pnTemp);
      memcpy(pnTemp, pn, pdt->asCount[0] * NAMESIZE);
      GlobalUnlock(hMem);
      free(pn);
      pdt->ah[0] = hMem;
      pdt->asLength[0] = 2;
    #endif
    return 0;
  } else {
    pdt = (DIRTREE*)GetWindowLong(hwnd, 0);
    if (!pdt) goto Default;
  }
  switch (wMsg) {
  case WM_SIZE:
    pdt->sWidth = max(LOWORD(lParam), 1);
    assert(pdt->sWidth > 0);
    pdt->sHeight = max(HIWORD(lParam), 1);
    assert(pdt->sHeight > 0);
    pdt->sIndent = pdt->sWidth / 12 + 1;
    if (pdt->sIndent < 2) {
      pdt->sIndent = 2;
    }
    break;
  case WM_SETFOCUS:
    if (GetWindowLong(hwnd, GWL_STYLE) & DTS_NOTIFY) {
      SendMessage(GetParent(hwnd), WM_COMMAND, GetWindowWord(hwnd, GWW_ID), MAKELONG(hwnd, DTN_SETFOCUS));
    }
    CreateCaret(hwnd, NULL, 1, 1);
    SetCaretPos(pdt->sCursorX, pdt->sCursorY);
    ShowCaret(hwnd);
    break;
  case WM_KILLFOCUS:
    if (GetWindowLong(hwnd, GWL_STYLE) & DTS_NOTIFY) {
      SendMessage(GetParent(hwnd), WM_COMMAND, GetWindowWord(hwnd, GWW_ID), MAKELONG(hwnd, DTN_KILLFOCUS));
    }
    DestroyCaret();
    break;
  case WM_SETTEXT:
    #ifdef PC64
      if (*(LPSTR)lParam && IsLink((LPSTR)lParam)) {
        strcpy(acDir, (LPSTR)lParam);
      } else {
        _fullpath(acDir, (LPSTR)lParam, sizeof acDir);
      }
    #else
      _fullpath(acDir, (LPSTR)lParam, sizeof acDir);
    #endif
    assert(*acDir);
    assert(strlen(acDir) < sizeof acDir);
    AnsiUpper(acDir);
    pc = strtok(acDir, "\\");
    for (sDepth = 0; sDepth < MAXDEPTH; sDepth++) {
      if (!pc) {
        assert(sDepth);
        Free(pdt, sDepth);
        break;
      }
      assert(*pc);
      if (!pdt->ah[sDepth]) {
        if (!ReadDir(pdt, hwnd)) {
          break;
        }
        assert(pdt->ah[sDepth]);
        assert(pdt->sDepth == sDepth);
      }
      NAMES* pnTemp = (NAMES*)GlobalLock(pdt->ah[sDepth]);
      assert(pnTemp);
      if (strcmp(pc, (*pnTemp)[pdt->asIndex[sDepth]])) {
        Free(pdt, sDepth + 1);
        assert(pdt->sDepth == sDepth);
        int iNext = 0;
        for (i = 0; i < pdt->asCount[sDepth]; i++) {
          int iComp = strcmp(pc, (*pnTemp)[i]);
          if (iComp == 0) {
            pdt->asIndex[sDepth] = i;
            goto Found;
          } else if (iComp > 0) {
            iNext = i + 1;
          }
        }
        if (iNext >= pdt->asCount[sDepth]) {
          pdt->asIndex[sDepth] = pdt->asCount[sDepth] - 1;
        } else {
          pdt->asIndex[sDepth] = iNext;
        }
        GlobalUnlock(pdt->ah[sDepth]);
        break;
      }
    Found:
      GlobalUnlock(pdt->ah[sDepth]);
      pc = strtok(NULL, "\\");
    }
    pdt->sLine = pdt->sHeight / 2;
    Update(hwnd);
    break;
  case WM_GETTEXT:
    if (!lParam) {
      return DT_ERR;
    }
    acDir[0] = 0;
    for (i = 0; i <= pdt->sDepth; i++) {
      NAMES* pnTemp = (NAMES*)GlobalLock(pdt->ah[i]);
      assert(pnTemp);
      strcat(acDir, (*pnTemp)[pdt->asIndex[i]]);
      strcat(acDir, "\\");
      GlobalUnlock(pdt->ah[i]);
    }
    wLength = strlen(acDir);
    assert(wLength >= 3 && wLength < 80);
    if (wLength != 3) {
      acDir[--wLength] = 0;
    }
    if (GetWindowLong(hwnd, GWL_STYLE) & DTS_LOWERCASE) {
      AnsiLower(acDir);
    }
    if (wLength >= wParam) {
      wLength = wParam - 1;
    }
    memcpy((LPSTR)lParam, acDir, wLength + 1);
    return wLength;
  case WM_PAINT:
    #ifdef PC64
      pdt->asCount[0] = wDrives;
    #endif
    hdc = BeginPaint(hwnd, &ps);
    if (pdt->sLine < 0) pdt->sLine = 0;
    else if (pdt->sLine > pdt->sHeight - 1) pdt->sLine = pdt->sHeight - 1;
    sDepth = pdt->sDepth;
    sScroll = pdt->sWidth - pdt->asLength[sDepth];
    if (sScroll > 0) {
      sScroll -= sDepth * pdt->sIndent;
      if (sScroll > 0) sScroll = 0;
    } else sScroll = -sDepth * pdt->sIndent;
    memcpy(asIndex, pdt->asIndex, sizeof asIndex);
    for (i = 0; i < pdt->sLine; i++) {
      if (asIndex[sDepth]) {
        asIndex[sDepth]--;
      } else {
        if (sDepth) {
          sDepth--;
        } else {
          pdt->sLine = i;
          break;
        }
      }
    }
    for (iLine = 0; iLine < pdt->sHeight; iLine++) {
      pc = acDir;
      for (i = 1; i < sDepth; i++) {
        if (asIndex[i] < pdt->asCount[i] - 1) *pc = '';
        else *pc = ' ';
        memset(pc + 1, ' ', pdt->sIndent - 1);
        pc += pdt->sIndent;
      }
      if (sDepth) {
        if (asIndex[i] < pdt->asCount[sDepth] - 1) *pc = '';
        else *pc = '';
        memset(pc + 1, '', pdt->sIndent - 2);
        pc += pdt->sIndent - 1;
      }
      assert(pdt->ah[sDepth]);
      NAMES* pnTemp = (NAMES*)GlobalLock(pdt->ah[sDepth]);
      assert(pnTemp);
      if (sDepth) {
        *pc++ = (*pnTemp)[asIndex[sDepth]][NAMESIZE - 1];
      }
      strcpy(pc, (*pnTemp)[asIndex[sDepth]]);
      GlobalUnlock(pdt->ah[sDepth]);
      if (GetWindowLong(hwnd, GWL_STYLE) & DTS_LOWERCASE) {
        AnsiLower(pc);
      }
      if (iLine == pdt->sLine) {
        TextOut(hdc, sScroll, iLine, acDir, pc - acDir);
        crBack = GetBkColor(hdc);
        crFore = GetTextColor(hdc);
        SetBkColor(hdc, crFore);
        SetTextColor(hdc, crBack);
        TextOut(hdc, sScroll + pc - acDir, iLine, pc, strlen(pc));
        SetBkColor(hdc, crBack);
        SetTextColor(hdc, crFore);
        pdt->sCursorX = sScroll + pc - acDir;
        pdt->sCursorY = iLine;
        if (GetFocus() == hwnd) {
          SetCaretPos(pdt->sCursorX, pdt->sCursorY);
        }
        asIndex[sDepth]++;
      } else {
        TextOut(hdc, sScroll, iLine, acDir, strlen(acDir));
        if (asIndex[sDepth] == pdt->asIndex[sDepth]) {
          sDepth++;
          assert(pdt->asCount[sDepth]);
        } else asIndex[sDepth]++;
      }
      while (asIndex[sDepth] == pdt->asCount[sDepth]) {
        if (!sDepth--) goto Finished;
        asIndex[sDepth]++;
      }
    }
  Finished:
    EndPaint(hwnd, &ps);
    break;
  case WM_LBUTTONDOWN:
    SetCapture(hwnd);
  case WM_MOUSEMOVE:
  case WM_MOUSEREPEAT:
    if (GetCapture() != hwnd) break;
    i = HIWORD(lParam) - pdt->sLine;
    if (i) {
      sDepth = pdt->sDepth;
      if (i > 0) {
        if (pdt->asIndex[sDepth] + i >= pdt->asCount[sDepth]) i = pdt->asCount[sDepth] - 1 - pdt->asIndex[sDepth];
      } else {
        if (i < -pdt->asIndex[sDepth]) i = -pdt->asIndex[sDepth];
      }
      if (i) {
        pdt->asIndex[sDepth] += i;
        pdt->sLine += i;
        Update(hwnd);
      }
    }
    break;
  case WM_LBUTTONUP:
    ReleaseCapture();
    break;
  case WM_LBUTTONDBLCLK:
    wParam = '+';
    goto KeyDown;
  case WM_RBUTTONDOWN:
    wParam = '-';
    goto KeyDown;
  case WM_RBUTTONDBLCLK:
    break;
  case WM_KEYDOWN:
  case WM_CHAR:
  KeyDown:
    sDepth = pdt->sDepth;
    switch(wParam) {
    case VK_UP:
      if (pdt->asIndex[sDepth]) {
        pdt->asIndex[sDepth]--;
        pdt->sLine--;
        Update(hwnd);
      }
      break;
    case VK_DOWN:
      if (pdt->asIndex[sDepth] < pdt->asCount[sDepth] - 1) {
        pdt->asIndex[sDepth]++;
        pdt->sLine++;
        Update(hwnd);
      }
      break;
    case VK_PGUP:
      i = pdt->asIndex[sDepth];
      if (i) {
        if (i > pdt->sHeight) i = pdt->sHeight;
        pdt->asIndex[sDepth] -= i;
        pdt->sLine -= i;
        Update(hwnd);
      }
      break;
    case VK_PGDN:
      i = pdt->asCount[sDepth] - pdt->asIndex[sDepth] - 1;
      if (i > 0) {
        if (i > pdt->sHeight) i = pdt->sHeight;
        pdt->asIndex[sDepth] += i;
        pdt->sLine += i;
        Update(hwnd);
      }
      break;
    case VK_END:
      i = pdt->asCount[sDepth] - pdt->asIndex[sDepth] - 1;
      if (i > 0) {
        pdt->asIndex[sDepth] += i;
        pdt->sLine += i;
        Update(hwnd);
      }
      break;
    case VK_HOME:
      i = pdt->asIndex[sDepth];
      if (i) {
        pdt->asIndex[sDepth] = 0;
        pdt->sLine -= i;
        Update(hwnd);
      }
      break;
    case '+':
    case VK_GREYPLUS:
    case VK_SPACE:
    case VK_RIGHT:
      if (sDepth < MAXDEPTH - 1 && ReadDir(pdt, hwnd)) {
        pdt->sLine++;
        Update(hwnd);
      }
      break;
    case '-':
    case VK_GREYMINUS:
    case VK_BACK:
    case VK_LEFT:
      if (sDepth) {
        pdt->sLine -= pdt->asIndex[sDepth] + 1;
        Free(pdt, sDepth);
        Update(hwnd);
      }
      break;
    case VK_CTRL_HOME:
      Free(pdt, 1);
      pdt->asIndex[0] = pdt->sLine = 0;
      Update(hwnd);
      break;
    default:
      if (HIBYTE(wParam) && HIBYTE(wParam) != 0xFF) goto Default;
      cLetter = (char)(LONG)AnsiUpper((LPSTR)(LONG)wParam);
      i = pdt->asIndex[sDepth];
      NAMES* pnTemp = (NAMES*)GlobalLock(pdt->ah[sDepth]);
      assert(pnTemp);
      do {
        if (++i >= pdt->asCount[sDepth]) i = 0;
        if (i == pdt->asIndex[sDepth]) {
          GlobalUnlock(pdt->ah[sDepth]);
          goto Default;
        }
      } while (cLetter != (*pnTemp)[i][0]);
      GlobalUnlock(pdt->ah[sDepth]);
      pdt->sLine += i - pdt->asIndex[sDepth];
      pdt->asIndex[sDepth] = i;
      Update(hwnd);
    }
    break;
  case WM_DESTROY:
    Free(pdt, 0);
    free(pdt);
    break;
  default:
  Default:
    return DefWindowProc(hwnd, wMsg, wParam, lParam);
  }
  return 0;
}

ATOM RegisterDirTreeClass(HANDLE hInstance) {
  WNDCLASS wc;
  memset(&wc, 0, sizeof WNDCLASS);
  wc.style = CS_HREDRAW | CS_VREDRAW | CS_DBLCLKS;
  wc.lpfnWndProc = DirTreeWndProc;
  wc.cbWndExtra = 4;
  wc.hInstance = hInstance;
  wc.hCursor = LoadCursor(NULL, IDC_ARROW);
  wc.hbrBackground = COLOR_WINDOW + 1;
  wc.lpszClassName = "DirTree";
  ATOM aReturn = RegisterClass(&wc);
  assert(aReturn);
  EXTWNDCLASS* pwc = ClassIDToClassStruct(ClassNameToClassID(wc.lpszClassName));
  assert(pwc);
  assert(pwc->idLowestClass == NORMAL_CLASS);
  pwc->idLowestClass = USER_CLASS - 1;
  #ifdef PC64
    hDrives = GlobalAlloc(GMEM_MOVEABLE, 90 * NAMESIZE);
    assert(hDrives);
    NAMES* pn = (NAMES*)GlobalLock(hDrives);
    assert(pn);
    assert(wDrives == 0);
    for (int i = 0; i < 26; i++) {
      if (GetDriveType(i)) {
        wsprintf((*pn)[wDrives++], "%c:", 'A' + i);
      }
    }
    GlobalUnlock(hDrives);
    assert(wDrives);
    if (!fWindowsNT) {
      ScanForC64Drives();
    }
  #endif
  return aReturn;
}
