// Speichern und Laden kompletter Emulator-Zustnde

#include "pc64.h"

char* GetFileWndTitle(FILEWND* pfw) {
  assert(pfw);
  if (pfw->acFileName[0]) {
    return pfw->acFileName;
  }
  if (pfw->acC64Name[0]) {
    return pfw->acC64Name;
  }
  return acUntitled;
}

void ReplaceFileWnd() {
  if (def.fReplace && hwndChild) {
    HWND hwnd = hwndChild;
    do {
      if (!IsIconic(hwnd) && GetWindowLong(hwnd, GWL_WNDPROC) == (LONG)FileWndProc) {
        SendMessage(hwnd, WM_CLOSE, 0, 0);
        break;
      }
      HWND hwndOld = hwnd;
      hwnd = GetWindow(hwndOld, GW_HWNDPREV);
      if (!hwnd) {
        hwnd = GetWindow(hwndOld, GW_HWNDLAST);
      }
      assert(hwnd);
    } while (hwnd != hwndChild);
  }
}

FILEWND* AllocFileWnd() {
  FILEWND* pfw = (FILEWND*)malloc(sizeof FILEWND);
  assert(pfw);
  memset(pfw, 0, sizeof FILEWND);
  return pfw;
}

HWND CreateFileWnd(FILEWND* pfw, WNDPOS* pwp) {
  assert(pfw);
  if (!SwitchToFileWnd(pfw)) {
    return NULL;
  }
  MDICREATESTRUCT mdic;
  mdic.szClass = "PC64File";
  if (pfw->acFileName[0]) {
    mdic.szTitle = pfw->acFileName;
  } else if (pfw->acC64Name[0]) {
    mdic.szTitle = pfw->acC64Name;
  } else {
    mdic.szTitle = acUntitled;
  }
  mdic.hOwner = hInst;
  if (pwp) {
    assert(!pwp->fIsLinkWnd);
    mdic.x = (muldiv(pwp->iLeft, iFrameWidth, 5000) + 1) / 2;
    mdic.y = (muldiv(pwp->iTop, iFrameHeight, 5000) + 1) / 2;
    mdic.cx = (muldiv(pwp->iRight, iFrameWidth, 5000) + 1) / 2 - mdic.x;
    mdic.cy = (muldiv(pwp->iBottom, iFrameHeight, 5000) + 1) / 2 - mdic.y;
    mdic.style = pwp->dStyle | WS_VSCROLL;
  } else {
    mdic.x = mdic.y = mdic.cx = mdic.cy = CW_USEDEFAULT;
    mdic.style = WS_VSCROLL;
  }
  mdic.lParam = 0;
  lSetWindowLong = (long)pfw;
  HWND hwnd = (HWND)SendMessage(hwndClient, WM_MDICREATE, 0, (LONG)(LPSTR)&mdic);
  assert(hwnd);
  return hwnd;
}

int FAR PASCAL GetTempFileName(BYTE bDrive, LPSTR pcPrefix, WORD wNumber, LPSTR pcBuffer) {
  assert(pcBuffer);
  assert(!bDrive);
  char* pc = getenv("TEMP");
  if (!pc) {
    pc = getenv("TMP");
    if (!pc) {
      pc = "C:\\";
    }
  }
  _fullpath(pcBuffer, pc, 80);
  assert(*pcBuffer);
  pc = strend(pcBuffer);
  if (pc[-1] != '\\') {
    *pc++ = '\\';
  }
  if (pcPrefix) {
    assert(strlen(pcPrefix) <= 3);
    strcpy(pc, pcPrefix);
    pc += strlen(pcPrefix);
  }
  if (!wNumber) {
    WORD wTry = wNumber = *(WORD*)0x0000046CL;
    do {
      wsprintf(pc, "%05u.TMP", ++wNumber);
      assert(wNumber != wTry);
    } while (!access(pcBuffer, 0));
  } else {
    wsprintf(pc, "%05u.TMP", wNumber);
  }
  assert(strlen(pcBuffer) < 80);
  AnsiUpper(pcBuffer);
  return (int)wNumber;
}

void PrepareToRun(FILEWND* pfw) {
  assert(pfw);
  iec = pfw->c.iec;
  memcpy(apDevice, pfw->apDevice, sizeof apDevice);
  fNationalKeys = pfw->c.keyboard.fNationalKeys;
  fTabSwitch = pfw->c.keyboard.fTabSwitch;
  awJoyPort[0] = pfw->c.joystick.awPort[0];
  awJoyPort[1] = pfw->c.joystick.awPort[1];
  awJoyFire[0] = pfw->c.joystick.awFire[0];
  awJoyFire[1] = pfw->c.joystick.awFire[1];
  alJoyStart[0] = pfw->c.joystick.alStart[0];
  alJoyStart[1] = pfw->c.joystick.alStart[1];
  alJoyCount[0] = pfw->c.joystick.alCount[0];
  alJoyCount[1] = pfw->c.joystick.alCount[1];
  awJoyFlags[0] = pfw->c.joystick.awFlags[0];
  awJoyFlags[1] = pfw->c.joystick.awFlags[1];
  bVolAdlib = (byte)(pfw->c.sound.abVolume[0] + 1);
  bVolDSP = (byte)(pfw->c.sound.abVolume[1] + 1);
  bVolMaster = pfw->c.sound.abVolume[2];
  bVolFM = pfw->c.sound.abVolume[3];
  bVolVoice = pfw->c.sound.abVolume[4];
  bAdlib = (byte)(wAdlib && pfw->c.sound.abEnable[0]);
  wDSPCommand = wDSPBase && pfw->c.sound.abEnable[1] ? wDSPBase + 12 : 0;
}

flag SwitchToFileWnd(FILEWND* pfw) {
  if (pfw == pfwLast) {
    return TRUE;
  }
  if (pfwLast && pfwLast->fModified) {
    if (!pfwLast->acTempName[0]) {
      GetTempFileName(0, "C64", 0, pfwLast->acTempName);
    }
    assert(pfwLast->acTempName[0]);
    if (!SaveImage(pfwLast, pfwLast->acTempName)) {
      return FALSE;
    }
    pfwLast->fTemp = TRUE;
    pfwLast->fModified = FALSE;
  }
  if (pfw) {
    flag fReset = FALSE;
    if (pfw->fTemp) {
      assert(pfw->acTempName[0]);
      if (!LoadImage(pfw, pfw->acTempName)) {
        return FALSE;
      }
    } else if (pfw->acFileName[0]) {
      if (!LoadImage(pfw, pfw->acFileName)) {
        return FALSE;
      }
    } else {
      if (!SetROMs(pfw)) {
        return FALSE;
      }
      SetKeyboard(pfw, FALSE);
      SetFloppies(pfw, FALSE);
      fReset = TRUE;
    }
    PrepareToRun(pfw);
    if (fReset) {
      memset(lpRam + 2, 0, 65534);
      memset(abColor, 0, 1024);
      fTotalReset = TRUE;
      Reset();
    }
  }
  pfwLast = pfw;
  return TRUE;
}

flag LoadImage(FILEWND* pfw, char* pcName) {
  int i;
  assert(pfw);
  assert(pcName && *pcName);
  OFSTRUCT of;
  int hFile = OpenFile(pcName, &of, OF_READ | OF_SHARE_DENY_NONE);
  if (hFile == -1) {
    return ErrorBox(pcName);
  }
  SetError(EREAD);
  char acTag[9];
  if (_lread(hFile, acTag, 9) != 9) {
    goto Error;
  }
  if (strcmp(acTag, "C64Image")) {
    _lclose(hFile);
    return ErrorBox(pcName, acNoImage);
  }
  if (_lread(hFile, lpRam + 0, 32768U) != 32768U) goto Error;
  if (_lread(hFile, lpRam + 32768U, 32768U) != 32768U) goto Error;
  if (_lread(hFile, abColor, 1024) != 1024) goto Error;
  if (_lread(hFile, pImageStart, iImageSize) != iImageSize) goto Error;
  if (_lread(hFile, &pfw->c, sizeof COMMON) != sizeof COMMON) goto Error;
  if (!SetROMs(pfw)) {
    _lclose(hFile);
    return FALSE;
  }
  SetKeyboard(pfw, FALSE);
  SetFloppies(pfw, FALSE);
  for (i = 0; i < 16; i++) {
    if (pfw->apDevice[i]) {
      if (!pfw->apDevice[i]->Load(hFile)) {
        goto Error;
      }
    }
  }
  if (_lclose(hFile) == -1) {
    return ErrorBox(pcName);
  }
  return TRUE;
Error:
  ErrorBox(pcName);
  _lclose(hFile);
  return FALSE;
}

FILEWND* OpenFileWnd() {
  static char acOpenDir[80];
  char acFileName[80];
  acFileName[0] = 0;
  OPENFILENAME ofn;
  memset(&ofn, 0, sizeof OPENFILENAME);
  ofn.lStructSize = sizeof OPENFILENAME;
  ofn.hwndOwner = hwndFrame;
  ofn.lpstrFilter = acImageFilter;
  ofn.nFilterIndex = 1;
  ofn.lpstrFile = acFileName;
  ofn.nMaxFile = 80;
  ofn.lpstrTitle = acFileOpen;
  ofn.Flags = OFN_HIDEREADONLY | OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST;
  ofn.lpstrDefExt = "C64";
  if (!acOpenDir[0]) {
    strcpy(acOpenDir, ::acDir);
  }
  _chdrive(acOpenDir[0] - '@');
  _chdir(acOpenDir);
  wDialogHelp = IDM_FILEOPEN;
  flag fOK = GetOpenFileName(&ofn);
  wDialogHelp = 0;
  if (!fOK) {
    return NULL;
  }
  _getcwd(acOpenDir, 80);
  FILEWND* pfw = AllocFileWnd();
  assert(pfw);
  addext(pfw->acFileName, acFileName, ".C64");
  return pfw;
}

flag ReloadFileWnd() {
  MessageBeep(MB_ICONQUESTION);
  if (MessageBox(hwndFrame, acReloadFile, acConfirm, MB_ICONQUESTION | MB_OKCANCEL) != IDOK) {
    return FALSE;
  }
  assert(pfw);
  pfw->fModified = FALSE;
  pfw->fTemp = FALSE;
  if (pfwLast == pfw) {
    pfwLast = NULL;
  }
  return SwitchToFileWnd(pfw);
}

flag DuplicateFileWnd() {
  assert(pfw);
  if (!SwitchToFileWnd(pfw)) return FALSE;
  FILEWND* pfwNew = AllocFileWnd();
  assert(pfwNew);
  pfwNew->c = pfw->c;
  char* pc;
  if (pfw->acFileName[0]) {
    strcpy(pfwNew->acFileName, pfw->acFileName);
    pc = strchr(pfwNew->acFileName, '.');
    if (!pc) {
      pc = strend(pfwNew->acFileName);
    }
    for (int i = 0; i < 1000; i++) {
      wsprintf(pc, ".%03d", i);
      if (access(pfwNew->acFileName, 0)) {
        break;
      }
    }
    pc = pfwNew->acFileName;
  } else {
    GetTempFileName(0, "C64", 0, pfwNew->acTempName);
    assert(pfwNew->acTempName[0]);
    pfwNew->fTemp = TRUE;
    pc = pfwNew->acTempName;
  }
  if (!SaveImage(pfw, pc)) {
    goto Error;
  }
  if (CreateFileWnd(pfwNew, NULL)) {
    return TRUE;
  }
  pfwLast = NULL;
  remove(pc);
Error:
  free(pfwNew);
  return FALSE;
}

flag SaveImage(FILEWND* pfw, char* pcName) {
  int i;
  assert(pfw);
  assert(pcName && *pcName);
  OFSTRUCT of;
  int hFile = OpenFile(pcName, &of, OF_CREATE | OF_WRITE | OF_SHARE_DENY_NONE);
  if (hFile == -1) {
    return ErrorBox(pcName);
  }
  SetError(EWRITE);
  if (_lwrite(hFile, "C64Image", 9) != 9) goto Error;
  if (_lwrite(hFile, lpRam + 0, 32768U) != 32768U) goto Error;
  if (_lwrite(hFile, lpRam + 32768U, 32768U) != 32768U) goto Error;
  if (_lwrite(hFile, abColor, 1024) != 1024) goto Error;
  if (_lwrite(hFile, pImageStart, iImageSize) != iImageSize) goto Error;
  if (_lwrite(hFile, &pfw->c, sizeof COMMON) != sizeof COMMON) goto Error;
  for (i = 0; i < 16; i++) {
    if (pfw->apDevice[i]) {
      if (!pfw->apDevice[i]->Save(hFile)) {
        goto Error;
      }
    }
  }
  if (_lclose(hFile) == -1) {
    return ErrorBox(pcName);
  }
  return TRUE;
Error:
  ErrorBox(pcName);
  _lclose(hFile);
  return FALSE;
}

flag SaveFileWnd(HWND hwnd, flag fPrompt) {
  FILEWND* pfw = (FILEWND*)GetWindowLong(hwnd, 0);
  assert(pfw);
  if (!SwitchToFileWnd(pfw)) {
    return FALSE;
  }
  if (fPrompt || !pfw->acFileName[0]) {
    char acFileName[80];
    if (pfw->acFileName[0]) {
      strcpy(acFileName, pfw->acFileName);
    } else if (pfw->acC64Name[0]) {
      ReduceName(pfw->acC64Name, acFileName);
    } else {
      acFileName[0] = 0;
    }
    OPENFILENAME ofn;
    memset(&ofn, 0, sizeof OPENFILENAME);
    ofn.lStructSize = sizeof OPENFILENAME;
    ofn.hwndOwner = hwnd;
    ofn.lpstrFilter = acImageFilter;
    ofn.nFilterIndex = 1;
    ofn.lpstrFile = acFileName;
    ofn.nMaxFile = 80;
    ofn.lpstrTitle = acFileSave;
    ofn.Flags = OFN_HIDEREADONLY | OFN_PATHMUSTEXIST | OFN_OVERWRITEPROMPT;
    ofn.lpstrDefExt = "C64";
    wDialogHelp = IDM_FILESAVEAS;
    flag fOK = GetSaveFileName(&ofn);
    wDialogHelp = 0;
    if (!fOK) {
      return NULL;
    }
    addext(pfw->acFileName, acFileName, ".C64");
    SetWindowText(hwnd, pfw->acFileName);
  }
  assert(pfw->acFileName[0]);
  while (!SaveImage(pfw, pfw->acFileName)) {
    MessageBeep(MB_ICONQUESTION);
    switch (MessageBox(hwndFrame, acNoCurrent, acConfirm, MB_ICONQUESTION | MB_YESNOCANCEL)) {
    case IDNO:
      return TRUE;
    case IDCANCEL:
      return FALSE;
    }
  }
  pfw->fModified = FALSE;
  pfw->fTemp = FALSE;
  return TRUE;
}

void DeleteLogFile(FILEWND* pfwDel) {
  if (pfwDel == pfw) {
    assert(hLogFile);
    if (hLogFile > 0) {
      _lclose(hLogFile);
    }
    hLogFile = NULL;
  }
  assert(pfwDel->acLogName[0]);
  remove(pfwDel->acLogName);
  pfwDel->acLogName[0] = 0;
}

void DeleteTempFiles(FILEWND* pfw) {
  if (pfw->acTempName[0]) {
    remove(pfw->acTempName);
    pfw->acTempName[0] = 0;
    pfw->fTemp = FALSE;
  }
  if (pfw->acLogName[0]) {
    DeleteLogFile(pfw);
  }
}

extern "C" int FAR PASCAL GetInternalWindowPos(HWND hwnd, LPRECT prc, LPPOINT ppt);
flag CloseFileWnd(HWND hwnd, int hFile) {
  char ac[256];
  FILEWND* pfw = (FILEWND*)GetWindowLong(hwnd, 0);
  assert(pfw);
  if (!pfw->fModified && !pfw->fTemp) {
    goto Position;
  }
  wsprintf(ac, acFileHasChanged, GetFileWndTitle(pfw));
  int i;
  if (pfw->acFileName[0]) {
    i = pfw->c.fileopt.wSaveName;
  } else {
    if (def.fSaveNoName) {
      i = 0;
    } else {
      i = 2;
    }
  }
  switch (i) {
  case 0:
    assert(strlen(ac) < 256);
    MessageBeep(MB_ICONQUESTION);
    switch (MessageBox(hwndFrame, ac, acConfirm, MB_ICONQUESTION | MB_YESNOCANCEL)) {
    case IDNO:
      goto Position;
    case IDCANCEL:
      return FALSE;
    }
    break;
  case 2:
    goto Position;
  }
  if (!SaveFileWnd(hwnd, FALSE)) {
    return FALSE;
  }
Position:
  if (hFile > 0 && pfw->acFileName[0]) {
    WNDPOS wp;
    memset(&wp, 0, sizeof WNDPOS);
    strcpy(wp.acName, pfw->acFileName);
    RECT rc;
    POINT pt;
    int iShow = GetInternalWindowPos(hwnd, &rc, &pt);
    ScreenToClient(hwndClient, (LPPOINT)&rc.left);
    ScreenToClient(hwndClient, (LPPOINT)&rc.right);
    wp.iLeft = muldiv(rc.left, 10000, iFrameWidth);
    wp.iTop = muldiv(rc.top, 10000, iFrameHeight);
    wp.iRight = muldiv(rc.right, 10000, iFrameWidth);
    wp.iBottom = muldiv(rc.bottom, 10000, iFrameHeight);
    switch (iShow) {
    case SW_SHOWMAXIMIZED:
      wp.dStyle = WS_MAXIMIZE;
      break;
    case SW_SHOWMINIMIZED:
      wp.dStyle = WS_MINIMIZE;
      break;
    }
    _lwrite(hFile, &wp, sizeof WNDPOS);
  }
  DeleteTempFiles(pfw);
  InvalidateRect(hwnd, NULL, TRUE);
  if (pfwLast == pfw) {
    pfwLast = NULL;
  }
  return TRUE;
}

flag CloseAllWindows() {
  assert(acConfig[0]);
  int hFile = _lopen(acConfig, OF_WRITE | OF_SHARE_DENY_NONE);
  if (hFile > 0) {
    _llseek(hFile, sizeof DEFAULT, 0);
  } else {
    ErrorBox(acConfig);
  }
  flag fReturn = FALSE;
  HWND hwnd = GetWindow(hwndClient, GW_CHILD);
  if (hwnd) {
    for (hwnd = GetWindow(hwnd, GW_HWNDLAST); hwnd; hwnd = GetWindow(hwnd, GW_HWNDPREV)) {
      LONG lWndProc = GetWindowLong(hwnd, GWL_WNDPROC);
      if (lWndProc == (LONG)FileWndProc) {
        if (!CloseFileWnd(hwnd, hFile)) {
          goto Error;
        }
      } else {
        if (lWndProc == (LONG)LinkWndProc) {
          if (!CloseLinkWnd(hwnd, hFile)) {
            goto Error;
          }
        }
      }
    }
  }
  fReturn = TRUE;
Error:
  if (hFile > 0) {
    _chsize(hFile, _tell(hFile));
    _lclose(hFile);
  }
  return fReturn;
}
