;Tastatur, Joystick und Timer-Interrupts

BIOSJOY equ 0

include pc64.inc

;Variablen und Funktionen aus anderen Modulen
data
extrn c wResume:word
extrn c fCtrlBreak:word
extrn c awUpdate:word
extrn c fNationalKeys:word
extrn c fTabSwitch:word
extrn c wCtrlTab:word
extrn c adPCTicks:dword
extrn c adC64Ticks:dword
extrn c wRealTime:word
extrn wPaddleSet:word
extrn c iClocksPerInt:word
extrn c wPerformance:word
extrn c dTicksPerRefresh:dword
extrn c wMainsFreq:word
extrn c wNewLine:word
extrn wLineCompare:word
extrn c UIReadDD00:dword
extrn c UIWriteDD00:dword
extrn c UIWriteDD02:dword
extrn c UIReadDD01:dword
extrn c UIWriteDD01:dword
extrn c UIWriteDD03:dword
data?
extrn c adTimeAdd:dword
extrn c fWindowsNT:word
extrn Events:byte
extrn VIC:byte
extrn VICBasePara:word
extrn CIA1:byte
extrn CIA2:byte
extrn Timer1A:word
extrn Timer1B:word
extrn Timer2A:word
extrn Timer2B:word
extrn CIAEvents:word
extrn dTime1Count:dword
extrn dTime2Count:dword
extrn dTime1Add:dword
extrn dTime2Add:dword
extrn abTime1:byte
extrn abTime2:byte
extrn abAlarm1:byte
extrn abAlarm2:byte
extrn bTime1Running:byte
extrn bTime2Running:byte
extrn bTime1Buffered:byte
extrn bTime2Buffered:byte
code
IMP WriteIOTable,word
IMP UpdateVIC,near
IMP BaseChanged,near
extrn c L64Joystick:far
extrn c ResetTimer:far
IMP GetPCTicks,near

ifdef DEBUG

const
extrn abControlTab:byte
extrn abKeyStick:byte
extrn c awCharTab:word
data
extrn bRShift:byte
extrn bCurrentKey:byte
extrn abMatrix:byte
extrn bExtended:byte
extrn wKeyStick:word
extrn lpOldInt09:dword
data?
extrn dC64Ticks:dword
extrn dUpdates:dword
extrn dDelay:dword
extrn c wRunDebug:word
extrn fnContinue:word
extrn wParam:word
extrn c awJoyPort:word
extrn c awJoyFire:word
extrn c alJoyStart:dword
extrn c alJoyCount:dword
extrn c awJoyFlags:word
extrn awUpdateIndex:word
extrn abReadDigital:byte
extrn awCalibrateLow:word
extrn awCalibrateHigh:word
extrn awPaddleScale:word
extrn awPaddles:word
extrn abPaddles:byte
extrn bPaddles:byte
extrn bReadPaddles:byte
extrn abSerialCounter:byte
code
extrn ShowVICConfig:near
extrn DumpText:far
extrn DumpTime:far
intseg
extrn NewInt09:far
extrn NewInt1C:far
extrn OldInt1C:word

else

;CIA 1 und 2 zurcksetzen
code 1
ResetCIA proc near
  public ResetCIA
  xor BP,BP
  push SI                               ;FS:SI fr CIA[13] setzen
  push FS
  push DS
  pop FS
  mov SI,3
@@Next:
  mov CX,offset @@Cont1                 ;Rcksprungadresse
  xor AL,AL                             ;Register auf 0 setzen
  jmp WriteIOTable[0C00h*2+EBP*2]
@@Cont1:
  mov CX,offset @@Cont2                 ;Rcksprungadresse
  xor AL,AL                             ;Register auf 0 setzen
  jmp WriteIOTable[0D00h*2+EBP*2]
@@Cont2:
  inc BP
  cmp BP,16
  jb @@Next
  pop FS
  pop SI
  mov abTime1[3],91h
  mov abTime2[3],91h
  mov abAlarm1[3],91h
  mov abAlarm2[3],91h
  ret
ResetCIA endp

;Scan-Codes von der PC-Tastatur in die C64-Matrixform wandeln
;  -----xxx = Matrix Spalte
;  ----1--- = Shift/Ctrl/CBM
;  -xxx---- = Matrix Zeile
;  1------- = ungltig
const 1
abScanCodes label byte
  db 80h,80h,70h,73h,10h,13h,20h,23h,30h,33h,40h,43h,50h,53h,00h,77h
  db 76h,11h,16h,21h,26h,31h,36h,41h,46h,51h,56h,61h,01h,7Ah,12h,15h
  db 22h,25h,32h,35h,42h,45h,52h,55h,62h,71h,1Fh,65h,14h,27h,24h,37h
  db 34h,47h,44h,57h,54h,67h,6Ch,80h,7Dh,74h,80h,04h,80h,05h,80h,06h
  db 80h,03h,80h,80h,80h,80h,80h,80h,07h,80h,80h,02h,80h,02h,80h,80h
  db 07h,80h,80h,80h,80h,80h,66h,80h,80h,80h,80h,80h,80h,80h,80h,80h
  db 80h,80h,80h,80h,80h,80h,80h,80h,80h,80h,80h,80h,80h,80h,80h,80h
  db 80h,80h,80h,80h,80h,80h,80h,80h,80h,80h,80h,80h,80h,80h,80h,80h

;Sondercodes in Verbindung mit Shift (Space, Enter) und Ctrl (0..9)
const 2
awCtrlCodes dw 0202h,0203h,0204h,0205h,0206h,0207h,0208h,0209h,020Ah,020Bh
  dw 0139h,011Ch,010Eh,0147h,0250h,0223h,0217h,010Fh
  dw 0052h,0152h,0252h,0452h,0053h,0153h,0253h,0453h,004Fh,014Fh,024Fh,044Fh
CTRLSIZE equ $-awCtrlCodes
;Zugehrige Zeichencodes fr DOS-Tastaturpuffer
abCtrlNew db 144,5,28,159,156,30,31,158,18,146
  db 160,141,148,147,143,8,9,131
  db 92,169,28,168,94,255,30,255,95,95,6,95

;Tabelle fr die gleichzeitig gedrckten Joystick-Richtungen
const 2
abKeyStick label byte
  public abKeyStick
  i=0
  repeat 256
    x=0
    if ((i and 11000001b) ne 0) and ((i and 00110010b) eq 0)
      x = x or 0001b
    endif
    if ((i and 00110010b) ne 0) and ((i and 11000001b) eq 0)
      x = x or 0010b
    endif
    if ((i and 10100100b) ne 0) and ((i and 01011000b) eq 0)
      x = x or 0100b
    endif
    if ((i and 01011000b) ne 0) and ((i and 10100100b) eq 0)
      x = x or 1000b
    endif
    db x
    i=i+1
  endm

data 2
lpOldInt09 dd ?                         ;Alter Tastatur-Handler
  public lpOldInt09
abMatrix db 8 dup (-1)                  ;Tastaturmatrix des C64
  public abMatrix
bExtended db 0                          ;Kennung fr E0h
  public bExtended
bCurrentKey db 40h                      ;Matrixcode der gedrckten Taste
  public bCurrentKey
wKeyStick dw 0                          ;Bitmuster des emulierten Joysticks
  public wKeyStick
data? 4
dC64Ticks dd ?                          ;Zhler fr C64 Takte
  public dC64Ticks
dUpdates dd ?                           ;Zhler fr Refresh-Durchlufe
  public dUpdates
dDelay dd ?                             ;Verzgerung in 1/1193182 Sekunden
  public dDelay
alJoyStart dd ?,?                       ;Geschwindigkeit des Dauerfeuers
  public c alJoyStart
alJoyCount dd ?,?                       ;Zhler fr Dauerfeuer
  public c alJoyCount
abReadDigital db 4 dup (?)              ;Digitale Joysticks (-1 fr Neulesen)
  public abReadDigital
awJoyPort dw ?,?                        ;Index in Joystickanschlu
  public c awJoyPort
awJoyFire dw ?,?                        ;Index in Dauerfeuerfunktion
  public c awJoyFire
awJoyFlags dw ?,?                       ;Zustand Schalter Dauerfeuer
  public c awJoyFlags
awUpdateIndex dw 6 dup (?)              ;0 oder 2 je nach Bit in wRealTime
  public awUpdateIndex
awCalibrateLow dw 4 dup (?)             ;Untergrenze fr Analogjoysticks
  public awCalibrateLow
awCalibrateHigh dw 4 dup (?)            ;Obergrenze fr Analogjoysticks
  public awCalibrateHigh
awPaddleScale dw 4 dup (?)              ;Multiplikationsfaktor * 128 fr SID
  public awPaddleScale
awPaddles dw 4 dup (?)                  ;aktuelle Positionen Analogjoysticks
  public awPaddles
abPaddles db ?,?                        ;Positionen in Digitalform
  public abPaddles
bPaddles db ?                           ;Analogjoysticks vorhanden?
  public bPaddles
bReadPaddles db ?                       ;Joysticks neu lesen
  public bReadPaddles
bRShift db ?                            ;Flags rechte Shifttaste gedrckt
  public bRShift
abSerialCounter db ?,?                  ;Zhler fr Schieberegister Ausgang
  public abSerialCounter
bReadDigital db ?                       ;Zhler alle 5 ms (Start bei 128)
  public bReadDigital

;Neuer Tastaturinterrupt wandelt PC-Scancodes in die Matrixform des C64 um
intseg 1
NewInt09 proc
  public NewInt09
  push AX                               ;Interrupt darf keine Register ndern
  push BX
  push CX
  push DS
  push ES
  mov AX,0040h                          ;Fr Zugriff auf BIOS-Variablen
  mov DS,AX
  mov AX,DGROUP                         ;Datensegment zeigt in den Weltraum
  mov ES,AX
  assume DS:nothing,ES:DGROUP
  mov AX,DS:[001Ch]                     ;Ist noch fr ein Zeichen Platz?
  sub AX,001Eh-2
  and AX,001Eh
  add AX,001Eh
  cmp AX,DS:[001Ah]
  jne @@SpaceOk
  sub AX,001Eh-2                        ;Erstes Zeichen im Puffer lschen
  and AX,001Eh
  add AX,001Eh
  mov DS:[001Ah],AX
@@SpaceOk:
  in AL,60h                             ;Tastaturport lesen
  cmp bExtended,0                       ;Ist es eine erweiterte Taste?
  jne @@Extended
  cmp AL,0E0h                           ;Folgt erweiterte Taste?
  je @@SetExtended
  mov AH,DS:[0017h]                     ;Shift-Status lesen
  and AH,00001111b
  cmp AL,01h                            ;Auf Systemfunktionen prfen
  je @@Escape
  cmp AX,083Eh
  je @@StopAndResume
  cmp AL,53h
  je @@TestReset
  cmp AL,0Eh
  je @@TestReset
  cmp AL,52h
  je @@TestReset
  cmp AL,4Eh
  je @@GreyPlus
  cmp AL,4Ah
  je @@GreyMinus
  cmp AL,43h
  je @@StopAndResume
  cmp AL,44h
  je @@StopAndResume
  cmp AL,57h
  je @@StopAndResume
  cmp AL,58h
  je @@Restore
  cmp AL,0Fh                            ;Auf Tab prfen
  jne @@NoTabSwitch
  cmp fTabSwitch,0                      ;Umschalten mit Strg+Tab erlaubt?
  je @@NoTabSwitch
  mov AH,DS:[0017h]                     ;Auf Strg oder Umschalt+Strg prfen
  test AH,00000100b
  je @@NoTabSwitch
  test AH,00001000b
  jne @@NoTabSwitch
  mov fCtrlBreak,1                      ;bertragungen abbrechen
  or Events,evSTOP                      ;Emulator beenden
  mov wCtrlTab,9404h                    ;Kennung fr Ctrl+Tab setzen
  test AH,00000011b
  je @@NextHandler
  mov wCtrlTab,9407h                    ;Kennung fr Shift+Ctrl+Tab setzen
  jmp @@NextHandler
@@TestReset:
  mov AH,DS:[0017h]                     ;Strg+Alt gedrckt?
  and AH,0Fh
  cmp AH,0Ch
  je @@StopAndResume
@@NoTabSwitch:
  ;test byte ptr DS:[0017h],00100000b    ;Num-Lock aktiv?
  ;jne @@Convert
  test wRealTime,4000h                  ;Laptop: Feuern mit Ctrl links
  je @@FireRightCtrl
  cmp AL,1Dh
  je @@SetFire                          ;Code fr Feuertaste KeyStick prfen
  cmp AL,9Dh
  je @@RelFire
@@FireRightCtrl:
  test wRealTime,8000h                  ;Laptop: Codes fr KeyStick prfen
  jne @@Convert
  mov BL,byte ptr wKeyStick[1]          ;Auf Codes fr KeyStick prfen
  xor BH,BH
  cmp AL,48h
  je @@SetUp
  cmp AL,0C8h
  je @@RelUp
  cmp AL,50h
  je @@SetDown
  cmp AL,0D0h
  je @@RelDown
  cmp AL,4Bh
  je @@SetLeft
  cmp AL,0CBh
  je @@RelLeft
  cmp AL,4Dh
  je @@SetRight
  cmp AL,0CDh
  je @@RelRight
  cmp AL,47h
  je @@SetLeftUp
  cmp AL,0C7h
  je @@RelLeftUp
  cmp AL,49h
  je @@SetRightUp
  cmp AL,0C9h
  je @@RelRightUp
  cmp AL,4Fh
  je @@SetLeftDown
  cmp AL,0CFh
  je @@RelLeftDown
  cmp AL,51h
  je @@SetRightDown
  cmp AL,0D1h
  je @@RelRightDown
@@Convert:
  cmp AL,36h
  je @@PressRShift
  cmp AL,0B6h
  je @@RelRShift
@@RShiftCont:
  mov BL,AL                             ;Zugehrige Position in Matrix holen
  and BX,0000000001111111b
  mov BL,abScanCodes[BX]
  and BL,BL                             ;Gltiger Tastaturcode?
  js @@NextHandler
@@GreyKey:
  mov CL,BL                             ;Bitmuster der Spalte berechnen
  and CL,00000111b                      ;CL = Bit 0-2
  mov AH,00000001b
  shl AH,CL
  mov CH,BL                             ;Index der Zeile berechnen
  shr BX,4
  and AL,AL                             ;Drcken oder Loslassen?
  js @@Release
  not AH                                ;Gewnschte Taste drcken
  and abMatrix[BX],AH
  test CH,00001000b                     ;Shift/Ctrl/CBM ignorieren
  jne @@NextHandler
  shr CH,1                              ;Matrixcode der Taste setzen
  and CH,00111000b
  or CH,CL
  mov bCurrentKey,CH
  mov AH,DS:[0017h]                     ;Shift, Ctrl und Alt holen
  shr AH,1
  setc CL
  or AH,CL
  and AH,00000111b
  mov BX,DI                             ;Tabelle nach Scancode durchsuchen
  mov DI,offset DGROUP:awCtrlCodes
  mov CX,CTRLSIZE/2
  repne scasw
  xchg BX,DI                            ;Nicht gefunden, dann Tastaturtreiber
  jne @@NextHandler
  sub BX,offset DGROUP:awCtrlCodes+2    ;Ersatzcode holen
  shr BX,1
  movzx AX,abCtrlNew[BX]
  mov BX,DS:[001Ch]                     ;In Tastaturpuffer schreiben
  pushf                                 ;Wegen Shift+Pos1-Shift-Pos1 = Shift
  call lpOldInt09                       ;bei nationalem Treiber
  mov DS:[BX],AX
  sub BX,001Eh-2                        ;Eingabezeiger erhhen
  and BX,001Eh
  add BX,001Eh
  mov DS:[001Ch],BX
  jmp @@Skip
@@Release:
  or abMatrix[BX],AH                    ;Gewnschte Taste loslassen
  mov bCurrentKey,40h                   ;Derzeit keine Taste gedrckt
@@NextHandler:
  cmp fNationalKeys,0                   ;Tottasten nicht bergeben, das
  jne @@OldHandler                      ;vermeidet Gepiepe
  cmp AL,07h
  je @@Skip
  cmp AL,0Dh
  je @@Skip
  cmp AL,1Ah
  je @@Skip
  cmp AL,1Bh
  je @@Skip
  cmp AL,28h
  je @@Skip
  cmp AL,29h
  je @@Skip
  cmp AL,2Bh
  je @@Skip
@@OldHandler:
  pushf                                 ;Ursprnglichen Routine ausfhren
  call lpOldInt09
  jmp @@Return
@@PressRShift:
  or bRShift,00000001b                  ;Rechte Shifttaste drcken
  and abMatrix[6],not 00010000b         ;Rechte Shifttaste drcken
  jmp @@NextHandler
@@RelRShift:
  and bRShift,not 00000001b             ;Rechte Shifttaste loslassen
  jne @@NextHandler
  or abMatrix[6],00010000b              ;Rechte Shifttaste loslassen
  jmp @@NextHandler
@@PressUpArrow:
  or bRShift,00000010b                  ;Pfeiltaste oben drcken
  and abMatrix[6],not 00010000b         ;Rechte Shifttaste drcken
  jmp @@RShiftCont
@@RelUpArrow:
  and bRShift,not 00000010b             ;Pfeiltaste oben loslassen
  jne @@RShiftCont
  or abMatrix[6],00010000b              ;Rechte Shifttaste loslassen
  jmp @@RShiftCont
@@PressLeftArrow:
  or bRShift,00000100b                  ;Pfeiltaste links drcken
  and abMatrix[6],not 00010000b         ;Rechte Shifttaste drcken
  jmp @@RShiftCont
@@RelLeftArrow:
  and bRShift,not 00000100b             ;Pfeiltaste links loslassen
  jne @@RShiftCont
  or abMatrix[6],00010000b              ;Rechte Shifttaste loslassen
  jmp @@RShiftCont
@@SetKeyStick:
  mov AL,byte ptr wKeyStick             ;Feuertaste bernehmen
  and AL,00010000b
  or AL,abKeyStick[BX]                  ;Neuer Wert fr Tastaturjoystick
  mov AH,BL                             ;aus der Tabelle lesen
  mov wKeyStick,AX
@@Ignore:
  mov AX,DS:[001Ch]                     ;Eingabe in Tastaturpuffer holen
  pushf                                 ;Ursprnglichen Routine ausfhren
  call lpOldInt09
  mov DS:[001Ch],AX                     ;Taste ignorieren
  jmp @@Return
@@GreyPlus:
  mov AH,DS:[0017h]
  and AH,0Fh
  cmp AH,0010b
  je @@IncPerformance
  cmp AH,0100b
  je @@IncClocksPerInt
  cmp AH,1000b
  jne @@Convert
  cmp wNewLine,0
  je @@OldHandler
  dec wNewLine
@@NewLineChanged:
  push DX
  mov AX,awUpdate[0]
  mul wNewLine
  mov BX,100
  div BX
  mov wLineCompare,AX                   ;Vergleichswert fr VIC Register 12h
  pop DX
  jmp @@OldHandler
@@IncClocksPerInt:
  cmp iClocksPerInt,30
  jge @@OldHandler
  inc iClocksPerInt
  jmp @@OldHandler
@@IncPerformance:
  mov AX,wPerformance
  cmp AX,1000
  jae @@Skip
  add AX,5
  cmp AX,155
  jb @@PerformanceChanged
  add AX,5
  jmp @@PerformanceChanged
@@GreyMinus:
  mov AH,DS:[0017h]
  and AH,0Fh
  cmp AH,0010b
  je @@DecPerformance
  cmp AH,0100b
  je @@DecClocksPerInt
  cmp AH,1000b
  jne @@Convert
  cmp wNewLine,100
  jae @@OldHandler
  inc wNewLine
  jmp @@NewLineChanged
@@DecClocksPerInt:
  cmp iClocksPerInt,-30
  jle @@OldHandler
  dec iClocksPerInt
  jmp @@OldHandler
@@DecPerformance:
  mov AX,wPerformance
  cmp AX,15
  jb @@Skip
  sub AX,5
  cmp AX,150
  jb @@PerformanceChanged
  sub AX,5
@@PerformanceChanged:
  mov wPerformance,AX
  push EAX
  push EBX
  push EDX
  mov EAX,119318200
  xor EDX,EDX
  xor EBX,EBX
  mov BX,wMainsFreq
  div EBX
  xor EDX,EDX
  mov BX,wPerformance
  div EBX
  mov dTicksPerRefresh,EAX
  pop EDX
  pop EBX
  pop EAX
  jmp @@Skip
@@CtrlBreak:
  mov fCtrlBreak,1                      ;Ctrl-Break gedrckt
  jmp @@NextHandler
@@StopAndResume:
  mov AH,AL                             ;Scancode ins HIBYTE
  mov AL,DS:[0017h]                     ;Shift-Status ins Lowbyte
  and AL,00001111b
  mov wResume,AX
@@Escape:
  test Events,evSTOP                    ;bertragung abbrechen bei zweimaligem
  je @@DontCtrlBreak                    ;Escape
  mov fCtrlBreak,1
@@DontCtrlBreak:
  or Events,evSTOP                      ;Emulator beenden
  jmp @@Skip
@@Restore:
  or Events,evNMI                       ;NMI ausfhren
  jmp @@Skip
@@SetExtended:
  mov bExtended,1                       ;Erweiterter Code folgt
  jmp @@NextHandler
@@Extended:
  mov bExtended,0                       ;Flag fr erweiterten Code lschen
  test wRealTime,4000h                  ;Laptop: Feuern mit Ctrl links
  jne @@FireLeftCtrl
  cmp AL,1Dh
  je @@SetFire                          ;Code fr Feuertaste KeyStick prfen
  cmp AL,9Dh
  je @@RelFire
@@FireLeftCtrl:
  cmp AL,46h                            ;Auf Ctrl+Break prfen
  je @@CtrlBreak
  mov BX,0060h                          ;Einfg wird zu Pfund
  cmp AL,52h
  je @@GreyKey
  cmp AL,0D2h
  je @@GreyKey
  mov BL,66h                            ;Entf wird zu Pfeil oben
  cmp AL,53h
  je @@GreyKey
  cmp AL,0D3h
  je @@GreyKey
  mov BL,63h                            ;Pos1 wird zu HOME
  cmp AL,47h
  je @@GreyKey
  cmp AL,0C7h
  je @@GreyKey
  mov BL,71h                            ;Ende wird zu Pfeil links
  cmp AL,4Fh
  je @@GreyKey
  cmp AL,0CFh
  je @@GreyKey
  mov BL,04h                            ;Bild oben wird zu F1
  cmp AL,49h
  je @@GreyKey
  cmp AL,0C9h
  je @@GreyKey
  mov BL,03h                            ;Bild unten wird zu F7
  cmp AL,51h
  je @@GreyKey
  cmp AL,0D1h
  je @@GreyKey
  test wRealTime,8000h                  ;Laptop: Codes fr KeyStick prfen
  je @@NoGreyCursor
  mov BL,byte ptr wKeyStick[1]
  xor BH,BH
  cmp AL,48h
  je @@SetUp
  cmp AL,0C8h
  je @@RelUp
  cmp AL,50h
  je @@SetDown
  cmp AL,0D0h
  je @@RelDown
  cmp AL,4Bh
  je @@SetLeft
  cmp AL,0CBh
  je @@RelLeft
  cmp AL,4Dh
  je @@SetRight
  cmp AL,0CDh
  je @@RelRight
  jmp @@NextHandler
@@NoGreyCursor:
  cmp AL,50h                            ;Auf graue Pfeiltasten prfen
  je @@RShiftCont
  cmp AL,0D0h
  je @@RShiftCont
  cmp AL,4Dh
  je @@RShiftCont
  cmp AL,0CDh
  je @@RShiftCont
  cmp AL,48h
  je @@PressUpArrow
  cmp AL,0C8h
  je @@RelUpArrow
  cmp AL,4Bh
  je @@PressLeftArrow
  cmp AL,0CBh
  je @@RelLeftArrow
  jmp @@NextHandler
@@SetUp:
  or BL,00000001b
  jmp @@SetKeyStick
@@RelUp:
  and BL,11111110b
  jmp @@SetKeyStick
@@SetDown:
  or BL,00000010b
  jmp @@SetKeyStick
@@RelDown:
  and BL,11111101b
  jmp @@SetKeyStick
@@SetLeft:
  or BL,00000100b
  jmp @@SetKeyStick
@@RelLeft:
  and BL,11111011b
  jmp @@SetKeyStick
@@SetRight:
  or BL,00001000b
  jmp @@SetKeyStick
@@RelRight:
  and BL,11110111b
  jmp @@SetKeyStick
@@SetLeftUp:
  or BL,10000000b
  jmp @@SetKeyStick
@@RelLeftUp:
  and BL,01111111b
  jmp @@SetKeyStick
@@SetRightUp:
  or BL,01000000b
  jmp @@SetKeyStick
@@RelRightUp:
  and BL,10111111b
  jmp @@SetKeyStick
@@SetLeftDown:
  or BL,00100000b
  jmp @@SetKeyStick
@@RelLeftDown:
  and BL,11011111b
  jmp @@SetKeyStick
@@SetRightDown:
  or BL,00010000b
  jmp @@SetKeyStick
@@RelRightDown:
  and BL,11101111b
  jmp @@SetKeyStick
@@SetFire:
  or wKeyStick,0000000000010000b        ;Feuer drcken
  pushf                                 ;Ursprnglichen Routine ausfhren
  call lpOldInt09
  and byte ptr DS:[0017h],11111011b     ;Rechte Ctrl-Taste BIOS ignorieren
  jmp @@Return
@@RelFire:
  and wKeyStick,1111111111101111b       ;Feuer loslassen
  jmp @@Ignore
@@Skip:
  in AL,61h                             ;Besttigung an Tastatur senden
  or AL,10000000b
  out 61h,AL
  and AL,01111111b
  out 61h,AL
  mov AL,61h                            ;EOI an Controller geben
  out 20h,AL
@@Return:
  pop ES                                ;Zurck zum unterbrochenen Programm
  pop DS
  pop CX
  pop BX
  pop AX
  iret
  assume DS:DGROUP,ES:nothing
NewInt09 endp

;Tabellen zur Umwandlung von PC-Tastatur in C64-Tastatur
const 2
abControlTab label byte
  public abControlTab
  db 0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0
  db 171,179,177,178,163,183,184,162,185,175,164,127,0  ,0  ,176,174
  db 172,187,165,180,181,161,182,0  ,0  ,0  ,0  ,0  ,173,189,188,190
  db 191,170,167,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,133,137,134,138,135
  db 139,136,140,0  ,0  ,0  ,0  ,19 ,145,133,0  ,157,0  ,29 ,0  ,0
  db 17 ,136,0  ,0  ,137,0  ,138,0  ,139,0  ,140,0  ,0  ,0  ,0  ,0
  db 0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0
  db 0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,129,149,150,151,152,153,154,155
  db 0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0
  db 0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0
  db 0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0
  db 0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0
  db 0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0
  db 0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0
  db 0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0
  db 0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0
awCharTab label word
  public c awCharTab
  dw 0   ,1   ,2   ,3   ,4   ,5   ,6   ,7   ,20  ,3   ,10  ,11  ,12  ,13  ,14  ,15
  dw 16  ,17  ,18  ,19  ,20  ,21  ,22  ,23  ,24  ,25  ,26  ,27  ,28  ,29  ,30  ,31
  dw ' ' ,'!' ,'"' ,'#' ,'$' ,'%' ,'&' ,"'" ,'(' ,')' ,'*' ,'+' ,',' ,'-' ,'.' ,'/'
  dw '0' ,'1' ,'2' ,'3' ,'4' ,'5' ,'6' ,'7' ,'8' ,'9' ,':' ,';' ,'<' ,'=' ,'>' ,'?'
  dw '@' ,'a' ,'b' ,'c' ,'d' ,'e' ,'f' ,'g' ,'h' ,'i' ,'j' ,'k' ,'l' ,'m' ,'n' ,'o'
  dw 'p' ,'q' ,'r' ,'s' ,'t' ,'u' ,'v' ,'w' ,'x' ,'y' ,'z' ,'[' ,0   ,']' ,'^' ,164
  dw 0   ,'A' ,'B' ,'C' ,'D' ,'E' ,'F' ,'G' ,'H' ,'I' ,'J' ,'K' ,'L' ,'M' ,'N' ,'O'
  dw 'P' ,'Q' ,'R' ,'S' ,'T' ,'U' ,'V' ,'W' ,'X' ,'Y' ,'Z' ,0   ,0   ,0   ,0   ,0
  dw 'c' ,'EU','E' ,'A' ,'EA','A' ,'A' ,'C' ,'E' ,'E' ,'E' ,'I' ,'I' ,'I' ,'Ea','a'
  dw 'e' ,0   ,'Ea','O' ,'EO','O' ,'U' ,'U' ,'Y' ,'Eo','Eu',0   ,92  ,0   ,0   ,0
  dw 'A' ,'I' ,'O' ,'U' ,'N' ,'n' ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0
  dw 166 ,166 ,166 ,125 ,179 ,179 ,179 ,174 ,174 ,179 ,125 ,174 ,189 ,189 ,189 ,174
  dw 173 ,177 ,178 ,171 ,96  ,155 ,171 ,171 ,173 ,176 ,177 ,178 ,171 ,96  ,123 ,177
  dw 177 ,178 ,178 ,173 ,173 ,176 ,176 ,123 ,123 ,189 ,176 ,32  ,162 ,161 ,161 ,162
  dw 0   ,'SS',0   ,126 ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0
  dw 0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,160

;Ticker-Handler (wird 18.2 mal pro Sekunde aufgerufen)
intseg 1
NewInt1C proc
  public NewInt1C
  push AX
  push DS
  mov AX,DGROUP
  mov DS,AX
  add adPCTicks[0],65536                ;PC-Ticks weiterzhlen
  adc adPCTicks[4],0
  mov AL,bPaddles                       ;Analogjoysticks neu lesen
  mov bReadPaddles,AL
  mov dword ptr abReadDigital,-1        ;Digitaljoysticks neu lesen
  pop DS
  pop AX
  db 0EAh                               ;Selbstmodifizierender Code
OldInt1C dw ?,?
  public OldInt1C
NewInt1C endp

endif

;Adressen der Joystick-Abfrageroutinen
data 2
EXP afnJoystick,word
afnJoystick dw NoStick,KeyStick,AnalogAStick,AnalogBStick
  dw Lpt1Stick,Lpt2Stick,Lpt3Stick,Lpt4Stick

;Nicht vorhandenen Joystick abfragen
code 4
EXP NoStick,near
NoStick proc near
  xor AL,AL
  ret
NoStick endp

;Abfrage des emulierten Joysticks auf der Tastatur
code 4
EXP KeyStick,near
KeyStick proc near
  mov AL,byte ptr wKeyStick
  ret
KeyStick endp

;Abfrage der Analogjoysticks A und B
code 4
EXP ReadPaddles,near
ReadPaddles proc near
  cmp bPaddles,0
  je @@Return

if BIOSJOY

  mov AH,84h
  mov DX,1
  int 15h
  mov awPaddles[0],AX
  mov awPaddles[2],BX
  mov awPaddles[4],CX
  mov awPaddles[6],DX

else

  cli
  mov DX,0201h
  in AL,DX
  and AL,00001111b
  mov BL,AL
  xor CX,CX
  mov AH,00001111b
  out DX,AL
  in AL,DX
  in AL,DX
  in AL,DX
align 4
@@Next:
  inc CX
  je @@TimeOut
  in AL,DX
  and AL,00001111b
  xor AL,AH
  je @@Next
  test AL,0001b
  je @@NoAX
  mov awPaddles[0],CX
@@NoAX:
  test AL,0010b
  je @@NoAY
  mov awPaddles[2],CX
@@NoAY:
  test AL,0100b
  je @@NoBX
  mov awPaddles[4],CX
@@NoBX:
  test AL,1000b
  je @@NoBY
  mov awPaddles[6],CX
@@NoBY:
  xor AH,AL
  cmp AH,BL
  jne @@Next
  sti
  mov AX,awPaddles[0]                   ;Stellung der Analogjoysticks holen
  mov BX,awPaddles[2]
  mov CX,awPaddles[4]
  mov DX,awPaddles[6]

endif

  xor DI,DI                             ;Analog in digital umrechnen
  cmp awCalibrateHigh[4],CX             ;B Rechts
  rcl DI,1
  cmp CX,awCalibrateLow[4]              ;B Links
  rcl DI,1
  cmp awCalibrateHigh[6],DX             ;B Unten
  rcl DI,1
  cmp DX,awCalibrateLow[6]              ;B Oben
  rcl DI,5
  cmp awCalibrateHigh[0],AX             ;A Rechts
  rcl DI,1
  cmp AX,awCalibrateLow[0]              ;A Links
  rcl DI,1
  cmp awCalibrateHigh[2],BX             ;A Unten
  rcl DI,1
  cmp BX,awCalibrateLow[2]              ;A Oben
  rcl DI,1
  mov word ptr abPaddles,DI             ;digitale Stellung A und B schreiben
@@Return:
  mov bReadPaddles,0                    ;Neulesen zurcksetzen
  ret
@@TimeOut:
  sti
  mov bPaddles,0
  cmp fWindowsNT,1
  jne @@Return
  mov fWindowsNT,3
  jmp @@Return
ReadPaddles endp

code 4
EXP AnalogAStick,near
AnalogAStick proc near
  push DX
  cmp bReadPaddles,0                    ;Neulesen erforderlich?
  je @@GetFire
  call ReadPaddles                      ;Paddles alle 1/18 sec neu einlesen
@@GetFire:
  mov DX,0201h                          ;Feuerknpfe lesen
  in AL,DX
  not AL
  and AL,00110000b
  je @@NoFire
  mov AL,00010000b
@@NoFire:
  or AL,abPaddles[0]                    ;mit Hebelstellung verknpfen
  pop DX
  ret
AnalogAStick endp

code 4
EXP AnalogBStick,near
AnalogBStick proc near
  push DX
  cmp bReadPaddles,0                    ;Neulesen erforderlich?
  je @@GetFire
  call ReadPaddles                      ;Paddles alle 1/18 sec neu einlesen
@@GetFire:
  mov DX,0201h                          ;Feuerknpfe lesen
  in AL,DX
  not AL
  and AL,11000000b
  je @@NoFire
  mov AL,00010000b
@@NoFire:
  or AL,abPaddles[1]                    ;mit Hebelstellung verknpfen
  pop DX
  ret
AnalogBStick endp

;Abfrage der vier digitalen Joysticks an den parallelen Schnittstellen
code 4
EXP Lpt1Stick,near
Lpt1Stick proc near
  mov AL,abReadDigital[0]
  and AL,AL
  jns @@DontRead
  push DX
  push 0
  call L64Joystick
  add SP,2
  pop DX
  mov abReadDigital[0],AL
@@DontRead:
  ret
Lpt1Stick endp

code 4
EXP Lpt2Stick,near
Lpt2Stick proc near
  mov AL,abReadDigital[1]
  and AL,AL
  jns @@DontRead
  push DX
  push 1
  call L64Joystick
  add SP,2
  pop DX
  mov abReadDigital[1],AL
@@DontRead:
  ret
Lpt2Stick endp

code 4
EXP Lpt3Stick,near
Lpt3Stick proc near
  mov AL,abReadDigital[2]
  and AL,AL
  jns @@DontRead
  push DX
  push 2
  call L64Joystick
  add SP,2
  pop DX
  mov abReadDigital[2],AL
@@DontRead:
  ret
Lpt3Stick endp

code 4
EXP Lpt4Stick,near
Lpt4Stick proc near
  mov AL,abReadDigital[3]
  and AL,AL
  jns @@DontRead
  push DX
  push 3
  call L64Joystick
  add SP,2
  pop DX
  mov abReadDigital[3],AL
@@DontRead:
  ret
Lpt4Stick endp

;Adressen der Joystick-Dauerfeuerroutinen
const 2
afnFire dw PressFire,SwitchFire,NoFire

;Dauerfeuer als Taster, awJoyFlags[BX]:
;  Bit  4 = Phase (0 gedrckt, 1 losgelassen)
;  Bit 12 = letzter Zustand Knopf
code 4
EXP PressFire,near
PressFire proc near
  mov CX,awJoyFlags[BX]
  test AL,00010000b                     ;Feuerknopf gedrckt?
  je @@Released
  test CH,00010000b                     ;Ja, dann letzten Status abfragen
  je @@NewPress
  not CL                                ;Phase in Bit 4 zurckgeben
  and AL,CL
  ret
align 4
@@Released:
  mov byte ptr awJoyFlags[BX]+1,0       ;Knopf ist losgelassen
  ret
align 4
@@NewPress:
  mov awJoyFlags[BX],0001000000000000b  ;Knopf neu gedrckt, Phase an
  mov ECX,alJoyStart[EBX*2]             ;Frequenzzhler auf Startwert setzen
  mov alJoyCount[EBX*2],ECX
  xor ECX,ECX                           ;HiByte mu 0 sein
  ret
PressFire endp

;Dauerfeuer als Schalter, awJoyFlags[BX]:
;  Bit  4 = Phase (0 gedrckt, 1 losgelassen)
;  Bit  7 = Letzter Zustand Knopf
;  Bit 12 = Aktueller Zustand ein/aus
code 4
EXP SwitchFire,near
SwitchFire proc near
  mov CX,awJoyFlags[BX]
  test AL,00010000b                     ;Taste gedrckt?
  jne @@Pressed
  test CL,10000000b                     ;Knopf war gedrckt?
  jne @@Release
@@Return:
  or AL,CH                              ;Aktuellen Schalterzustand setzen
  not CL                                ;Phase in Bit 4 zurckgeben
  and AL,CL
  ret
align 4
@@Release:
  and CL,00010000b                      ;Knopf ist losgelassen
  mov awJoyFlags[BX],CX
  jmp @@Return
align 4
@@Pressed:
  test CL,10000000b                     ;Knopf war losgelassen?
  jne @@Return
  or CL,10000000b                       ;Knopf ist gedrckt
  xor CH,00010000b                      ;Aktuellen Schalterzustand umdrehen
  mov awJoyFlags[BX],CX
  je @@Return                           ;Schalterzustand neu an?
  and CL,10000000b                      ;Phase an
  mov awJoyFlags[BX],CX
  mov ECX,alJoyStart[EBX*2]             ;Frequenzzhler auf Startwert setzen
  mov alJoyCount[EBX*2],ECX
  xor ECX,ECX                           ;HiByte mu 0 sein
  jmp @@Return
SwitchFire endp

code 4
EXP NoFire,near
NoFire proc near
  ret
NoFire endp

;Taste ber BIOS Int16 einlesen und in C64-Puffer schreiben
code 4
abJoyToKey label byte
  db 64,64,57,64,50,64,57,50,59,64,59,59,59,59,59,59
  db 60,64,60,60,60,64,60,60,60,64,60,60,60,60,60,60
abJoyToShift label byte
  db 0,0,0,0,4,4,4,4,0,0,0,0,4,4,4,4
  db 1,1,1,1,5,5,5,5,1,1,1,1,5,5,5,5
EXP ReadKey,near
ReadKey proc near
  mov DI,awJoyPort[0]                   ;Joystick an Port 1 lesen
  call afnJoystick[EDI*2]
  xor BX,BX                             ;Dauerfeuer fr Port 1 berechnen
  mov DI,awJoyFire[0]
  call afnFire[EDI*2]
  and AX,0000000000011111b              ;Entsprechende Taste holen
  mov DI,AX
  mov AL,abJoyToKey[DI]
  cmp AL,64                             ;Wenn nicht gedrckt, dann Tastatur
  jne @@SetKey
  mov AL,bCurrentKey                    ;Matrixcode der aktuellen Taste setzen
@@SetKey:
  mov GS:[00CBh],AL
  mov GS:[00C5h],AL
  mov AH,02h                            ;Status der Sondertasten holen
  int 16h
  test AL,01000000b                     ;Shift-Lock auf Shift ummappen
  je @@NoShiftLock
  or AL,00000010b
@@NoShiftLock:
  mov AH,AL                             ;????aclr????aclr
  and AX,0000101000000101b              ;----a-l------c-r
  shr AH,2                              ;------a------c-r
  adc AH,0                              ;------al-----c-r
  or AL,AH                              ;-----cas
  cmp bRShift,0                         ;Shift bei Cursor oben und links
  je @@NoRShift
  or AL,00000001b
@@NoRShift:
  or AL,abJoyToShift[DI]                ;Feuer = Shift, links = Ctrl
  mov GS:[028Dh],AL                     ;Neuen Status setzen
  mov AH,AL
  xchg AL,GS:[028Eh]
  cmp AH,00000011b                      ;Shift+CBM gedrckt?
  jne @@NoShiftCBM
  cmp AL,AH                             ;Shift+CBM bereits erkannt?
  je @@NoShiftCBM
  cmp byte ptr GS:[0291h],80h           ;Umschalten erlaubt?
  jae @@NoShiftCBM
  xor VIC[24],00000010b                 ;Zwischen klein und gro umschalten
  ifdef DEBUG
    push offset @@NoShiftCBM
    mov CX,offset ShowVICConfig
  else
    mov CX,offset @@NoShiftCBM
  endif
  jmp BaseChanged
align 4
@@NoShiftCBM:
  mov AH,01h                            ;Was liegt an?
  int 16h
  je @@NoKey
  xor AH,AH                             ;Taste holen
  int 16h
  and AH,AH                             ;Umwandlung bereits erfolgt?
  je @@StoreKey
  and AL,AL                             ;Ist es eine Steuertaste?
  jne @@NoControl
  movzx BX,AH                           ;Ersatzcode holem
  movzx AX,abControlTab[BX]
  jmp @@CheckKey
@@NoControl:
  movzx BX,AL                           ;Normale Zeichentaste umwandeln
  mov AX,awCharTab[EBX*2]
@@CheckKey:
  and AX,AX                             ;Taste gltig?
  je @@NoKey
@@StoreKey:
  movzx BX,byte ptr GS:[00C6h]          ;Index Tastaturpuffer Eingabe
@@AnotherKey:
  cmp BL,GS:[0289h]                     ;Noch Platz im Puffer?
  jae @@NoSpace
  mov GS:[0277h+BX],AL                  ;Zeichen schreiben
  inc BL                                ;Zeiger erhhen
  shr AX,8                              ;Ist es ein 2-Zeichen-Code?
  jne @@AnotherKey
@@NoSpace:
  mov GS:[00C6h],BL                     ;Neuen Pufferzeiger setzen
@@NoKey:
  pop CX                                ;Rcksprungadresse holen
  mov AL,7Fh                            ;Tastatur-Matrix Abfrage auf Default
  jmp WriteCIA10
ReadKey endp

;Emulator einschalten
code 1
WaitTicker proc near
  push ES
  mov AX,0
  mov ES,AX
  mov AL,ES:[046Ch]
@@Wait:
  cmp ES:[046Ch],AL
  je @@Wait
  pop ES
  ret
WaitTicker endp
EXP StartCIA,near
StartCIA proc near
  xor EAX,EAX
  mov dword ptr awPaddles[0],EAX
  mov dword ptr awPaddles[4],EAX
  test fWindowsNT,2
  jne @@NoPaddles
  call WaitTicker
  mov bPaddles,1
  call ReadPaddles
  ;read twice, or some joysticks will not read correctly
  call WaitTicker
  mov bPaddles,1
  call ReadPaddles
@@NoPaddles:
  mov DI,6                              ;Joysticks kalibrieren
@@NextPaddle:
  mov SI,awPaddles[DI]                  ;Multiplikationsfaktor * 128 fr SID
  xor AX,AX
  and SI,SI
  je @@Zero
  mov AX,128*128
  xor DX,DX
  div SI
@@Zero:
  mov awPaddleScale[DI],AX
  imul EBX,ESI,5                        ;Grenzen fr Umwandlung in digital
  shr EBX,4
  lea AX,[SI+BX]
  mov awCalibrateHigh[DI],AX
  sub SI,BX
  mov awCalibrateLow[DI],SI
  sub DI,2
  jns @@NextPaddle
  mov AL,bPaddles                       ;Stellung beim nchsten Zugriff lesen
  mov bReadPaddles,AL
  mov word ptr abPaddles,0
  mov dword ptr abReadDigital,-1        ;Alle 4 digitalen Joysticks lesen
  mov dword ptr abMatrix[0],-1          ;Tastaturvariablen zurcksetzen
  mov dword ptr abMatrix[4],-1
  mov bExtended,0
  mov wKeyStick,0
  mov bCurrentKey,40h
  mov bRShift,0
  mov AX,3509h                          ;Vektor fr Tastaturinterrupt setzen
  int 21h
  mov word ptr lpOldInt09[0],BX
  mov word ptr lpOldInt09[2],ES
  push DS
  mov AX,INTSEG_TEXT
  mov DS,AX
  mov DX,offset INTSEG_TEXT:NewInt09
  mov AX,2509h
  int 21h
  pop DS
  xor EAX,EAX                           ;Zeitdifferenzen initialisieren
  mov adPCTicks[0],EAX
  mov adPCTicks[4],EAX
  mov adC64Ticks[0],EAX
  mov adC64Ticks[4],EAX
  mov word ptr abSerialCounter,AX       ;Kein Interrupt durch Schieberegister
  call ResetTimer                       ;Timer 0 in Modus 2 schalten
  mov AX,351Ch                          ;Interruptvektor fr Ticker setzen
  int 21h
  push DS
  mov AX,INTSEG_TEXT
  mov DS,AX
  assume DS:INTSEG_TEXT
  mov OldInt1C[0],BX
  mov OldInt1C[2],ES
  mov AX,251Ch
  mov DX,offset NewInt1C
  int 21h
  pop DS
  assume DS:DGROUP
  ifdef DEBUG
    xor EAX,EAX                         ;Feld awUpdateIndex[] lschen
    mov dword ptr awUpdateIndex[0],EAX
    mov dword ptr awUpdateIndex[4],EAX
    mov dword ptr awUpdateIndex[8],EAX
  else
    movzx EAX,wRealTime                 ;Feld awUpdateIndex[] initialisieren
    shl AX,1
    xor BX,BX
  @@NextIndex:
    mov CX,AX
    and CX,0000000000000010b
    mov awUpdateIndex[BX],CX
    shr AX,1
    add BX,2
    cmp BX,6*2
    jb @@NextIndex
  endif
  ret
StartCIA endp

;Emulator ausschalten
code 1
EXP StopCIA,near
StopCIA proc near
  push DS
  mov AX,INTSEG_TEXT
  mov DS,AX
  assume DS:INTSEG_TEXT
  lds DX,dword ptr OldInt1C
  assume DS:nothing
  mov AX,251Ch
  int 21h
  pop DS
  assume DS:DGROUP
  push DS
  lds DX,lpOldInt09
  assume DS:nothing
  mov AX,2509h
  int 21h
  pop DS
  assume DS:DGROUP
  mov AX,0C00h
  int 21h
  ret
StopCIA endp

comment #  Funktionsweise von Daten- und Richtungsregister
           ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Beim Schreiben werden alle Bits in das Datenregister bernommen. Tatschlich
werden jedoch nur diejenigen Leitungen nach Low gezogen, die im
Richtungsregister auf Ausgabe geschaltet sind. Das Datenregister mu also
bei jedem Lesen mit dem 1er-Komplement des Richtungsregisters oder-verknpft
werden. Hinzu kommen dann noch die verschiedenen Leitungen von Tastatur und
Joysticks, die allesamt Low-aktiv sind.

                        Port B (abMatrix Inhalt)
              7     6     5     4     3     2     1     0
           Ŀ
         7 STOP   Q   CBM SPACE  2  CTRL       1  
           Ĵ
         6   /        =  RSHFTHOME   ;    *      
           Ĵ
         5   ,    @    :    .    -    L    P    +  
           Ĵ
 Port A  4   N    O    K    M    0    J    I    9  
abMatrix   Ĵ
 Index   3   V    U    H    B    8    G    Y    7  
           Ĵ
         2   X    T    F    C    6    D    R    5  
           Ĵ
         1 LSHFT  E    S    Z    4    A    W    3  
           Ĵ
         0     F5   F3   F1   F7     ENTER DEL 
           

Port A lesen (Low-aktiv)
  1) Alle Ausgnge, die 0 sind (~Richtung A | Daten A)
& 2) Alle Indices i, bei denen es bereinstimmende Nullbits zwischen
     abMatrix[i] und Port B gibt
& 3) Joystick 2

Port B lesen (Low-aktiv)
  1) Alle Ausgnge, die 0 sind (~Richtung B | Daten B)
& 2) Alle abMatrix[i], bei denen Port A[i] Ausgang und Null ist
& 3) Joystick 1
#

;CIA1 Register 0

code 4
EXP ReadCIA10,near
ReadCIA10 proc near
  push CX                               ;Rcksprungadresse retten
  push BX                               ;Wird gebraucht fr Emulator
  mov AX,word ptr CIA1[2]               ;Alle 0-Bits in beiden Ports holen
  not AX
  or AX,word ptr CIA1[0]
  mov CX,word ptr abMatrix[0]           ;bereinstimmende Null-Bits zwischen
  or CL,AH                              ;Port B und abMatrix suchen
  cmp CL,-1
  rcr BH,1
  or CH,AH
  cmp CH,-1
  rcr BH,1
  mov CX,word ptr abMatrix[2]
  or CL,AH
  cmp CL,-1
  rcr BH,1
  or CH,AH
  cmp CH,-1
  rcr BH,1
  mov CX,word ptr abMatrix[4]
  or CL,AH
  cmp CL,-1
  rcr BH,1
  or CH,AH
  cmp CH,-1
  rcr BH,1
  mov CX,word ptr abMatrix[6]
  or CL,AH
  cmp CL,-1
  rcr BH,1
  or CH,AH
  cmp CH,-1
  rcr BH,1
  not BH
  and BH,AL
  push BX
  mov DI,awJoyPort[2]                   ;Joystick an Port 2 lesen
  call afnJoystick[EDI*2]
  mov BX,2                              ;Dauerfeuer fr Port 2 berechnen
  mov DI,awJoyFire[2]
  call afnFire[EDI*2]
  not AL                                ;Kontakte sind Low-Aktiv
  pop BX                                ;Mit Tastaturmatrix verknpfen
  and AL,BH
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "1 Daten Port A (Tastatur Ausgang und Joystick 2) lesen (%02X)",<AX>
    else
      INFO "1 read Port A data (keyboard matrix output and joystick 2) (%02X)",<AX>
    endif
    TEND
  endif
  pop BX
  ret
ReadCIA10 endp

code 4
EXP WriteCIA10,near
WriteCIA10 proc near
  mov CIA1[0],AL
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "1 Daten Port A schreiben (%02X)",<AX>
    else
      INFO "1 write Port A data (%02X)",<AX>
    endif
    TEND
  endif
  mov AL,CIA1[2]                        ;Mit Richtungsregister verknpfen
CIA1AChanged label near
  not AL
  or AL,CIA1[0]
  mov DI,-4                             ;Keine Paddles ausgewhlt
  and AL,11000000b
  je @@Return
  mov DI,4                              ;Bit 7 = Set Nummer 2
  js @@Return
  mov DI,0                              ;Bit 6 = Set Nummer 1
@@Return:
  mov wPaddleSet,DI
  ifdef DEBUG
    TBEG CIA
    sar DI,2
    inc DI
    if GERMAN
      INFO2 "Bit 6,7: Paddle %d ausgewhlt",<DI>
    else
      INFO2 "Bit 6,7: paddle %d selected",<DI>
    endif
    TEND
  endif
  jmp CX
WriteCIA10 endp

;CIA1 Register 1

code 4
EXP ReadCIA11,near
ReadCIA11 proc near
  push CX                               ;Rcksprungadresse retten
  push BX                               ;Wird gebraucht fr Emulator
  mov AX,word ptr CIA1[2]               ;Alle 0-Bits in beiden Ports holen
  not AX
  or AX,word ptr CIA1[0]
  shr AL,1                              ;CL = 0 oder -1
  sbb CL,CL
  shr AL,1                              ;CH = 0 oder -1
  sbb CH,CH
  or CX,word ptr abMatrix[0]            ;Die ersten beiden Zeilen holen
  and AH,CL                             ;und zum Ergebnis hinzufgen
  and AH,CH
  shr AL,1                              ;Dasselbe noch 3 mal
  sbb CL,CL
  shr AL,1
  sbb CH,CH
  or CX,word ptr abMatrix[2]
  and AH,CL
  and AH,CH
  shr AL,1
  sbb CL,CL
  shr AL,1
  sbb CH,CH
  or CX,word ptr abMatrix[4]
  and AH,CL
  and AH,CH
  shr AL,1
  sbb CL,CL
  shr AL,1
  sbb CH,CH
  or CX,word ptr abMatrix[6]
  and AH,CL
  and AH,CH
  push AX
  mov DI,awJoyPort[0]                   ;Joystick an Port 1 lesen
  call afnJoystick[EDI*2]
  xor BX,BX                             ;Dauerfeuer fr Port 1 berechnen
  mov DI,awJoyFire[0]
  call afnFire[EDI*2]
  not AL                                ;Kontakte sind Low-Aktiv
  pop BX                                ;Mit Tastaturmatrix verknpfen
  and AL,BH
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "1 Daten Port B (Tastatur Eingang und Joystick 1) lesen (%02X)",<AX>
    else
      INFO "1 read Port B data (keyboard matrix input and joystick 1) (%02X)",<AX>
    endif
    TEND
  endif
  pop BX
  ret
ReadCIA11 endp

code 4
EXP WriteCIA11,near
WriteCIA11 proc near
  mov CIA1[1],AL
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "1 Daten Port B schreiben (%02X)",<AX>
    else
      INFO "1 write Port B data (%02X)",<AX>
    endif
    TEND
  endif
  jmp CX
WriteCIA11 endp

;CIA1 Register 2

code 4
EXP ReadCIA12,near
ReadCIA12 proc near
  mov AL,CIA1[2]
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "1 Richtung Port A lesen (%02X)",<AX>
    else
      INFO "1 read Port A data direction (%02X)",<AX>
    endif
    TEND
  endif
  jmp CX
ReadCIA12 endp

code 4
EXP WriteCIA12,near
WriteCIA12 proc near
  mov CIA1[2],AL
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "1 Richtung Port A schreiben (%02X)",<AX>
    else
      INFO "1 write Port A data direction (%02X)",<AX>
    endif
    TEND
  endif
  jmp CIA1AChanged
WriteCIA12 endp

;CIA1 Register 3

code 4
EXP ReadCIA13,near
ReadCIA13 proc near
  mov AL,CIA1[3]
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "1 Richtung Port B lesen (%02X)",<AX>
    else
      INFO "1 read Port B data direction (%02X)",<AX>
    endif
    TEND
  endif
  jmp CX
ReadCIA13 endp

code 4
EXP WriteCIA13,near
WriteCIA13 proc near
  mov CIA1[3],AL
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "1 Richtung Port B schreiben (%02X)",<AX>
    else
      INFO "1 write Port B data direction (%02X)",<AX>
    endif
    TEND
  endif
  jmp CX
WriteCIA13 endp

;CIA1 Register 4

code 4
EXP ReadCIA14,near
ReadCIA14 proc near
  mov AX,Timer1A
  ifdef DEBUG
    TBEG CIA
    if GERMAN
      INFO "1 Low Byte Zhler A lesen (%04X)",<AX>
    else
      INFO "1 read Timer A low byte (%04X)",<AX>
    endif
    TEND
  endif
  jmp CX
ReadCIA14 endp

code 4
EXP WriteCIA14,near
WriteCIA14 proc near
  mov CIA1[4],AL
  ifdef DEBUG
    TBEG CIA
    mov AX,word ptr CIA1[4]
    if GERMAN
      INFO "1 Low Byte Zhler A schreiben (%04X)",<AX>
    else
      INFO "1 write Timer A low byte (%04X)",<AX>
    endif
    TEND
  endif
  jmp CX
WriteCIA14 endp

;CIA1 Register 5

code 4
EXP ReadCIA15,near
ReadCIA15 proc near
  ifdef DEBUG
    TBEG CIA
    mov AX,Timer1A
    if GERMAN
      INFO "1 High Byte Zhler A lesen (%04X)",<AX>
    else
      INFO "1 read Timer A high byte (%04X)",<AX>
    endif
    TEND
  endif
  mov AL,byte ptr Timer1A[1]
  jmp CX
ReadCIA15 endp

code 4
EXP WriteCIA15,near
WriteCIA15 proc near
  mov CIA1[5],AL
  ifdef DEBUG
    TBEG CIA
    mov AX,word ptr CIA1[4]
    if GERMAN
      INFO "1 High Byte Zhler A schreiben (%04X)",<AX>
    else
      INFO "1 write Timer A high byte (%04X)",<AX>
    endif
    TEND
  endif
  test CIA1[14],00000001b               ;Zhler angehalten?
  jne @@Running
  mov AX,word ptr CIA1[4]               ;Ja, dann Wert bernehmen
  mov Timer1A,AX
@@Running:
  jmp CX
WriteCIA15 endp

;CIA1 Register 6

code 4
EXP ReadCIA16,near
ReadCIA16 proc near
  mov AX,Timer1B
  ifdef DEBUG
    TBEG CIA
    if GERMAN
      INFO "1 Low Byte Zhler B lesen (%04X)",<AX>
    else
      INFO "1 read Timer B low byte (%04X)",<AX>
    endif
    TEND
  endif
  jmp CX
ReadCIA16 endp

code 4
EXP WriteCIA16,near
WriteCIA16 proc near
  mov CIA1[6],AL
  ifdef DEBUG
    TBEG CIA
    mov AX,word ptr CIA1[6]
    if GERMAN
      INFO "1 Low Byte Zhler B schreiben (%04X)",<AX>
    else
      INFO "1 write Timer B low byte (%04X)",<AX>
    endif
    TEND
  endif
  jmp CX
WriteCIA16 endp

;CIA1 Register 7

code 4
EXP ReadCIA17,near
ReadCIA17 proc near
  ifdef DEBUG
    TBEG CIA
    mov AX,Timer1B
    if GERMAN
      INFO "1 High Byte Zhler B lesen (%04X)",<AX>
    else
      INFO "1 read Timer B high byte (%04X)",<AX>
    endif
    TEND
  endif
  mov AL,byte ptr Timer1B[1]
  jmp CX
ReadCIA17 endp

code 4
EXP WriteCIA17,near
WriteCIA17 proc near
  mov CIA1[7],AL
  ifdef DEBUG
    TBEG CIA
    mov AX,word ptr CIA1[6]
    if GERMAN
      INFO "1 High Byte Zhler B schreiben (%04X)",<AX>
    else
      INFO "1 write Timer B high byte (%04X)",<AX>
    endif
    TEND
  endif
  test CIA1[15],00000001b               ;Zhler angehalten?
  jne @@Running
  mov AX,word ptr CIA1[6]               ;Ja, dann Wert bernehmen
  mov Timer1B,AX
@@Running:
  jmp CX
WriteCIA17 endp

;CIA1 Register 8

code 4
EXP ReadCIA18,near
ReadCIA18 proc near
  mov AL,abTime1[0]
  cmp bTime1Buffered,0
  je @@NoBuffer
  mov AL,CIA1[8]
  ifdef DEBUG
  @@NoBuffer:
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "1 Uhrzeit Zehntelsekunden lesen (%02X)",<AX>
    else
      INFO "1 read TOD clock 10ths of seconds (%02X)",<AX>
    endif
    cmp bTime1Buffered,0
    je @@NoBuffer1
    mov bTime1Buffered,0
    if GERMAN
      INFO2 "Pufferung wird aufgehoben"
    else
      INFO2 "TOD clock buffering cancelled"
    endif
  @@NoBuffer1:
    TEND
  else
    mov bTime1Buffered,0
  @@NoBuffer:
  endif
  jmp CX
ReadCIA18 endp

code 4
EXP WriteCIA18,near
WriteCIA18 proc near
  test CIA1[15],10000000b
  jne @@SetAlarm
  mov abTime1[0],AL
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "1 Uhrzeit Zehntelsekunden schreiben (%02X)",<AX>
    else
      INFO "1 write TOD clock 10ths of seconds (%02X)",<AX>
    endif
    cmp bTime1Running,0
    jne @@Running
    if GERMAN
      INFO2 "Uhr luft weiter"
    else
      INFO2 "The TOD clock starts to run again"
    endif
  @@Running:
    TEND
  endif
  mov bTime1Running,1
  jmp CX
align 4
@@SetAlarm:
  mov abAlarm1[0],AL
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "1 Alarmzeit Zehntelsekunden schreiben (%02X)",<AX>
    else
      INFO "1 write TOD alarm time 10ths of seconds (%02X)",<AX>
    endif
    TEND
  endif
  jmp CX
WriteCIA18 endp

;CIA1 Register 9

code 4
EXP ReadCIA19,near
ReadCIA19 proc near
  mov AL,abTime1[1]
  cmp bTime1Buffered,0
  je @@NoBuffer
  mov AL,CIA1[9]
@@NoBuffer:
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "1 Uhrzeit Sekunden lesen (%02X)",<AX>
    else
      INFO "1 read TOD clock seconds (%02X)",<AX>
    endif
    TEND
  endif
  jmp CX
ReadCIA19 endp

code 4
EXP WriteCIA19,near
WriteCIA19 proc near
  test CIA1[15],10000000b
  jne @@SetAlarm
  mov abTime1[1],AL
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "1 Uhrzeit Sekunden schreiben (%02X)",<AX>
    else
      INFO "1 write TOD clock seconds (%02X)",<AX>
    endif
    TEND
  endif
  jmp CX
align 4
@@SetAlarm:
  mov abAlarm1[1],AL
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "1 Alarmzeit Sekunden schreiben (%02X)",<AX>
    else
      INFO "1 write TOD alarm time seconds (%02X)",<AX>
    endif
    TEND
  endif
  jmp CX
WriteCIA19 endp

;CIA1 Register 10

code 4
EXP ReadCIA110,near
ReadCIA110 proc near
  mov AL,abTime1[2]
  cmp bTime1Buffered,0
  je @@NoBuffer
  mov AL,CIA1[10]
@@NoBuffer:
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "1 Uhrzeit Minuten lesen (%02X)",<AX>
    else
      INFO "1 read TOD clock minutes (%02X)",<AX>
    endif
    TEND
  endif
  jmp CX
ReadCIA110 endp

code 4
EXP WriteCIA110,near
WriteCIA110 proc near
  test CIA1[15],10000000b
  jne @@SetAlarm
  mov abTime1[2],AL
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "1 Uhrzeit Minuten schreiben (%02X)",<AX>
    else
      INFO "1 write TOD clock minutes (%02X)",<AX>
    endif
    TEND
  endif
  jmp CX
align 4
@@SetAlarm:
  mov abAlarm1[2],AL
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "1 Alarmzeit Minuten schreiben (%02X)",<AX>
    else
      INFO "1 write TOD alarm time minutes (%02X)",<AX>
    endif
    TEND
  endif
  jmp CX
WriteCIA110 endp

;CIA1 Register 11

code 4
EXP ReadCIA111,near
ReadCIA111 proc near
  mov EAX,dword ptr abTime1[0]
  mov dword ptr CIA1[8],EAX
  mov bTime1Buffered,1
  shr EAX,24
  ifdef DEBUG
    TBEG CIA
    if GERMAN
      INFO "1 Uhrzeit Stunden lesen (%02X)",<AX>
    else
      INFO "1 read TOD clock hours (%02X)",<AX>
    endif
    if GERMAN
      INFO2 "Restliche Register werden gepuffert"
    else
      INFO2 "Other TOD registers will be buffered"
    endif
    TEND
  endif
  jmp CX
ReadCIA111 endp

code 4
EXP WriteCIA111,near
WriteCIA111 proc near
  test CIA1[15],10000000b
  jne @@SetAlarm
  mov abTime1[3],AL
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "1 Uhrzeit Stunden schreiben (%02X)",<AX>
    else
      INFO "1 write TOD clock hours (%02X)",<AX>
    endif
    if GERMAN
      INFO2 "Uhr wird angehalten"
    else
      INFO2 "The TOD clock is stopped"
    endif
    TEND
  endif
  mov bTime1Running,0
  jmp CX
align 4
@@SetAlarm:
  mov abAlarm1[3],AL
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "1 Alarmzeit Stunden schreiben (%02X)",<AX>
    else
      INFO "1 write TOD alarm time hours (%02X)",<AX>
    endif
    TEND
  endif
  jmp CX
WriteCIA111 endp

;CIA1 Register 12

code 4
EXP ReadCIA112,near
ReadCIA112 proc near
  mov AL,CIA1[12]
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      WARN "1 Serial Data lesen (%02X) wird nicht untersttzt!",<AX>
    else
      WARN "1 Reading serial data (%02X) is not supported!",<AX>
    endif
    TEND
  endif
  jmp CX
ReadCIA112 endp

code 4
EXP WriteCIA112,near
WriteCIA112 proc near
  mov CIA1[12],AL
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "1 Serial Data schreiben (%02X)",<AX>
    else
      INFO "1 Writing serial data (%02X)",<AX>
    endif
    TEND
  endif
  test CIA1[14],01000000b
  je @@NoSDR
  cmp abSerialCounter[0],16
  jae @@NoSDR
  add abSerialCounter[0],16
@@NoSDR:
  jmp CX
WriteCIA112 endp

;CIA1 Register 13

code 4
EXP ReadCIA113,near
ReadCIA113 proc near
  mov AL,byte ptr CIAEvents[0]          ;Lesen
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "1 Interrupt Control lesen und zurcksetzen (%02X)",<AX>
    else
      INFO "1 read and reset Interrupt Control Register (%02X)",<AX>
    endif
    TEND
  endif
  mov byte ptr CIAEvents[0],0           ;Zurcksetzen
  and Events,not evIRQCIA               ;Interruptleitung an CPU zurcksetzen
  ifdef DEBUG
    TBEG CIA
    test AL,00000001b
    je @@NoBit0
    if GERMAN
      INFO2 "Bit 0 = 1: Unterlauf Timer A aufgetreten"
    else
      INFO2 "Bit 0 = 1: Timer A underflow occurred"
    endif
  @@NoBit0:
    test AL,00000010b
    je @@NoBit1
    if GERMAN
      INFO2 "Bit 1 = 1: Unterlauf Timer B aufgetreten"
    else
      INFO2 "Bit 1 = 1: Timer B underflow occurred"
    endif
  @@NoBit1:
    test AL,00000100b
    je @@NoBit2
    if GERMAN
      INFO2 "Bit 2 = 1: Eingestellte Alarmzeit wurde erreicht"
    else
      INFO2 "Bit 2 = 1: The TOD alarm went off"
    endif
  @@NoBit2:
    test AL,00001000b
    je @@NoBit3
    if GERMAN
      INFO2 "Bit 3 = 1: Schieberegister ist voll bzw. leer"
    else
      INFO2 "Bit 3 = 1: The shift register is full or empty"
    endif
  @@NoBit3:
    test AL,00010000b
    je @@NoBit4
    if GERMAN
      INFO2 "Bit 4 = 1: Signal am Pin FLAG aufgetreten"
    else
      INFO2 "Bit 4 = 1: There was a negative edge on the FLAG line"
    endif
  @@NoBit4:
    test AL,10000000b
    je @@NoBit7
    if GERMAN
      INFO2 "Bit 7 = 1: IRQ-Leitung war aktiv (pegelgetriggert)"
    else
      INFO2 "Bit 7 = 1: The IRQ line was active (level triggered)"
    endif
  @@NoBit7:
    TEND
  endif
  jmp CX
ReadCIA113 endp

code 4
EXP WriteCIA113,near
WriteCIA113 proc near
  mov BX,FS:[SI-3]
  cmp BL,99h
  je @@Reset
  cmp BL,9Dh
  je @@Reset
  cmp BH,01h
  jne @@NoReset
@@Reset:
  mov byte ptr CIAEvents[0],0           ;Zurcksetzen
  and Events,not evIRQCIA               ;Interruptleitung an CPU zurcksetzen
  ifdef DEBUG
    TBEG CIA
    if GERMAN
      INFO "1 Interrupt Control zurcksetzen (undokumentiert)"
    else
      INFO "1 reset Interrupt Control Register (undocumented)"
    endif
    TEND
  endif
@@NoReset:
  and AL,AL                             ;Setzen oder lschen?
  js @@Set
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "1 Interruptmaske lschen (%02X). Neuer Zustand:",<AX>
    else
      INFO "1 clear interrupt mask (%02X). New state:",<AX>
    endif
    TEND
  endif
  not AL                                ;Interruptmaske lschen
  and CIA1[13],AL
@@Show:
  ifdef DEBUG
    TBEG CIA
    mov AL,CIA1[13]
    test AL,00011111b
    jne @@SomeBits
    if GERMAN
      INFO2 "berhaupt keine Ereignisse zugelassen"
    else
      INFO2 "No CIA1 IRQ events allowed"
    endif
    jmp @@NoBit4
  @@SomeBits:
    test AL,00000001b
    je @@NoBit0
    if GERMAN
      INFO2 "Bit 0 = 1: IRQ beim Unterlauf von Timer A"
    else
      INFO2 "Bit 0 = 1: IRQ on Timer A underflow"
    endif
  @@NoBit0:
    test AL,00000010b
    je @@NoBit1
    if GERMAN
      INFO2 "Bit 1 = 1: IRQ beim Unterlauf von Timer B"
    else
      INFO2 "Bit 1 = 1: IRQ on Timer B underflow"
    endif
  @@NoBit1:
    test AL,00000100b
    je @@NoBit2
    if GERMAN
      INFO2 "Bit 2 = 1: IRQ beim Erreichen der eingestellten Alarmzeit"
    else
      INFO2 "Bit 2 = 1: IRQ on TOD clock alarm"
    endif
  @@NoBit2:
    test AL,00001000b
    je @@NoBit3
    if GERMAN
      WARN2 "Bit 3 = 1: Serial Data wird nicht untersttzt!"
    else
      WARN2 "Bit 3 = 1: The shift register is not supported!"
    endif
  @@NoBit3:
    test AL,00010000b
    je @@NoBit4
    if GERMAN
      WARN2 "Bit 4 = 1: FLAG-Anschlu wird nicht untersttzt!"
    else
      WARN2 "Bit 4 = 1: The FLAG signal is not supported!"
    endif
  @@NoBit4:
    test AL,01100000b
    je @@NoBits56
    if GERMAN
      WARN2 "Bits 5 und 6 sollten immer 0 sein!"
    else
      WARN2 "Bits 5 and 6 should always be 0!"
    endif
  @@NoBits56:
    TEND
  endif
  mov AL,CIA1[13]                       ;IRQ-Leitung fr CPU setzen/lschen
  test AL,byte ptr CIAEvents[0]
  je @@CheckClear
  ifdef DEBUG
    TBEG CIA
      test Events,evIRQCIA
      jne @@Quiet
      if GERMAN
        INFO2 "Bedingung fr IRQ erfllt, Interruptleitung wurde gesetzt"
      else
        INFO2 "IRQ condition occurred, interrupt line was set"
      endif
    @@Quiet:
    TEND
  endif
  or Events,evIRQCIA
  jmp CX
align 4
@@CheckClear:
  ifdef DEBUG
    TBEG CIA
      test Events,evIRQCIA
      je @@Quiet
      if GERMAN
        INFO2 "Interruptleitung wurde zurckgesetzt"
      else
        INFO2 "Interrupt line was reset"
      endif
    @@Quiet:
    TEND
  endif
  and Events,not evIRQCIA               ;Interruptleitung an CPU zurcksetzen
  jmp CX
align 4
@@Set:
  or CIA1[13],AL                        ;Interruptmaske setzen
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "1 Interruptmaske setzen (%02X). Neuer Zustand:",<AX>
    else
      INFO "1 Set IRQ mask (%02X). New state:",<AX>
    endif
    TEND
  endif
  jmp @@Show
WriteCIA113 endp

;CIA1 Register 14

code 4
EXP ReadCIA114,near
ReadCIA114 proc near
  mov AL,CIA1[14]
  and AL,11101111b
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "1 Control A lesen (%02X)",<AX>
    else
      INFO "1 read Control Register A (%02X)",<AX>
    endif
  CIAAChanged label near
    test AL,00000001b
    jne @@Bit0
    if GERMAN
      INFO2 "Bit 0 = 0: Timer A angehalten"
    else
      INFO2 "Bit 0 = 0: Timer A stopped"
    endif
    jmp @@NoBit0
  @@Bit0:
    if GERMAN
      INFO2 "Bit 0 = 1: Timer A luft"
    else
      INFO2 "Bit 0 = 1: Timer A is running"
    endif
  @@NoBit0:
    test AL,00000010b
    je @@NoBit1
    if GERMAN
      WARN2 "Bit 1 = 1: Anschlu PB6 wird nicht untersttzt!"
    else
      WARN2 "Bit 1 = 1: The PB6 pin is not supported!"
    endif
  @@NoBit1:
    test AL,00001000b
    je @@NoBit3
    if GERMAN
      INFO2 "Bit 3 = 1: Timer A hlt nach dem ersten Unterlauf an (Single Shot)"
    else
      INFO2 "Bit 3 = 1: Timer A stops on underflow (Single Shot mode)"
    endif
  @@NoBit3:
    test AL,00010000b
    je @@NoBit4
    if GERMAN
      INFO2 "Bit 4 = 1: Timer A wird auf Startwert zurckgesetzt"
    else
      INFO2 "Bit 4 = 1: Timer A will be loaded from start value (Force load)"
    endif
  @@NoBit4:
    test AL,00100000b
    je @@NoBit5
    if GERMAN
      WARN2 "Bit 5 = 1: Anschlu CNT wird nicht untersttzt!"
    else
      WARN2 "Bit 5 = 1: The CNT pin is not supported!"
    endif
  @@NoBit5:
    test AL,01000000b
    je @@NoBit6
    if GERMAN
      INFO2 "Bit 6 = 1: SDR ist Ausgang"
    else
      INFO2 "Bit 6 = 1: SDR is output"
    endif
  @@NoBit6:
    test AL,10000000b
    jne @@Bit7
    if GERMAN
      INFO2 "Bit 7 = 0: Echtzeituhr luft mit 60 Hertz (NTSC-M)"
    else
      INFO2 "Bit 7 = 0: TOD clock runs at 60 hertz (NTSC-M)"
    endif
    jmp @@NoBit7
  @@Bit7:
    if GERMAN
      INFO2 "Bit 7 = 1: Echtzeituhr luft mit 50 Hertz (PAL-B)"
    else
      INFO2 "Bit 7 = 1: TOD clock runs at 50 Hertz (PAL-B)"
    endif
  @@NoBit7:
    TEND
  endif
  jmp CX
ReadCIA114 endp

code 4
EXP WriteCIA114,near
WriteCIA114 proc near
  mov CIA1[14],AL
  test AL,00010000b
  je @@DontReset
  mov DI,word ptr CIA1[4]
  mov Timer1A,DI
@@DontReset:
  push EAX
  shr AX,5
  and AX,0000000000000100b
  mov EAX,adTimeAdd[EAX]
  mov dTime1Add,EAX
  pop EAX
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "1 Control A schreiben (%02X)",<AX>
    else
      INFO "1 write Control Register A (%02X)",<AX>
    endif
    jmp CIAAChanged
    TEND
  endif
  jmp CX
WriteCIA114 endp

;CIA1 Register 15

code 4
EXP ReadCIA115,near
ReadCIA115 proc near
  mov AL,CIA1[15]
  and AL,11101111b
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "1 Control B lesen (%02X)",<AX>
    else
      INFO "1 read Control Register B (%02X)",<AX>
    endif
  CIABChanged label near
    test AL,00000001b
    jne @@Bit0
    if GERMAN
      INFO2 "Bit 0 = 0: Timer B angehalten"
    else
      INFO2 "Bit 0 = 0: Timer B stopped"
    endif
    jmp @@NoBit0
  @@Bit0:
    if GERMAN
      INFO2 "Bit 0 = 1: Timer B luft"
    else
      INFO2 "Bit 0 = 1: Timer B is running"
    endif
  @@NoBit0:
    test AL,00000010b
    je @@NoBit1
    if GERMAN
      WARN2 "Bit 1 = 1: Anschlu PB7 wird nicht untersttzt!"
    else
      WARN2 "Bit 1 = 1: The PB7 pin is not supported!"
    endif
  @@NoBit1:
    test AL,00001000b
    je @@NoBit3
    if GERMAN
      INFO2 "Bit 3 = 1: Timer B hlt nach dem ersten Unterlauf an (Single Shot)"
    else
      INFO2 "Bit 3 = 1: Timer B stops on underflow (Single Shot mode)"
    endif
  @@NoBit3:
    test AL,00010000b
    je @@NoBit4
    if GERMAN
      INFO2 "Bit 4 = 1: Timer B wird auf Startwert zurckgesetzt"
    else
      INFO2 "Bit 4 = 1: Timer B will be loaded from start value (Force load)"
    endif
  @@NoBit4:
    test AL,00100000b
    je @@NoBit5
    if GERMAN
      WARN2 "Bit 5 = 1: Anschlu CNT wird nicht untersttzt!"
    else
      WARN2 "Bit 5 = 1: The CNT pin is not supported!"
    endif
  @@NoBit5:
    test AL,01000000b
    je @@NoBit6
    if GERMAN
      INFO2 "Bit 6 = 1: Timer B zhlt Unterlufe von Timer A (Kaskadierung)"
    else
      INFO2 "Bit 6 = 1: Timer B counts Timer A underflows (Cascaded operation)"
    endif
  @@NoBit6:
    test AL,10000000b
    je @@NoBit7
    if GERMAN
      INFO2 "Bit 7 = 1: Alarmzeit setzen"
    else
      INFO2 "Bit 7 = 1: Set TOD clock alarm time"
    endif
  @@NoBit7:
    TEND
  endif
  jmp CX
ReadCIA115 endp

code 4
EXP WriteCIA115,near
WriteCIA115 proc near
  mov CIA1[15],AL
  test AL,00010000b
  je @@DontReset
  mov DI,word ptr CIA1[6]
  mov Timer1B,DI
@@DontReset:
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "1 Control B schreiben (%02X)",<AX>
    else
      INFO "1 write Control Register B (%02X)",<AX>
    endif
    jmp CIABChanged
    TEND
  endif
  jmp CX
WriteCIA115 endp

;CIA2 Register 0

code 4
EXP ReadCIA20,near
ReadCIA20 proc near
  cmp UIReadDD00,0
  jne UI
  mov AL,CIA2[2]                        ;Mit Richtungsregister verknpfen
  not AL
  or AL,CIA2[0]
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "2 Daten Port A lesen (%02X)",<AX>
    else
      INFO "2 read Port A data (%02X)",<AX>
    endif
    if GERMAN
      WARN2 "Bit 2 Userport und Bit 3-7 serieller Bus werden nicht untersttzt!"
    else
      WARN2 "Bit 2 (Userport PA2) and bits 3-7 (serial bus) are not supported!"
    endif
    TEND
  endif
  jmp CX
UI:
  call UIReadDD00
  mov AH,CIA2[2]                        ;Mit Richtungsregister verknpfen
  not AH
  or AH,CIA2[0]
  and AX,0000001111111100b
  or AL,AH
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "2 Daten Port A ber Userport Interface lesen (%02X)",<AX>
    else
      INFO "2 read Port A data via the Userport interface (%02X)",<AX>
    endif
    TEND
  endif
  jmp CX
ReadCIA20 endp

code 4
EXP WriteCIA20,near
WriteCIA20 proc near
  mov CIA2[0],AL
  cmp UIWriteDD00,0
  je NoUI
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "2 Daten Port A ber Userport Interface schreiben (%02X)",<AX>
    else
      INFO "2 write Port A data via the Userport interface (%02X)",<AX>
    endif
    TEND
  endif
  call UIWriteDD00
  ifdef DEBUG
    jmp UIEnd
  endif
NoUI:
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "2 Daten Port A schreiben (%02X)",<AX>
    else
      INFO "2 write Port A data (%02X)",<AX>
    endif
    if GERMAN
      WARN2 "Bit 2 Userport und Bit 3-7 serieller Bus werden nicht untersttzt!"
    else
      WARN2 "Bit 2 (Userport PA2) and bits 3-7 (serial bus) are not supported!"
    endif
    TEND
  UIEnd:
  endif
  mov AL,CIA2[2]                        ;Mit Richtungsregister verknpfen
CIA2AChanged label near
  not AL
  or AL,CIA2[0]
  xor AL,00000011b                      ;Basisadresse Videoram berechnen
  shl AX,10
  and AH,00001100b
  cmp AX,VICBasePara
  je SameBase
  mov VICBasePara,AX
  ifdef DEBUG
    TBEG VIC
    if GERMAN
      INFO "Basisadresse Videoram wurde gendert"
    else
      INFO "Video memory base address was changed"
    endif
    TEND
    push CX
    mov CX,offset ShowVICConfig
  endif
  jmp BaseChanged
SameBase:
  jmp CX
WriteCIA20 endp

;CIA2 Register 1

code 4
EXP ReadCIA21,near
ReadCIA21 proc near
  cmp UIReadDD01,0
  jne UI
  mov AL,CIA2[3]                        ;Mit Richtungsregister verknpfen
  not AL
  or AL,CIA2[1]
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "2 Daten Port B lesen (%02X)",<AX>
    else
      INFO "2 read Port B data (%02X)",<AX>
    endif
    if GERMAN
      WARN2 "Userport wird nicht untersttzt!"
    else
      WARN2 "Userport is not supported!"
    endif
    TEND
  endif
  jmp CX
UI:
  call UIReadDD01
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "2 Daten Port B ber Userport Interface lesen (%02X)",<AX>
    else
      INFO "2 read Port B data via the Userport interface (%02X)",<AX>
    endif
    TEND
  endif
  jmp CX
ReadCIA21 endp

code 4
EXP WriteCIA21,near
WriteCIA21 proc near
  mov CIA2[1],AL
  cmp UIWriteDD01,0
  je NoUI
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "2 Daten Port B ber Userport Interface schreiben (%02X)",<AX>
    else
      INFO "2 write Port B data via the Userport interface (%02X)",<AX>
    endif
    TEND
  endif
  call UIWriteDD01
  jmp CX
NoUI:
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN ;fuck MS
      INFO "2 Daten Port B schreiben (%02X)",<AX>
    else
      INFO "2 write Port B data (%02X)",<AX>
    endif
    if GERMAN
      WARN2 "Userport wird nicht untersttzt!"
    else
      WARN2 "Userport is not supported!"
    endif
    TEND
  endif
  jmp CX
WriteCIA21 endp

;CIA2 Register 2

code 4
EXP ReadCIA22,near
ReadCIA22 proc near
  mov AL,CIA2[2]
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "2 Richtung Port A lesen (%02X)",<AX>
    else
      INFO "2 read Port A data direction (%02X)",<AX>
    endif
    TEND
  endif
  jmp CX
ReadCIA22 endp

code 4
EXP WriteCIA22,near
WriteCIA22 proc near
  mov CIA2[2],AL
  cmp UIWriteDD02,0
  je NoUI
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "2 Richtung Port A ber Userport Interface schreiben (%02X)",<AX>
    else
      INFO "2 write Port A data direction via the Userport interface (%02X)",<AX>
    endif
    TEND
  endif
  call UIWriteDD02
  mov AL,CIA2[2]
  jmp CIA2AChanged
NoUI:
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "2 Richtung Port A schreiben (%02X)",<AX>
    else
      INFO "2 write Port A data direction (%02X)",<AX>
    endif
    TEND
  endif
  jmp CIA2AChanged
WriteCIA22 endp

;CIA2 Register 3

code 4
EXP ReadCIA23,near
ReadCIA23 proc near
  mov AL,CIA2[3]
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "2 Richtung Port B lesen (%02X)",<AX>
    else
      INFO "2 read Port B data direction (%02X)",<AX>
    endif
    cmp UIWriteDD03,0
    jne NoWarn
    if GERMAN
      WARN2 "Userport wird nicht untersttzt!"
    else
      WARN2 "Userport is not supported!"
    endif
  NoWarn:
    TEND
  endif
  jmp CX
ReadCIA23 endp

code 4
EXP WriteCIA23,near
WriteCIA23 proc near
  mov CIA2[3],AL
  cmp UIWriteDD03,0
  je NoUI
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "2 Richtung Port B ber Userport Interface schreiben (%02X)",<AX>
    else
      INFO "2 write Port B data direction via the Userport interface (%02X)",<AX>
    endif
    TEND
  endif
  call UIWriteDD03
  jmp CX
NoUI:
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "2 Richtung Port B schreiben (%02X)",<AX>
    else
      INFO "2 write Port B data direction (%02X)",<AX>
    endif
    if GERMAN
      WARN2 "Userport wird nicht untersttzt!"
    else
      WARN2 "Userport is not supported!"
    endif
    TEND
  endif
  jmp CX
WriteCIA23 endp

;CIA2 Register 4

code 4
EXP ReadCIA24,near
ReadCIA24 proc near
  mov AX,Timer2A
  ifdef DEBUG
    TBEG CIA
    if GERMAN
      INFO "2 Low Byte Zhler A lesen (%04X)",<AX>
    else
      INFO "2 read Timer A low byte (%04X)",<AX>
    endif
    TEND
  endif
  jmp CX
ReadCIA24 endp

code 4
EXP WriteCIA24,near
WriteCIA24 proc near
  mov CIA2[4],AL
  ifdef DEBUG
    TBEG CIA
    mov AX,word ptr CIA2[4]
    if GERMAN
      INFO "2 Low Byte Zhler A schreiben (%04X)",<AX>
    else
      INFO "2 write Timer A low byte (%04X)",<AX>
    endif
    TEND
  endif
  jmp CX
WriteCIA24 endp

;CIA2 Register 5

code 4
EXP ReadCIA25,near
ReadCIA25 proc near
  ifdef DEBUG
    TBEG CIA
    mov AX,Timer2A
    if GERMAN
      INFO "2 High Byte Zhler A lesen (%04X)",<AX>
    else
      INFO "2 read Timer A high byte (%04X)",<AX>
    endif
    TEND
  endif
  mov AL,byte ptr Timer2A[1]
  jmp CX
ReadCIA25 endp

code 4
EXP WriteCIA25,near
WriteCIA25 proc near
  mov CIA2[5],AL
  ifdef DEBUG
    TBEG CIA
    mov AX,word ptr CIA2[4]
    if GERMAN
      INFO "2 High Byte Zhler A schreiben (%04X)",<AX>
    else
      INFO "2 write Timer A high byte (%04X)",<AX>
    endif
    TEND
  endif
  test CIA2[14],00000001b               ;Zhler angehalten?
  jne @@Running
  mov AX,word ptr CIA2[4]               ;Ja, dann Wert bernehmen
  mov Timer2A,AX
@@Running:
  jmp CX
WriteCIA25 endp

;CIA2 Register 6

code 4
EXP ReadCIA26,near
ReadCIA26 proc near
  mov AX,Timer2B
  ifdef DEBUG
    TBEG CIA
    if GERMAN
      INFO "2 Low Byte Zhler B lesen (%04X)",<AX>
    else
      INFO "2 read Timer B low byte (%04X)",<AX>
    endif
    TEND
  endif
  jmp CX
ReadCIA26 endp

code 4
EXP WriteCIA26,near
WriteCIA26 proc near
  mov CIA2[6],AL
  ifdef DEBUG
    TBEG CIA
    mov AX,word ptr CIA2[6]
    if GERMAN
      INFO "2 Low Byte Zhler B schreiben (%04X)",<AX>
    else
      INFO "2 write Timer B low byte (%04X)",<AX>
    endif
    TEND
  endif
  jmp CX
WriteCIA26 endp

;CIA2 Register 7

code 4
EXP ReadCIA27,near
ReadCIA27 proc near
  ifdef DEBUG
    TBEG CIA
    mov AX,Timer2B
    if GERMAN
      INFO "2 High Byte Zhler B lesen (%04X)",<AX>
    else
      INFO "2 read Timer B high byte (%04X)",<AX>
    endif
    TEND
  endif
  mov AL,byte ptr Timer2B[1]
  jmp CX
ReadCIA27 endp

code 4
EXP WriteCIA27,near
WriteCIA27 proc near
  mov CIA2[7],AL
  ifdef DEBUG
    TBEG CIA
    mov AX,word ptr CIA2[6]
    if GERMAN
      INFO "2 High Byte Zhler B schreiben (%04X)",<AX>
    else
      INFO "2 write Timer B high byte (%04X)",<AX>
    endif
    TEND
  endif
  test CIA2[15],00000001b               ;Zhler angehalten?
  jne @@Running
  mov AX,word ptr CIA2[6]               ;Ja, dann Wert bernehmen
  mov Timer2B,AX
@@Running:
  jmp CX
WriteCIA27 endp

;CIA2 Register 8

code 4
EXP ReadCIA28,near
ReadCIA28 proc near
  mov AL,abTime2[0]
  cmp bTime2Buffered,0
  je @@NoBuffer
  mov AL,CIA2[8]
  ifdef DEBUG
  @@NoBuffer:
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "2 Uhrzeit Zehntelsekunden lesen (%02X)",<AX>
    else
      INFO "2 read TOD clock 10ths of seconds (%02X)",<AX>
    endif
    cmp bTime2Buffered,0
    je @@NoBuffer1
    mov bTime2Buffered,0
    if GERMAN
      INFO2 "Pufferung wird aufgehoben"
    else
      INFO2 "TOD clock buffering cancelled"
    endif
  @@NoBuffer1:
    TEND
  else
    mov bTime2Buffered,0
  @@NoBuffer:
  endif
  jmp CX
ReadCIA28 endp

code 4
EXP WriteCIA28,near
WriteCIA28 proc near
  test CIA2[15],10000000b
  jne @@SetAlarm
  mov abTime2[0],AL
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "2 Uhrzeit Zehntelsekunden schreiben (%02X)",<AX>
    else
      INFO "2 write TOD clock 10ths of seconds (%02X)",<AX>
    endif
    cmp bTime2Running,0
    jne @@Running
    if GERMAN
      INFO2 "Uhr luft weiter"
    else
      INFO2 "The TOD clock starts to run again"
    endif
  @@Running:
    TEND
  endif
  mov bTime2Running,1
  jmp CX
align 4
@@SetAlarm:
  mov abAlarm2[0],AL
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "2 Alarmzeit Zehntelsekunden schreiben (%02X)",<AX>
    else
      INFO "2 write TOD alarm time 10ths of seconds (%02X)",<AX>
    endif
    TEND
  endif
  jmp CX
WriteCIA28 endp

;CIA2 Register 9

code 4
EXP ReadCIA29,near
ReadCIA29 proc near
  mov AL,abTime2[1]
  cmp bTime2Buffered,0
  je @@NoBuffer
  mov AL,CIA2[9]
@@NoBuffer:
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "2 Uhrzeit Sekunden lesen (%02X)",<AX>
    else
      INFO "2 read TOD clock seconds (%02X)",<AX>
    endif
    TEND
  endif
  jmp CX
ReadCIA29 endp

code 4
EXP WriteCIA29,near
WriteCIA29 proc near
  test CIA2[15],10000000b
  jne @@SetAlarm
  mov abTime2[1],AL
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "2 Uhrzeit Sekunden schreiben (%02X)",<AX>
    else
      INFO "2 write TOD clock seconds (%02X)",<AX>
    endif
    TEND
  endif
  jmp CX
align 4
@@SetAlarm:
  mov abAlarm2[1],AL
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "2 Alarmzeit Sekunden schreiben (%02X)",<AX>
    else
      INFO "2 write TOD alarm time seconds (%02X)",<AX>
    endif
    TEND
  endif
  jmp CX
WriteCIA29 endp

;CIA2 Register 10

code 4
EXP ReadCIA210,near
ReadCIA210 proc near
  mov AL,abTime2[2]
  cmp bTime2Buffered,0
  je @@NoBuffer
  mov AL,CIA2[10]
@@NoBuffer:
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "2 Uhrzeit Minuten lesen (%02X)",<AX>
    else
      INFO "2 read TOD clock minutes (%02X)",<AX>
    endif
    TEND
  endif
  jmp CX
ReadCIA210 endp

code 4
EXP WriteCIA210,near
WriteCIA210 proc near
  test CIA2[15],10000000b
  jne @@SetAlarm
  mov abTime2[2],AL
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "2 Uhrzeit Minuten schreiben (%02X)",<AX>
    else
      INFO "2 write TOD clock minutes (%02X)",<AX>
    endif
    TEND
  endif
  jmp CX
align 4
@@SetAlarm:
  mov abAlarm2[2],AL
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "2 Alarmzeit Minuten schreiben (%02X)",<AX>
    else
      INFO "2 write TOD alarm time minutes (%02X)",<AX>
    endif
    TEND
  endif
  jmp CX
WriteCIA210 endp

;CIA2 Register 11

code 4
EXP ReadCIA211,near
ReadCIA211 proc near
  mov EAX,dword ptr abTime2[0]
  mov dword ptr CIA2[8],EAX
  mov bTime2Buffered,1
  shr EAX,24
  ifdef DEBUG
    TBEG CIA
    if GERMAN
      INFO "2 Uhrzeit Stunden lesen (%02X)",<AX>
    else
      INFO "2 read TOD clock hours (%02X)",<AX>
    endif
    if GERMAN
      INFO2 "Restliche Register werden gepuffert"
    else
      INFO2 "Other TOD registers will be buffered"
    endif
    TEND
  endif
  jmp CX
ReadCIA211 endp

code 4
EXP WriteCIA211,near
WriteCIA211 proc near
  test CIA2[15],10000000b
  jne @@SetAlarm
  mov abTime2[3],AL
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN ;remove this comment and you get cia.asm(3045): error A2039: line too long
      INFO "2 Uhrzeit Stunden schreiben (%02X)",<AX>
    else
      INFO "2 write TOD clock hours (%02X)",<AX>
    endif
    if GERMAN
      INFO2 "Uhr wird angehalten"
    else
      INFO2 "The TOD clock is stopped"
    endif
    TEND
  endif
  mov bTime2Running,0
  jmp CX
align 4
@@SetAlarm:
  mov abAlarm2[3],AL
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "2 Alarmzeit Stunden schreiben (%02X)",<AX>
    else
      INFO "2 write TOD alarm time hours (%02X)",<AX>
    endif
    TEND
  endif
  jmp CX
WriteCIA211 endp

;CIA2 Register 12

code 4
EXP ReadCIA212,near
ReadCIA212 proc near
  mov AL,CIA2[12]
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      WARN "2 Serial Data lesen (%02X) wird nicht untersttzt!",<AX>
    else
      WARN "2 Reading serial data (%02X) is not supported!",<AX>
    endif
    TEND
  endif
  jmp CX
ReadCIA212 endp

code 4
EXP WriteCIA212,near
WriteCIA212 proc near
  mov CIA2[12],AL
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "2 Serial Data schreiben (%02X)",<AX>
    else
      INFO "2 Writing serial data (%02X)",<AX>
    endif
    TEND
  endif
  test CIA2[14],01000000b
  je @@NoSDR
  cmp abSerialCounter[1],16
  jae @@NoSDR
  add abSerialCounter[1],16
@@NoSDR:
  jmp CX
WriteCIA212 endp

;CIA2 Register 13

code 4
EXP ReadCIA213,near
ReadCIA213 proc near
  mov AL,byte ptr CIAEvents[1]
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "2 Interrupt Control lesen und zurcksetzen (%02X)",<AX>
    else
      INFO "2 read and reset Interrupt Control Register (%02X)",<AX>
    endif
    TEND
  endif
  mov byte ptr CIAEvents[1],0           ;Zurcksetzen
  ifdef DEBUG
    TBEG CIA
    test AL,00000001b
    je @@NoBit0
    if GERMAN
      INFO2 "Bit 0 = 1: Unterlauf Timer A aufgetreten"
    else
      INFO2 "Bit 0 = 1: Timer A underflow occurred"
    endif
  @@NoBit0:
    test AL,00000010b
    je @@NoBit1
    if GERMAN
      INFO2 "Bit 1 = 1: Unterlauf Timer B aufgetreten"
    else
      INFO2 "Bit 1 = 1: Timer B underflow occurred"
    endif
  @@NoBit1:
    test AL,00000100b
    je @@NoBit2
    if GERMAN
      INFO2 "Bit 2 = 1: Eingestellte Alarmzeit wurde erreicht"
    else
      INFO2 "Bit 2 = 1: The TOD alarm went off"
    endif
  @@NoBit2:
    test AL,00001000b
    je @@NoBit3
    if GERMAN
      INFO2 "Bit 3 = 1: Schieberegister ist voll bzw. leer"
    else
      INFO2 "Bit 3 = 1: The shift register is full or empty"
    endif
  @@NoBit3:
    test AL,00010000b
    je @@NoBit4
    if GERMAN
      INFO2 "Bit 4 = 1: Signal am Pin FLAG aufgetreten"
    else
      INFO2 "Bit 4 = 1: There was a negative edge on the FLAG line"
    endif
  @@NoBit4:
    test AL,10000000b
    je @@NoBit7
    if GERMAN
      INFO2 "Bit 7 = 1: NMI-Leitung war aktiv (flankengetriggert)"
    else
      INFO2 "Bit 7 = 1: The NMI line was active (triggered on falling edge)"
    endif
  @@NoBit7:
    TEND
  endif
  jmp CX
ReadCIA213 endp

code 4
EXP WriteCIA213,near
WriteCIA213 proc near
  mov BX,FS:[SI-3]
  cmp BL,99h
  je @@Reset
  cmp BL,9Dh
  je @@Reset
  cmp BH,01h
  jne @@NoReset
@@Reset:
  mov byte ptr CIAEvents[1],0           ;Zurcksetzen
  ifdef DEBUG
    TBEG CIA
    if GERMAN
      INFO "2 Interrupt Control zurcksetzen (undokumentiert)"
    else
      INFO "2 reset Interrupt Control Register (undocumented)"
    endif
    TEND
  endif
@@NoReset:
  mov AH,CIA2[13]
  push AX
  and AL,AL                             ;Setzen oder lschen?
  js @@Set
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "2 Interruptmaske lschen (%02X). Neuer Zustand:",<AX>
    else
      INFO "2 Clear interrupt mask (%02X). New state:",<AX>
    endif
    TEND
  endif
  not AL                                ;Interruptmaske lschen
  and CIA2[13],AL
@@Show:
  ifdef DEBUG
    TBEG CIA
    mov AL,CIA2[13]
    test AL,00011111b
    jne @@SomeBits
    if GERMAN
      INFO2 "berhaupt keine Ereignisse zugelassen"
    else
      INFO2 "No CIA2 NMI events allowed"
    endif
    jmp @@NoBit4
  @@SomeBits:
    test AL,00000001b
    je @@NoBit0
    if GERMAN
      INFO2 "Bit 0 = 1: NMI beim Unterlauf von Timer A"
    else
      INFO2 "Bit 0 = 1: NMI on Timer A underflow"
    endif
  @@NoBit0:
    test AL,00000010b
    je @@NoBit1
    if GERMAN
      INFO2 "Bit 1 = 1: NMI on Timer B underflow"
    else
      INFO2 "Bit 1 = 1: NMI beim Unterlauf von Timer B"
    endif
  @@NoBit1:
    test AL,00000100b
    je @@NoBit2
    if GERMAN
      INFO2 "Bit 2 = 1: NMI beim Erreichen der eingestellten Alarmzeit"
    else
      INFO2 "Bit 2 = 1: NMI on TOD clock alarm"
    endif
  @@NoBit2:
    test AL,00001000b
    je @@NoBit3
    if GERMAN
      WARN2 "Bit 3 = 1: Serial Data wird nicht untersttzt!"
    else
      WARN2 "Bit 3 = 1: The shift register is not supported!"
    endif
  @@NoBit3:
    test AL,00010000b
    je @@NoBit4
    if GERMAN
      WARN2 "Bit 4 = 1: FLAG-Anschlu wird nicht untersttzt!"
    else
      WARN2 "Bit 4 = 1: The FLAG signal is not supported!"
    endif
  @@NoBit4:
    test AL,01100000b
    je @@NoBits56
    if GERMAN
      WARN2 "Bits 5 und 6 sollten immer 0 sein!"
    else
      WARN2 "Bits 5 and 6 should always be 0!"
    endif
  @@NoBits56:
    TEND
  endif
  pop AX
  mov AL,CIA2[13]                       ;NMI fr CPU auslsen?
  test AL,byte ptr CIAEvents[1]
  je @@NoNMI
  test AH,byte ptr CIAEvents[1]         ;lag NMI schon an? (flankengetriggert)
  jne @@NoNMI
  ifdef DEBUG
    TBEG CIA
      if GERMAN
        INFO2 "Interruptbedingung erfllt, NMI wurde ausgelst"
      else
        INFO2 "Interrupt condition occurred, NMI has been triggered"
      endif
    TEND
  endif
  or Events,evNMI
@@NoNMI:
  jmp CX
align 4
@@Set:
  or CIA2[13],AL                        ;Interruptmaske setzen
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "2 Interruptmaske setzen (%02X). Neuer Zustand:",<AX>
    else
      INFO "2 Set interrupt mask (%02X). New state:",<AX>
    endif
    TEND
  endif
  jmp @@Show
WriteCIA213 endp

;CIA2 Register 14

code 4
EXP ReadCIA214,near
ReadCIA214 proc near
  mov AL,CIA2[14]
  and AL,11101111b
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "2 Control A lesen (%02X)",<AX>
    else
      INFO "2 read Control Register A (%02X)",<AX>
    endif
    jmp CIAAChanged
    TEND
  endif
  jmp CX
ReadCIA214 endp

code 4
EXP WriteCIA214,near
WriteCIA214 proc near
  mov CIA2[14],AL
  test AL,00010000b
  je @@DontReset
  mov DI,word ptr CIA2[4]
  mov Timer2A,DI
@@DontReset:
  push EAX
  shr AX,5
  and AX,0000000000000100b
  mov EAX,adTimeAdd[EAX]
  mov dTime2Add,EAX
  pop EAX
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "2 Control A schreiben (%02X)",<AX>
    else
      INFO "2 write Control Register A (%02X)",<AX>
    endif
    jmp CIAAChanged
    TEND
  endif
  jmp CX
WriteCIA214 endp

;CIA2 Register 15

code 4
EXP ReadCIA215,near
ReadCIA215 proc near
  mov AL,CIA2[15]
  and AL,11101111b
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "2 Control B lesen (%02X)",<AX>
    else
      INFO "2 read Control Register B (%02X)",<AX>
    endif
    jmp CIABChanged
    TEND
  endif
  jmp CX
ReadCIA215 endp

code 4
EXP WriteCIA215,near
WriteCIA215 proc near
  mov CIA2[15],AL
  test AL,00010000b
  je @@DontReset
  mov DI,word ptr CIA2[6]
  mov Timer2B,DI
@@DontReset:
  ifdef DEBUG
    TBEG CIA
    xor AH,AH
    if GERMAN
      INFO "2 Control B schreiben (%02X)",<AX>
    else
      INFO "2 write Control Register B (%02X)",<AX>
    endif
    jmp CIABChanged
    TEND
  endif
  jmp CX
WriteCIA215 endp

;BCD-Zeit in EAX um 1/10 sec erhhen
code 1
abIncTab label byte
  db 01h,02h,03h,04h,05h,06h,07h,08h,09h,10h,0Bh,0Ch,0Dh,0Eh,0Fh,00h
  db 11h,12h,13h,14h,15h,16h,17h,18h,19h,20h,1Bh,1Ch,1Dh,1Eh,1Fh,10h
  db 21h,22h,23h,24h,25h,26h,27h,28h,29h,30h,2Bh,2Ch,2Dh,2Eh,2Fh,20h
  db 31h,32h,33h,34h,35h,36h,37h,38h,39h,40h,3Bh,3Ch,3Dh,3Eh,3Fh,30h
  db 41h,42h,43h,44h,45h,46h,47h,48h,49h,50h,4Bh,4Ch,4Dh,4Eh,4Fh,40h
  db 51h,52h,53h,54h,55h,56h,57h,58h,59h,00h,5Bh,5Ch,5Dh,5Eh,5Fh,50h
  db 61h,62h,63h,64h,65h,66h,67h,68h,69h,70h,6Bh,6Ch,6Dh,6Eh,6Fh,60h
  db 71h,72h,73h,74h,75h,76h,77h,78h,79h,00h,7Bh,7Ch,7Dh,7Eh,7Fh,70h
abHour label byte
  db 01h,02h,03h,04h,05h,06h,07h,08h,09h,10h,0Bh,0Ch,0Dh,0Eh,0Fh,00h
  db 11h,92h,01h,14h,15h,16h,17h,18h,19h,1Ah,0Bh,0Ch,0Dh,0Eh,0Fh,00h
code 4
IncTime proc near
  cmp AL,09h
  je @@IncSec
  inc AL
  and AL,00001111b
  jmp @@Return
@@IncSec:
  xor AL,AL                             ;Sekunden erhhen
  mov BL,AH
  xor BH,BH
  mov AH,abIncTab[BX]
  and AH,AH
  jne @@Return
  ror EAX,16                            ;Minuten erhhen
  mov BL,AL
  mov AL,abIncTab[BX]
  and AL,AL
  jne @@RetNRoll
  mov BL,AH                             ;Stunden erhhen
  and BL,00011111b
  and AH,10000000b
  xor AH,abHour[BX]
@@RetNRoll:
  rol EAX,16
@@Return:
  ret
IncTime endp

;Periodische Arbeiten alle 64us
code 4
EXP UpdateCIA,near
UpdateCIA proc near
  xor CX,CX                             ;Ereignismasken fr Interrupts
  mov BX,word ptr CIA1[14]              ;Luft Timer 1A?
  test BL,00000001b
  je @@Check1B
  mov AX,Timer1A                        ;Timer 1A weiterzhlen
  mov DI,awUpdateIndex[1*2]
  sub AX,awUpdate[DI]
  jnc @@Set1A
  ifdef DEBUG
    TBEG CIA
    if GERMAN
      INFO "1 Unterlauf Timer A aufgetreten"
    else
      INFO "1 Timer A underflow occurred"
    endif
    TEND
  endif
  mov CL,00000001b                      ;Unterlauf Timer 1A aufgetreten

  cmp abSerialCounter[0],0
  je @@NoSerial1
  cmp abSerialCounter[0],2
  jne @@NoEmpty1
  ifdef DEBUG
    TBEG INT
    if GERMAN
      INFO2 "Schieberegister wurde ausgegeben"
    else
      INFO2 "The shift register was shifted out"
    endif
    TEND
  endif
  or CL,00001000b
@@NoEmpty1:
  dec abSerialCounter[0]
@@NoSerial1:

  test BL,00001000b                     ;Timer im Single-Shot-Modus?
  jne @@Single1A
@@Add1A:
  stc                                   ;Timer 1A mit Startwert laden
  adc AX,word ptr CIA1[4]
  jc @@Set1A
  jmp @@Add1A                           ;Wegen Rasterung wiederholen
@@Single1A:
  ifdef DEBUG
    TBEG CIA
    if GERMAN
      INFO2 "Single Shot Modus ist aktiviert, Timer A wird angehalten"
    else
      INFO2 "Single Shot mode active, Timer A stopped"
    endif
    TEND
  endif
  and BL,11111110b                      ;Timer anhalten
  mov CIA1[14],BL
  mov AX,word ptr CIA1[4]               ;Endwert wegen Rasterung setzen
@@Set1A:
  mov Timer1A,AX
@@Check1B:
  test BH,00000001b                     ;Luft Timer 1B?
  je @@Check2A
  mov AX,Timer1B
  test BH,01000000b                     ;Timer hintereinandergeschaltet?
  jne @@Cascade1
  mov DI,awUpdateIndex[2*2]             ;Timer 1B weiterzhlen
  sub AX,awUpdate[DI]
  jnc @@Set1B
@@Shoot1B:
  ifdef DEBUG
    TBEG CIA
    if GERMAN
      INFO "1 Unterlauf Timer B aufgetreten"
    else
      INFO "1 Timer B underflow occurred"
    endif
    TEND
  endif
  or CL,00000010b                       ;Unterlauf Timer 1B aufgetreten
  test BH,00001000b                     ;Timer im Single-Shot-Modus?
  jne @@Single1B
@@Add1B:
  stc                                   ;Timer 1B mit Startwert laden
  adc AX,word ptr CIA1[6]
  jc @@Set1B
  jmp @@Add1B                           ;Wegen Rasterung wiederholen
@@Single1B:
  ifdef DEBUG
    TBEG CIA
    if GERMAN
      INFO2 "Single Shot Modus ist aktiviert, Timer B wird angehalten"
    else
      INFO2 "Single Shot mode active, Timer B stopped"
    endif
    TEND
  endif
  and BH,11111110b                      ;Timer anhalten
  mov CIA1[15],BH
  mov AX,word ptr CIA1[6]               ;Endwert wegen Rasterung setzen
@@Set1B:
  mov Timer1B,AX
@@Check2A:
  mov BX,word ptr CIA2[14]              ;Luft Timer 2A?
  test BL,00000001b
  je @@Check2B
  mov AX,Timer2A                        ;Timer 2A weiterzhlen
  mov DI,awUpdateIndex[4*2]
  sub AX,awUpdate[DI]
  jnc @@Set2A
  ifdef DEBUG
    TBEG CIA
    if GERMAN
      INFO "2 Unterlauf Timer A aufgetreten"
    else
      INFO "2 Timer A underflow occurred"
    endif
    TEND
  endif
  mov CH,00000001b                      ;Unterlauf Timer 2A aufgetreten

  cmp abSerialCounter[1],0
  je @@NoSerial2
  cmp abSerialCounter[1],2
  jne @@NoEmpty2
  ifdef DEBUG
    TBEG INT
    if GERMAN
      INFO2 "Schieberegister wurde ausgegeben"
    else
      INFO2 "The shift register was shifted out"
    endif
    TEND
  endif
  or CH,00001000b
@@NoEmpty2:
  dec abSerialCounter[1]
@@NoSerial2:

  test BL,00001000b                     ;Timer im Single-Shot-Modus?
  jne @@Single2A
@@Add2A:
  stc                                   ;Timer 2A mit Startwert laden
  adc AX,word ptr CIA2[4]
  jc @@Set2A
  jmp @@Add2A                           ;Wegen Rasterung wiederholen
@@Single2A:
  ifdef DEBUG
    TBEG CIA
    if GERMAN
      INFO2 "Single Shot Modus ist aktiviert, Timer A wird angehalten"
    else
      INFO2 "Single Shot mode active, Timer A stopped"
    endif
    TEND
  endif
  and BL,11111110b                      ;Timer anhalten
  mov CIA2[14],BL
  mov AX,word ptr CIA2[4]               ;Endwert wegen Rasterung setzen
@@Set2A:
  mov Timer2A,AX
@@Check2B:
  test BH,0000001b                      ;Luft Timer 2B?
  je @@EndTimer
  mov AX,Timer2B
  test BH,01000000b                     ;Timer hintereinandergeschaltet?
  jne @@Cascade2
  mov DI,awUpdateIndex[5*2]             ;Timer 2B weiterzhlen
  sub AX,awUpdate[DI]
  jnc @@Set2B
@@Shoot2B:
  ifdef DEBUG
    TBEG CIA
    if GERMAN
      INFO "2 Unterlauf Timer B aufgetreten"
    else
      INFO "2 Timer B underflow occurred"
    endif
    TEND
  endif
  or CH,00000010b                       ;Unterlauf Timer 2B aufgetreten
  test BH,00001000b                     ;Timer im Single-Shot-Modus?
  jne @@Single2B
@@Add2B:
  stc                                   ;Timer 2B mit Startwert laden
  adc AX,word ptr CIA2[6]
  jc @@Set2B
  jmp @@Add2B                           ;Wegen Rasterung wiederholen
@@Single2B:
  ifdef DEBUG
    TBEG CIA
    if GERMAN
      INFO2 "Single Shot Modus ist aktiviert, Timer B wird angehalten"
    else
      INFO2 "Single Shot mode active, Timer B stopped"
    endif
    TEND
  endif
  and BH,11111110b                      ;Timer anhalten
  mov CIA2[15],BH
  mov AX,word ptr CIA2[6]               ;Endwert wegen Rasterung setzen
@@Set2B:
  mov Timer2B,AX
@@EndTimer:
  cmp bTime1Running,0                   ;Echtzeituhr CIA1
  je @@Time1Cont
  mov DI,awUpdateIndex[0*2]
  movzx EAX,awUpdate[DI]
  sub dTime1Count,EAX
  js @@Time1
@@Time1Cont:
  cmp bTime2Running,0                   ;Echtzeituhr CIA2
  je @@Time2Cont
  mov DI,awUpdateIndex[3*2]
  movzx EAX,awUpdate[DI]
  sub dTime2Count,EAX
  js @@Time2
@@Time2Cont:
  mov BX,CIAEvents                      ;Alte Ereignismaske holen
  test CIA1[13],CL                      ;Ereignis auf CIA1 eingetreten?
  je @@NoEvent1
  and BL,BL                             ;Interrupt bereits gesetzt?
  js @@NoEvent1
  ifdef DEBUG
    TBEG INT
    if GERMAN
      INFO "CIA 1 lst IRQ aus"
    else
      INFO "CIA 1 IRQ occurred"
    endif
    TEND
  endif
  or BL,10000000b                       ;Interrupt in Ereignismaske vermerken
  or Events,evIRQCIA                    ;Kennung fr IRQ setzen
@@NoEvent1:
  test CIA2[13],CH                      ;Ereignis auf CIA2 eingetreten?
  je @@NoEvent2
  and BH,BH                             ;Interrupt bereits gesetzt?
  js @@NoEvent2
  ifdef DEBUG
    TBEG INT
    if GERMAN
      INFO "CIA 2 lst NMI aus"
    else
      INFO "CIA 2 NMI occurred"
    endif
    TEND
  endif
  or BH,10000000b                       ;Interrupt in Ereignismaske vermerken
  or Events,evNMI                       ;Kennung fr NMI setzen
@@NoEvent2:
  or BX,CX                              ;Neue Ereignisse schreiben
  mov CIAEvents,BX
  mov BX,awUpdate[2]                    ;Dauerfeuer weiterzhlen
  sub alJoyCount[0],EBX
  js @@Fire1
@@Fire1Cont:
  sub alJoyCount[4],EBX
  jns UpdateVIC
  mov EAX,alJoyStart[4]
  add alJoyCount[4],EAX
  xor awJoyFlags[2],0000000000010000b
  jmp UpdateVIC
align 4
@@Fire1:
  mov EAX,alJoyStart[0]
  add alJoyCount[0],EAX
  xor awJoyFlags[0],0000000000010000b
  jmp @@Fire1Cont
align 4
@@Cascade1:
  and CL,CL
  je @@Check2A
  sub AX,1
  jnc @@Set1B
  jmp @@Shoot1B
align 4
@@Cascade2:
  and CH,CH
  je @@EndTimer
  sub AX,1
  jnc @@Set2B
  jmp @@Shoot2B
align 4
@@Time1:
  mov EAX,dTime1Add                     ;1/10 sec zur Echtzeituhr CIA 1 addieren
  add dTime1Count,EAX
  mov EAX,dword ptr abTime1             ;Zeit um 1/10 sec erhhen
  call IncTime
  mov dword ptr abTime1,EAX
  ifdef DEBUG
    TBEG CIA
    mov fnContinue,offset @@Continue
    call DumpTime
    db " CIA 1 Uhrzeit erhhen (%cM %02X:%02X:%02X.%X)",0
  @@Continue:
    TEND
  endif
  cmp EAX,dword ptr abAlarm1            ;Mit Alarmzeit vergleichen
  jne @@Time1Cont
  ifdef DEBUG
    TBEG CIA
    if GERMAN
      INFO2 "Eingestellte Alarmzeit wurde erreicht"
    else
      INFO2 "The TOD clock alarm went off"
    endif
    TEND
  endif
  or CL,00000100b
  jmp @@Time1Cont
align 4
@@Time2:
  mov EAX,dTime2Add                     ;1/10 sec zur Echtzeituhr CIA 1 addieren
  add dTime2Count,EAX
  mov EAX,dword ptr abTime2             ;Zeit um 1/10 sec erhhen
  call IncTime
  mov dword ptr abTime2,EAX
  ifdef DEBUG
    TBEG CIA
    mov fnContinue,offset @@Continue
    call DumpTime
    db " CIA 2 Uhrzeit erhhen (%cM %02X:%02X:%02X.%X)",0
  @@Continue:
    TEND
  endif
  cmp EAX,dword ptr abAlarm2            ;Mit Alarmzeit vergleichen
  jne @@Time2Cont
  ifdef DEBUG
    TBEG CIA
    if GERMAN
      INFO2 "Eingestellte Alarmzeit wurde erreicht"
    else
      INFO2 "The TOD clock alarm went off"
    endif
    TEND
  endif
  or CH,00000100b
  jmp @@Time2Cont
UpdateCIA endp

ende
